//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsCompressor.h                     \n
 * Section: libModule                           \n
 * Date:    2005/03/03                          \n
 *
 * $Id: mdsCompressor.h 383 2007-06-21 12:17:06Z spanel $
 *
 * Description:
 * - Channel compression tools.
 */

#ifndef MDS_COMPRESSOR_H
#define MDS_COMPRESSOR_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsSharedPtr.h>

#include "mdsChannel.h"

// STL
#include <string>


namespace mds
{
namespace mod
{

//==============================================================================
/*
 * Compression type.
 */

//! Available channel compression methods.
enum EChannelCompression
{
    CC_RAW      = 0,    //! Raw/uncompressed data.
    CC_RLE8     = 1,    //! 8-bit RLE compression.
    CC_RLE16    = 2     //! 16-bit RLE compression.
};


//==============================================================================
/*!
 * Abstract class encapsulating a channel compressor.
 */
class CChannelCompressor : public CChannel
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CChannelCompressor);

    //! Smart pointer type.
    MDS_SHAREDPTR(CChannelCompressor);

public:
    //! Default constructor.
    //! - Compression type.
    //! - Pointer to an output/input channel.
    CChannelCompressor(EChannelCompression eMethod, CChannel *pChannel)
        : m_eChannelCompression(eMethod)
        , m_spChannel(pChannel)
    {
        MDS_ASSERT(pChannel);
        
        m_eType = m_spChannel->getType();
        m_eMedium = m_spChannel->getMedium();
    }

    //! Virtual destructor
    virtual ~CChannelCompressor() {}

    //! Returns the compression method.
    EChannelCompression getCompressionMethod() { return m_eChannelCompression; }

    //! Returns pointer to the input/output channel.
    CChannel *getChannel() { return m_spChannel; }

    //! Sets the input/output channel.
    void setChannel(CChannel *pChannel) { m_spChannel = pChannel; }


    //! Connects to the opposite channel side.
    //! - If the channel is connection oriented, this function must be called
    //!   on the server side (reader) first.
    //! - Returns false on failure and/or timeout elapsed.
    virtual bool connect(unsigned uTimeout = CH_CONNECTION_TIMEOUT)
    {
        return m_spChannel->connect(uTimeout);
    }

    //! Disconnects from the opossite channel side.
    //! - Returns false on failure.
    virtual void disconnect() { m_spChannel->disconnect(); }

    //! Method for testing channel errors such as:
    //! - Disconnected or broken channel.
    //! - Reading after end of input.
    virtual bool isConnected() { return m_spChannel->isConnected(); }


    //! Waits for data.
    //! - Returns false if a timeout elapses and/or the channel is not input.
    virtual bool wait(unsigned uTimeout = CH_WAIT_TIMEOUT)
    {
        return m_spChannel->wait();
    }

    //! Reads and decompresses data from the input channel.
    //! - In case of error or reading after end of input
    //!   the number of successfully readed bytes is returned.
    virtual int read(char *pcData, int iLength) = 0;

    //! Writes compressed data to the output channel.
    virtual bool write(const char *pcData, int iLength) = 0;

    //! Finalizes all writing operations (flushes internal buffers, etc.).
    virtual bool flush()
    {
        return m_spChannel->flush();
    }


    //! Channel compressor creation function.
    //! - Returns NULL on failure.
    static CChannelCompressor *create(EChannelCompression eCompression, CChannel *pChannel);

protected:
    //! Compression method.
    EChannelCompression m_eChannelCompression;

    //! Output/input channel.
    CChannelPtr m_spChannel;
};


//==============================================================================
/*!
 * Smart pointer to channel compressor.
 */
typedef CChannelCompressor::tSmartPtr  CChannelCompressorPtr;


//==============================================================================
/*!
 * Writes uncompressed data.
 */
class CRawCompressor : public CChannelCompressor
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CRawCompressor);

    //! Smart pointer type.
    MDS_SHAREDPTR(CRawCompressor);

public:
    //! Constructor.
    CRawCompressor(CChannel *pChannel) : CChannelCompressor(CC_RAW, pChannel) {}

    //! Destructor.
    virtual ~CRawCompressor() {}

    // Vitual methods.
    virtual int read(char *pcData, int iLength)
    {
        return m_spChannel->read(pcData, iLength);
    }
    virtual bool write(const char *pcData, int iLength)
    {
        return m_spChannel->write(pcData, iLength);
    }
};


//==============================================================================
/*!
 * Smart pointer compressor.
 */
typedef CRawCompressor::tSmartPtr   CRawCompressorPtr;


//==============================================================================
/*!
 * Class providing 8-bit RLE compression.
 */
class CRLE8Compressor : public CChannelCompressor
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CRLE8Compressor);

    //! Smart pointer type.
    MDS_SHAREDPTR(CRLE8Compressor);

public:
    //! Constructor.
    CRLE8Compressor(CChannel *pChannel);

    //! Destructor.
    virtual ~CRLE8Compressor();

    // Virtual methods.
    virtual int read(char *pcData, int iLength);
    virtual bool write(const char *pcData, int iLength);

protected:
    //! Size of the internal buffer.
    enum { BUFFER_SIZE = 0x7F };

    //! RLE compressor states.
    enum EState
    {
        S_BEGIN, S_FIRST, S_COMPRESSIBLE, S_UNCOMPRESSIBLE
    };

    //! Helper buffer of data bytes.
    char m_pcBuffer[BUFFER_SIZE + 1];

    //! The number of remaining unread bytes in the last block.
    int m_iNumOfUnreadBytes;
};


//==============================================================================
/*!
 * Smart pointer to 8-bit RLE compressor.
 */
typedef CRLE8Compressor::tSmartPtr     CRLE8CompressorPtr;


//==============================================================================
/*!
 * Class providing 16-bit RLE compression.
 */
class CRLE16Compressor : public CChannelCompressor
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CRLE16Compressor);

    //! Smart pointer type.
    MDS_SHAREDPTR(CRLE16Compressor);

public:
    //! Constructor.
    CRLE16Compressor(CChannel *pChannel);

    //! Destructor.
    virtual ~CRLE16Compressor();

    // Virtual methods.
    virtual int read(char *pcData, int iLength);
    virtual bool write(const char *pcData, int iLength);

protected:
    //! Size of the internal buffer.
    enum { BUFFER_SIZE = 0x7F };

    //! RLE compressor states.
    enum EState
    {
        S_BEGIN, S_FIRST, S_COMPRESSIBLE, S_UNCOMPRESSIBLE
    };

    //! Helper array of data bytes.
    char m_pcBuffer[2 * BUFFER_SIZE + 1];

    //! The number of remaining unread words in the last block.
    int m_iNumOfUnreadBytes;
};


//==============================================================================
/*!
 * Smart pointer to the 16-bit RLE compressor.
 */
typedef CRLE16Compressor::tSmartPtr    CRLE16CompressorPtr;


} // namespace mod
} // namespace mds

#endif // MDS_COMPRESSOR_H

