//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsChannel.h                        \n
 * Section: libModule                           \n
 * Date:    2003/10/23                          \n
 *
 * $Id: mdsChannel.h 439 2007-08-08 09:40:26Z spanel $
 * 
 * Description:
 * - I/O interface by means of virtual channels.
 * - E.g. stdin/stdout, files and named pipes.
 * - Some trick have to be used to switch stdin and stdout to the binary mode!
 */

#ifndef MDS_CHANNEL_H
#define MDS_CHANNEL_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsSharedPtr.h>
#include <MDSTk/Base/mdsLock.h>
#include <MDSTk/System/mdsNamedPipe.h>
#include <MDSTk/System/mdsSharedMem.h>
#include <MDSTk/System/mdsSemaphore.h>

// STL
#include <cstdio>
#include <string>
#include <vector>


namespace mds
{
namespace mod
{

//==============================================================================
/*
 * Global definitions and constants.
 */

//! Channel types. Usually, the listed types cannot be combined together,
//! channel can be either input or output. However, there are some bidirectional
//! channels which can be opened in both directions. See the comments.
enum EChannelType
{
    CH_IN               = 1,                //! Input channel.
    CH_OUT              = 2,                //! Output channel.
    CH_BIDIRECTIONAL    = CH_IN | CH_OUT    //! Bidirectional channel.
};

//! Channel transmission medium.
enum EChannelMedium
{
    CH_NULL     = 0,        //! Null channel.
    CH_STDIO    = 1,        //! Channel via stdin and stdout.
    CH_FILE,                //! Channel implemented over files.
    CH_PIPE,                //! Channel using named pipes.
    CH_SHM,                 //! Channel over a shared memory.
//  CH_TCP                  //! Communication over network (not yet implemented).
};

//! Timeout used for connection establishing.
const unsigned CH_CONNECTION_TIMEOUT    = 15000;

//! Timeout used for waiting for data.
const unsigned CH_WAIT_TIMEOUT          = 1000;


//==============================================================================
/*!
 * Base abstract class for all classes implementing channels.
 */
class CChannel : public mds::CObject, public mds::base::CLibraryLockableObject<CChannel>
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CChannel);

    //! Smart pointer type.
    MDS_SHAREDPTR(CChannel);

    //! Lock that have to be used to provide mutual access to write(), read()
    //! and etc. functions.
    typedef mds::base::CLibraryLockableObject<CChannel>::CLock tLock;

public:
    //! Default constructor.
    //! - It doesn't initialize the channel type and medium!
    //! - Use with care!
    CChannel() {}

    //! Constructor.
    //! - Initializes the channel type and medium.
    CChannel(EChannelMedium eMedium, EChannelType eType)
        : m_eType(eType)
        , m_eMedium(eMedium)
    {}

    //! Virtual destructor.
    virtual ~CChannel() {}

    //! Method returns the channel type.
    EChannelType getType() { return m_eType; }

    //! Returns the channel medium.
    EChannelMedium getMedium() { return m_eMedium; }


    //! Connects to the opposite channel side.
    //! - If the channel is connection oriented, this function must be called
    //!   on the server side (reader) first.
    //! - Returns false on failure and/or timeout elapsed.
    virtual bool connect(unsigned uTimeout = CH_CONNECTION_TIMEOUT) = 0;

    //! Disconnects from the opossite channel side.
    //! - Returns false on failure.
    virtual void disconnect() = 0;

    //! Method for testing channel errors such as:
    //! - Disconnected or broken channel.
    //! - Reading after end of input.
    virtual bool isConnected() = 0;


    //! Waits for data.
    //! - Returns false if a timeout elapses and/or the channel is not input.
    virtual bool wait(unsigned uTimeout = CH_WAIT_TIMEOUT) = 0;

    //! Reads bytes from the input channel.
    //! - In case of error or reading after end of input
    //!   the number of successfully readed bytes is returned.
    virtual int read(char *pcData, int iLength) = 0;

    //! Writes data to the output channel.
    virtual bool write(const char *pcData, int iLength) = 0;

    //! Finalizes all writing operations (flushes internal buffers, etc.).
    virtual bool flush() = 0;


    //! Skips a given number of bytes.
    //! - If the channel is open for reading, the data are discarded.
    //!   Otherwise, zeros are written to the output channel.
    //! - This method cannot be used if the channel is opened for reading
    //!   and writting simultaneously!
    virtual bool skip(int iLength);


    //! Creation of a channel which is described by a textual string.
    //! - String has the form: "chmedium[:chattrib1[:chattrib2[:...]]]".
    //! - Actually, the channel medium can be "stdio", "file", "pipe"
    //!   and "shm".
    //! - Channel attributes depend on the medium.
    //! - "stdio" has empty attributes, "file" (:filename), "pipe" (:pipename)
    //!   and "shm" (:shmname).
    //! - Returns NULL on failure.
    static CChannel *create(EChannelType eType, const std::string& sDesc);

    //! Parses a given string and separates descriptions of several channels.
    //! - Channels are separated by two successive colons "::" in the string.
    //! - Example: "chmedium1[:chattribs1][::chmedium2[:chattribs2][::...]]".
    static void separate(const std::string& sDescs,
                         std::vector<std::string>& Descs
                         );

protected:
    //! Channel type.
    EChannelType m_eType;

    //! Channel medium.
    EChannelMedium m_eMedium;
};


//==============================================================================
/*!
 * Smart pointer to channel.
 */
typedef CChannel::tSmartPtr     CChannelPtr;


//==============================================================================
/*!
 * Null channel that doesn't write or read any data. It just counts the number
 * of written and read bytes of data.
 * - This channel is bidirectional. It can be opened for reading
 *   and writing simultaneously.
 */
class CNullChannel : public CChannel
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CNullChannel);

    //! Smart pointer type.
    MDS_SHAREDPTR(CNullChannel);

public:
    //! Constructor.
    //! - Channel type.
    CNullChannel(EChannelType eType)
        : CChannel(CH_NULL, eType)
        , m_iReadCount(0)
        , m_iWrittenCount(0)
    {}

    //! Destructor.
    virtual ~CNullChannel() {}

    // Returns the number of bytes read from the channel.
    int getNumOfReadBytes()
    {
        tLock Lock(*this);

        return m_iReadCount;
    }
    
    // Returns the number of bytes written to the channel.
    int getNumOfWrittenBytes()
    {
        tLock Lock(*this);

        return m_iWrittenCount;
    }

    // Clear the number of read an written bytes.
    void clearNumOfBytes()
    {
        tLock Lock(*this);

        m_iReadCount = 0;
        m_iWrittenCount = 0;
    }

    // Virtual methods
    virtual bool connect(unsigned uTimeout = CH_CONNECTION_TIMEOUT) { return true; }
    virtual void disconnect() {}
    virtual bool isConnected() { return true; }
    virtual bool wait(unsigned uTimeout = CH_WAIT_TIMEOUT) { return true; }
    virtual int read(char *pcData, int iLength)
    {
        tLock Lock(*this);

        m_iReadCount += iLength;
        return iLength;
    }
    virtual bool write(const char *pcData, int iLength)
    {
        tLock Lock(*this);

        m_iWrittenCount += iLength;
        return true;
    }
    virtual bool flush() { return true; } 

protected:
    //! The number of read/written bytes.
    int m_iReadCount, m_iWrittenCount;
};


//==============================================================================
/*!
 * Smart pointer to null channel.
 */
typedef CNullChannel::tSmartPtr     CNullChannelPtr;


//==============================================================================
/*!
 * Channel implemented via reading stdin and writing stdout.
 */
class CStdChannel : public CChannel
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CStdChannel);

    //! Smart pointer type.
    MDS_SHAREDPTR(CStdChannel);

public:
    //! Constructor.
    //! - Channel type.
    CStdChannel(EChannelType eType);

    //! Destructor.
    virtual ~CStdChannel();

    // Virtual methods
    virtual bool connect(unsigned uTimeout = CH_CONNECTION_TIMEOUT);
    virtual void disconnect();
    virtual bool isConnected();
    virtual bool wait(unsigned uTimeout = CH_WAIT_TIMEOUT);
    virtual int read(char *pcData, int iLength);
    virtual bool write(const char *pcData, int iLength);
    virtual bool flush(); 
};


//==============================================================================
/*!
 * Smart pointer to stdin/stdout channel.
 */
typedef CStdChannel::tSmartPtr      CStdChannelPtr;


//==============================================================================
/*!
 * I/O by means of reading and writing files.
 */
class CFileChannel : public CChannel
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CFileChannel);

    //! Smart pointer type.
    MDS_SHAREDPTR(CFileChannel);

public:
    //! Constructor.
    //! - Allowed channel types are CH_IN and CH_OUT.
    //! - Input/output file.
    CFileChannel(EChannelType eType, const std::string& sFileName);

    //! Constructor.
    //! - Allowed channel types are CH_IN and CH_OUT.
    CFileChannel(EChannelType eType);

    //! Destructor.
    virtual ~CFileChannel();

    // Virtual methods
    virtual bool connect(unsigned uTimeout = CH_CONNECTION_TIMEOUT);
    virtual bool connect(const std::string& sFileName, unsigned uTimeout = CH_CONNECTION_TIMEOUT);
    virtual void disconnect();
    virtual bool isConnected();
    virtual bool wait(unsigned uTimeout = CH_WAIT_TIMEOUT);
    virtual int read(char *pcData, int iLength);
    virtual bool write(const char *pcData, int iLength);
    virtual bool flush(); 

protected:
    //! Input and/or output binary file
    FILE *m_File;
};


//==============================================================================
/*!
 * Smart pointer to file channel.
 */
typedef CFileChannel::tSmartPtr     CFileChannelPtr;


//==============================================================================
/*!
 * Channel via named pipes.
 */
class CPipeChannel : public CChannel
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CPipeChannel);

    //! Smart pointer type.
    MDS_SHAREDPTR(CPipeChannel);

public:
    //! Constructor.
    //! - Allowed channel types are CH_IN and CH_OUT.
    //! - Name of the pipe.
    CPipeChannel(EChannelType eType, const std::string& ssName);

    //! Destructor.
    virtual ~CPipeChannel();

    // Virtual methods
    virtual bool connect(unsigned uTimeout = CH_CONNECTION_TIMEOUT);
    virtual void disconnect();
    virtual bool isConnected();
    virtual bool wait(unsigned uTimeout = CH_WAIT_TIMEOUT);
    virtual int read(char *pcData, int iLength);
    virtual bool write(const char *pcData, int iLength);
    virtual bool flush(); 

protected:
    //! Input and/or output named pipe
    mds::sys::CNamedPipe m_Pipe;
};


//==============================================================================
/*!
 * Smart pointer to pipe channel.
 */
typedef CPipeChannel::tSmartPtr     CPipeChannelPtr;


//==============================================================================
/*!
 * Channel over shared memory.
 * - Only for testing!
 */
class CSharedMemChannel : public CChannel
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CSharedMemChannel);

    //! Smart pointer type.
    MDS_SHAREDPTR(CSharedMemChannel);

public:
    //! Constructor.
    //! - Opens an existing shared memory.
    //! - Allowed channel types are CH_IN and CH_OUT.
    //! - Name of the shared memory.
    CSharedMemChannel(EChannelType eType, const std::string& ssName);

    //! Destructor.
    virtual ~CSharedMemChannel();

    //! Returns pointer to the shared memory.
    mds::sys::CSharedMem *getSharedMemPtr() { return m_spSharedMem; }

    //! Returns current pointer to the remaining (unread) data.
    char *getDataPtr() { return m_pcData; }

    // Virtual methods
    virtual bool connect(unsigned uTimeout = CH_CONNECTION_TIMEOUT);
    virtual void disconnect();
    virtual bool isConnected();
    virtual bool wait(unsigned uTimeout = CH_WAIT_TIMEOUT);
    virtual int read(char *pcData, int iLength);
    virtual bool write(const char *pcData, int iLength);
    virtual bool flush();

    //! Skips a given number of bytes.
    virtual bool skip(int iLength);

protected:
    //! Named semaphore used to lock the shared memory.
    mds::sys::CSemaphorePtr m_spSemaphore;

    //! Shared memory.
    mds::sys::CSharedMemPtr m_spSharedMem;

    //! The number of bytes remaining in the the shared memory.
    int m_iNumOfBytes;

    //! Pointer to the first unread data byte in the shared memory.
    char *m_pcData;

    //! Pointer to the number of bytes written to the shared memory.
    int *m_piNumOfWrittenBytes;
};


//==============================================================================
/*!
 * Smart pointer to shared memory channel.
 */
typedef CSharedMemChannel::tSmartPtr    CSharedMemChannelPtr;


//==============================================================================
/*
 * Useful macros.
 */

//! Macro that reads a given number of bytes from channel.
//! - The Stmt command is executed in case of some error.
#define MDS_CH_READ_BYTES(Channel, pDst, Size, Stmt) \
    if( (Channel).read((char *)(pDst), int(Size)) != int(Size) ) \
    { \
        Stmt; \
    }

//! Macro writes a given number of bytes to channel.
//! - The Stmt command is executed in case of some error.
#define MDS_CH_WRITE_BYTES(Channel, pSrc, Size, Stmt) \
    if( !(Channel).write((char *)(pSrc), int(Size)) ) \
    { \
        Stmt; \
    }


} // namespace mod
} // namespace mds

#endif // MDS_CHANNEL_H

