//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsBinarySerializer.h               \n
 * Section: libModule                           \n
 * Date:    2007/06/20                          \n
 *
 * $Id:$
 *
 * Description:
 * - Binary serialization of objects (data entities) over channels.
 */

#ifndef MDS_BINARYSERIALIZER_H
#define MDS_BINARYSERIALIZER_H

#include "mdsSerializer.h"
#include "mdsBlockChannel.h"
#include "mdsCompressor.h"


namespace mds
{
namespace mod
{

//==============================================================================
/*!
 * Serializer which encapsulates serialization of data entities
 * and their members via channels.
 */
class CBinarySerializer : public CChannelSerializer
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CBinarySerializer);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CBinarySerializer);

public:
    //! Default constructor.
    CBinarySerializer(CChannel *pChannel, unsigned int uiFlags = 0);
    
    //! Destructor.
    virtual ~CBinarySerializer();


    //! Serializes a given data entity.
    //! - Returns false on failure.
    virtual bool writeRoot(CSerializable& Object);

    //! Creates a new logical group of class members in the serialized data.
    //! - Serializer properties such as compression method may be changed.
    //! - Must be called before any writeX() method.
    //! - Sets the ERROR_FLAG on failure.
    virtual void beginWrite(const char *pcName, EChannelCompression Compression, tSize BlockSize);
    
    //! End of the logical group.
    virtual void endWrite();

    //! Serializes a given primitive value.
    //! - Sets the ERROR_FLAG on failure.
    virtual void writeBool(bool bValue);
    virtual void writeInt(int iValue);
    virtual void writeUInt(unsigned int uiValue);
    virtual void writeFloat(float fValue);
    virtual void writeDouble(double dValue);
    virtual void writeSize(tSize Value);
    virtual void writeBytes(const void *pData, tSize Length);


    //! Deserializes a given data entity.
    //! - Returns false on failure.
    virtual bool readRoot(CSerializable& Object);

    //! Reads information on logical group of class members from the serialized data.
    //! - Must be called before any readX() method.
    //! - Sets the ERROR_FLAG on failure.
    virtual void beginRead(const char *pcName);

    //! Reads end of the logical group.
    virtual void endRead();

    //! Deserializes a given primitive value.
    //! - Sets the ERROR_FLAG on failure.
    virtual void readBool(bool& bValue);
    virtual void readInt(int& iValue);
    virtual void readUInt(unsigned int& uiValue);
    virtual void readFloat(float& fValue);
    virtual void readDouble(double& dValue);
    virtual void readSize(tSize& Value);
    virtual void readBytes(void *pData, tSize Length);

protected:
    //! Smart pointer to actively used channel.
    CChannelPtr m_spActiveChannel;
};


//==============================================================================
/*!
 * Smart pointer to binary serializer.
 */
typedef CBinarySerializer::tSmartPtr   CBinarySerializerPtr;


//==============================================================================
/*
 * Global functions.
 */

namespace binary_serializer
{

//! Writes data entity header to a binary channel.
//! - Returns false on failure.
bool writeEntityHeader(CChannel& Channel);

//! Reads and checks the data entity header.
//! - Returns false on failure.
bool readEntityHeader(CChannel& Channel);

//! Writes header of data entity logical group.
//! - Returns false on failure.
bool writeGroupHeader(CChannel& Channel,
                      const char *pcName,
                      EChannelCompression eCompression,
                      tSize BlockSize
                      );

//! Reads and decodes header of data entity group.
//! - Returns false on failure.
bool readGroupHeader(CChannel& Channel,
                     const char *pcName,
                     EChannelCompression& eCompression,
                     tSize& BlockSize
                     );

//! Writes terminal to a binary channel.
//! - Returns false on failure.
bool writeTerminal(CChannel& Channel);

//! Finds and reads terminal.
//! - Returns false on failure.
bool readTerminal(CChannel& Channel);   

} // namespace binary_serializer


} // namespace mod
} // namespace mds

#endif // MDS_BINARYSERIALIZER_H

