//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsVectorIterator.h                 \n
 * Section: libMath                             \n
 * Date:    2006/09/15                          \n
 *
 * $Id: mdsVectorIterator.h 345 2007-06-11 13:23:09Z spanel $
 * 
 * Description:
 * - Vector iterator definition.
 */

#ifndef MDS_VECTORITERATOR_H
#define MDS_VECTORITERATOR_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsIterator.h>


namespace mds
{
namespace math
{

//==============================================================================
/*!
 * Iterator used to traverse vector elements.
 * - Template parameter V is the vector type.
 * - Parameter T is the vector element type.
 */
template <class V, typename T>
class CVectorIterator : public mds::base::CComplexIteratorBase<V,T>
{
public:
    //! Check that V is a vector. You will see name of this enum somewhere
    //! in compiler error message if the type V is not vector.
    enum { TEMPLATE_PARAMETER_IS_NOT_VECTOR = V::CLASS_VECTOR };

    //! Iterator base.
    typedef mds::base::CComplexIteratorBase<V,T> base;

    //! Vector type.
    typedef V tVector;

    //! Element type.
    typedef T tElement;

public:
    //! Default constructor.
    CVectorIterator() {}

    //! Constructor.
    CVectorIterator(tVector& Vector, tSize i)
        : mds::base::CComplexIteratorBase<V,T>(&Vector, Vector.getPtr(i))
        , m_i(i)
    {}

    //! Copy constructor.
    CVectorIterator(const CVectorIterator& It)
        : mds::base::CComplexIteratorBase<V,T>(It)
        , m_i(It.m_i)
    {}

    //! Destructor.
    ~CVectorIterator() {}

    //! Assignment operator.
    CVectorIterator& operator=(const CVectorIterator& It)
    {
        base::m_pContainer = It.m_pContainer;
        base::m_pItem = It.m_pItem;
        m_i = It.m_i;
        return *this;
    }


    //! Operator moves the iterator to the next pixel.
    //! - Prefix notation.
    CVectorIterator& operator++()
    {
        advance();
        return *this;
    }

    //! Operator moves the iterator to the next pixel.
    //! - Postfix notation.
    CVectorIterator operator++(int)
    {
        CVectorIterator Temp(*this);
        advance();
        return Temp;
    }

    //! Returns the number of increments between the positions addressed
    //! by two iterators.
    tSize getDistance(const CVectorIterator& End)
    {
        tSize Count = End.m_i - m_i;
        return (Count > 0) ? Count : 0;
    }

    //! Returns current iterator position.
    tSize getPosition() const { return m_i; }

    //! Sets the iterator position.
    void setPosition(tVector& Vector, tSize i)
    {
        base::m_pContainer = &Vector;
        base::m_pItem = Vector.getPtr(i);
        m_i = i;
    }

protected:
    //! Iterator position.
    tSize m_i;
    
protected:
    //! Moves iterator to the next element.
    void advance()
    {
        base::m_pItem += base::m_pContainer->getStride();
        ++m_i;
    }
};


} // namespace math
} // namespace mds

#endif // MDS_VECTORITERATOR_H

