//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsVector.hxx                       \n
 * Section: libMath                             \n
 * Date:    2003/12/05                          \n
 *
 * $Id: mdsVector.hxx 95 2006-09-24 20:11:00Z spanel $
 *
 * Description:
 * - Vector template implementation.
 */


//=============================================================================
/*
 * Methods templates.
 */

// Constructor
// - Creates a new vector of a given size
template <typename T>
CVector<T>::CVector(tSize Size)
    : m_Size(Size)
    , m_Stride(1)
{
    m_DataStorage.init(Size);
    m_pData = m_DataStorage.getPtr();
}


// Constructor
// - Desired size and initial value are given
template <typename T>
CVector<T>::CVector(tSize Size, const T& Value)
    : m_Size(Size)
    , m_Stride(1)
{
    m_DataStorage.init(Size);
    m_pData = m_DataStorage.getPtr();

    // Fill the vector
    fill(Value);
}


// Constructor
// - Creates subvector of a given vector
// - Makes a new copy of the data
template <typename T>
CVector<T>::CVector(const CVector<T>& Vector, tSize First, tSize Size)
{
    MDS_ASSERT(First >= 0 && Size >= 0 && First < Vector.m_Size);

    // Size of the vector
    m_Size = MIN(First + Size, Vector.m_Size) - First;

    // Allocate the data
    m_Stride = 1;
    m_DataStorage.init(m_Size);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    memSparseCopy(m_pData, m_Stride, Vector.getPtr(First), Vector.m_Stride, m_Size);
}


// Constructor
// - Creates subvector of a given vector
// - Makes only reference to the data
template <typename T>
CVector<T>::CVector(const CVector<T>& Vector, tSize First, tSize Size, EMakeRef)
    : m_Stride(Vector.m_Stride)
    , m_DataStorage(Vector.m_DataStorage, REFERENCE)
    , m_pData(Vector.m_pData)
{
    MDS_ASSERT(First >= 0 && Size >= 0 && First < Vector.m_Size);

    // Size of the vector
    m_Size = MIN(First + Size, Vector.m_Size) - First;

    // Vector data
    m_pData += First * m_Stride;
}


// Copy constructor
// - Makes a new copy of the data
template <typename T>
CVector<T>::CVector(const CVector<T>& Vector)
    : m_Size(Vector.m_Size)
    , m_Stride(1)
{
    // Allocate the data
    m_DataStorage.init(m_Size);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    memSparseCopy(m_pData, m_Stride, Vector.m_pData, Vector.m_Stride, m_Size);
}


// Constructor
// - Makes a new copy of the data
template <typename T>
template <tSize M>
CVector<T>::CVector(const CStaticVector<T,M>& Vector)
    : m_Size(M)
    , m_Stride(1)
{
    // Allocate the data
    m_DataStorage.init(m_Size);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    memSparseCopy(m_pData, m_Stride, Vector.m_pData, 1, M);
}


// Copy constructor
// - Makes only reference to the data
template <typename T>
CVector<T>::CVector(const CVector<T>& Vector, EMakeRef)
    : m_Size(Vector.m_Size)
    , m_Stride(Vector.m_Stride)
    , m_DataStorage(Vector.m_DataStorage, REFERENCE)
    , m_pData(Vector.m_pData)
{
}


template <typename T>
CVector<T>::CVector(CMatrix<T>& Matrix, tSize Row, EMakeRowRef)
    : m_Size(Matrix.getNumOfCols())
    , m_Stride(Matrix.getColOffset())
    , m_DataStorage(Matrix.getDataStorage(), REFERENCE)
    , m_pData(Matrix.getRowPtr(Row))
{
    MDS_ASSERT(Row >= 0 && Row < Matrix.getNumOfRows());
}


template <typename T>
CVector<T>::CVector(CMatrix<T>& Matrix, tSize Col, EMakeColRef)
    : m_Size(Matrix.getNumOfRows())
    , m_Stride(Matrix.getRowOffset())
    , m_DataStorage(Matrix.getDataStorage(), REFERENCE)
    , m_pData(Matrix.getColPtr(Col))
{
    MDS_ASSERT(Col >= 0 && Col < Matrix.getNumOfCols());
}


// Destructor
template <typename T>
CVector<T>::~CVector()
{
}


// Creates a new vector
template <typename T>
bool CVector<T>::create(tSize Size)
{
    // Reallocate the data
    m_Size = Size;
    m_Stride = 1;
    m_DataStorage.resize(Size);
    m_pData = m_DataStorage.getPtr();

    // O.K.
    return true;
}


// Creates a new vector
template <typename T>
bool CVector<T>::create(tSize Size, const T& Value)
{
    // Reallocate the data
    m_Size = Size;
    m_Stride = 1;
    m_DataStorage.resize(Size);
    m_pData = m_DataStorage.getPtr();

    // Fill the vector
    memSet(m_pData, Value, m_Size);

    // O.K.
    return true;
}


// Creates a new vector
template <typename T>
bool CVector<T>::create(const CVector<T>& Vector, tSize First, tSize Size)
{
    MDS_CHECK(First >= 0 && Size >= 0 && First < Vector.m_Size, return false);

    // Size of the vector
    m_Size = MIN(First + Size, Vector.m_Size) - First;

    // Allocate the data
    m_Stride = 1;
    m_DataStorage.resize(m_Size);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    memSparseCopy(m_pData, m_Stride, Vector.getPtr(First), Vector.m_Stride, m_Size);

    // O.K.
    return true;
}


// Creates a new reference to specified data
template <typename T>
bool CVector<T>::create(const CVector<T>& Vector, tSize First, tSize Size, EMakeRef)
{
    MDS_CHECK(First >= 0 && Size >= 0 && First < Vector.m_Size, return false);

    // Size of the vector
    m_Size = MIN(First + Size, Vector.m_Size) - First;

    // Create a reference to the vector data
    m_Stride = Vector.m_Stride;
    m_DataStorage.create(Vector.m_DataStorage, REFERENCE);
    m_pData = Vector.m_pData + First * m_Stride;

    // O.K.
    return true;
}


//! Creates a new vector
//! - Makes a new copy of the data
template <typename T>
bool CVector<T>::create(const CVector<T>& Vector)
{
    // Allocate the data
    m_Size = Vector.m_Size;
    m_Stride = 1;
    m_DataStorage.resize(m_Size);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    memSparseCopy(m_pData, m_Stride, Vector.m_pData, Vector.m_Stride, m_Size);

    // O.K.
    return true;
}


//! Creates a new vector
//! - Makes a new copy of the data
template <typename T>
template <tSize M>
bool CVector<T>::create(const CStaticVector<T,M>& Vector)
{
    // Allocate the data
    m_Size = M;
    m_Stride = 1;
    m_DataStorage.resize(m_Size);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    memSparseCopy(m_pData, m_Stride, Vector.getPtr(), 1, M);

    // O.K.
    return true;
}


//! Creates a new vector
//! - Makes only reference to the data
template <typename T>
bool CVector<T>::create(const CVector<T>& Vector, EMakeRef)
{
    // Create a reference to the vector data
    m_Size = Vector.m_Size;
    m_Stride = Vector.m_Stride;
    m_DataStorage.create(Vector.m_DataStorage, REFERENCE);
    m_pData = Vector.m_pData;

    // O.K.
    return true;
}


template <typename T>
bool CVector<T>::create(CMatrix<T>& Matrix, tSize Row, EMakeRowRef)
{
    // Check if a given row index is allowed
    MDS_CHECK(Row >= 0 && Row < Matrix.getNumOfRows(), return false);

    // Create a reference to the matrix data
    m_Size = Matrix.getNumOfCols();
    m_Stride = Matrix.getColOffset();
    m_DataStorage.create(Matrix.getDataStorage(), REFERENCE);
    m_pData = Matrix.getRowPtr(Row);

    // O.K.
    return true;
}


template <typename T>
bool CVector<T>::create(CMatrix<T>& Matrix, tSize Col, EMakeColRef)
{
    // Check if a given column index is allowed
    MDS_CHECK(Col >= 0 && Col < Matrix.getNumOfCols(), return false);

    // Create a reference to the matrix data
    m_Size = Matrix.getNumOfRows();
    m_Stride = Matrix.getRowOffset();
    m_DataStorage.create(Matrix.getDataStorage(), REFERENCE);
    m_pData = Matrix.getColPtr(Col);

    // O.K.
    return true;
}


// Assignment operator
template <typename T>
CVector<T>& CVector<T>::operator =(const CVector<T>& Vector)
{
    if( &Vector == this )
    {
        return *this;
    }
    tSize Count = MIN(m_Size, Vector.m_Size);
    memSparseCopy(m_pData, m_Stride, Vector.m_pData, Vector.m_Stride, Count);

    return *this;
}


// Assignment operator
template <typename T>
template <tSize M>
CVector<T>& CVector<T>::operator =(const CStaticVector<T,M>& Vector)
{
    tSize Count = MIN(m_Size, M);
    memSparseCopy(m_pData, m_Stride, Vector.getPtr(), 1, Count);

    return *this;
}


template <typename T>
inline CVector<T>& CVector<T>::operator +=(const CVector<T>& Vector)
{
    tSize Count = MIN(m_Size, Vector.m_Size);
    memVectSparseAdd(m_pData, m_Stride, Vector.m_pData, Vector.m_Stride, Count);

    return *this;
}


template <typename T>
inline CVector<T>& CVector<T>::operator -=(const CVector<T>& Vector)
{
    tSize Count = MIN(m_Size, Vector.m_Size);
    memVectSparseSub(m_pData, m_Stride, Vector.m_pData, Vector.m_Stride, Count);

    return *this;
}


template <typename T>
inline CVector<T>& CVector<T>::operator *=(const CVector<T>& Vector)
{
    tSize Count = MIN(m_Size, Vector.m_Size);
    memVectSparseMult(m_pData, m_Stride, Vector.m_pData, Vector.m_Stride, Count);

    return *this;
}


template <typename T>
inline CVector<T>& CVector<T>::operator /=(const CVector<T>& Vector)
{
    tSize Count = MIN(m_Size, Vector.m_Size);
    memVectSparseDiv(m_pData, m_Stride, Vector.m_pData, Vector.m_Stride, Count);

    return *this;
}


template <typename T>
template <typename U>
inline CVector<T>& CVector<T>::operator +=(const U& c)
{
    memSparseAdd(m_pData, m_Stride, c, m_Size);

    return *this;
}


template <typename T>
template <typename U>
inline CVector<T>& CVector<T>::operator -=(const U& c)
{
    memSparseSub(m_pData, m_Stride, c, m_Size);

    return *this;
}


template <typename T>
template <typename U>
inline CVector<T>& CVector<T>::operator *=(const U& c)
{
    memSparseMult(m_pData, m_Stride, c, m_Size);

    return *this;
}


template <typename T>
template <typename U>
inline CVector<T>& CVector<T>::operator /=(const U& c)
{
    memSparseDiv(m_pData, m_Stride, c, m_Size);

    return *this;
}


template <typename T>
inline void CVector<T>::zeros()
{
    memSparseSet(m_pData, m_Stride, T(0), m_Size);
}


template <typename T>
inline void CVector<T>::ones()
{
    memSparseSet(m_pData, m_Stride, T(1), m_Size);
}


template <typename T>
inline void CVector<T>::abs()
{
    T *p = m_pData;
    T *pMax = m_pData + m_Size * m_Stride;
    while( p < pMax )
    {
        if( !(*p > T(0)) )
        {
            *p = -(*p);
        }
        p += m_Stride;
    }
}


template <typename T>
inline void CVector<T>::fill(const T& Value)
{
    memSparseSet(m_pData, m_Stride, Value, m_Size);
}


template <typename T>
inline void CVector<T>::clip(const T& Lower, const T& Upper)
{
    T *p = m_pData;
    T *pMax = m_pData + m_Size * m_Stride;
    while( p < pMax )
    {
        if( *p < Lower )
        {
            *p = Lower;
        }
        else if( *p > Upper )
        {
            *p = Upper;
        }
        p += m_Stride;
    }
}


// Subsample vector
template <typename T>
inline void CVector<T>::subSample(const CVector<T>& Vector, tSize k)
{
    MDS_CHECK(k > 0, return);

    tSize Count = MIN(Vector.m_Size / k, m_Size);
    T *p = m_pData;
    T *pMax = m_pData + Count * m_Stride;
    const T *pV = Vector.m_pData;
    while( p < pMax )
    {
        *p = *pV;
        p += m_Stride;
        pV += k * Vector.m_Stride;
    }
}


// Concatenate two vectors
template <typename T>
inline void CVector<T>::concat(const CVector<T>& Vector1,
                               const CVector<T>& Vector2
                              )
{
    tSize Count = MIN(Vector1.m_Size, m_Size);
    memSparseCopy(m_pData, m_Stride,
                  Vector1.m_pData, Vector1.m_Stride,
                  Count
                 );

    if( m_Size > Vector1.m_Size )
    {
        Count = MIN(Vector2.m_Size, m_Size - Vector1.m_Size);
        memSparseCopy(m_pData + Vector1.m_Size * m_Stride, m_Stride,
                      Vector2.m_pData, Vector2.m_Stride,
                      Count
                     );
    }
}


// Matrix vs. vector multiplication
template <typename T>
void CVector<T>::mult(const CMatrix<T>& Matrix, const CVector<T>& Vector)
{
    if( Matrix.getNumOfCols() != Vector.m_Size || Matrix.getNumOfRows()!= m_Size )
    {
        MDS_THROW_WARNING("Method CVector::mult() failed");
    }

    for( tSize i = 0; i < m_Size; ++i )
    {
        T *p = m_pData + i * m_Stride;

        const T *pV = Vector.m_pData;
        const T *pVMax = Vector.m_pData + Vector.m_Size * Vector.m_Stride;
        const T *pM = Matrix.getRowPtr(i);

        *p = T(0);
        while (pV < pVMax)
        {
            *p += *(pM++) * *pV;
            pV += Vector.m_Stride;
        }
    }
}


// Vector vs. matrix multiplication
template <typename T>
void CVector<T>::mult(const CVector<T>& Vector, const CMatrix<T>& Matrix)
{
    if( Matrix.getNumOfRows() != Vector.m_Size || Matrix.getNumOfCols() != m_Size )
    {
        MDS_THROW_WARNING("Method CVector::mult() failed");
    }

    for( tSize i = 0; i < m_Size; ++i )
    {
        T *p = m_pData + i * m_Stride;

        const T *pV = Vector.m_pData;
        const T *pVMax = Vector.m_pData + Vector.m_Size * Vector.m_Stride;
        const T *pM = Matrix.getColPtr(i);

        *p = T(0);
        for( tSize k = 0; k < Matrix.m_Rows; ++k )
        {
            *p += *pM * *pV;
            pM += Matrix.getRowOffset();
            pV += Vector.m_Stride;
        }
    }
}

