//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsVector.h                         \n
 * Section: libMath                             \n
 * Date:    2003/12/05                          \n
 *
 * $Id: mdsVector.h 345 2007-06-11 13:23:09Z spanel $
 * 
 * Description:
 * - Common vector operations.
 */

#ifndef MDS_VECTOR_H
#define MDS_VECTOR_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsWarning.h>
#include <MDSTk/Base/mdsError.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsMemory.h>
#include <MDSTk/Base/mdsRefData.h>
#include <MDSTk/Base/mdsSharedPtr.h>

#include "mdsBase.h"
#include "mdsVectorBase.h"
#include "mdsVectorIterator.h"
#include "mdsStaticVector.h"
#include "mdsMatrix.h"

#include <cmath>


namespace mds
{
namespace math
{

//==============================================================================
/*
 * Global constants.
 */

//! Enumeration used for making reference to a matrix row. It is usually
//! the last parameter of some constructor.
enum EMakeRowRef { ROW_REFERENCE };

//! Enumeration used for making reference to a matrix column.
enum EMakeColRef { COL_REFERENCE };


//==============================================================================
/*!
 * Template providing basic vector operations.
 * - T is vector element type.
 */
template <typename T>
class CVector : public mds::CObject, public CVectorBase<CVector<T> >
{
public:
    //! Templates that require members of the CVector class can use
    //! this enum to check the existence.
    enum { CLASS_VECTOR };

    //! Standard method getClassName().
    MDS_CLASS_NAME(CVector);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CVector);

    //! Vector element type.
    typedef T tElement;

    //! Vector data.
    typedef mds::base::CRefData<T> tDataStorage;

    //! Iterator type.
    //! - Declares types tIterator and tConstIterator.
    MDS_COMPLEX_ITERATOR(CVector, tElement, CVectorIterator);

public:
    //! Constructor.
    //! - Creates a new vector of a given size.
    CVector(tSize Size = 0);

    //! Constructor.
    //! - Desired size and initial value are given.
    CVector(tSize Size, const T& Value);

    //! Constructor.
    //! - Creates subvector of a given vector.
    //! - Makes a new copy of the data.
    CVector(const CVector& Vector, tSize First, tSize Size);

    //! Constructor.
    //! - Creates subvector of a given vector.
    //! - Makes only reference to the data.
    CVector(const CVector& Vector, tSize First, tSize Size, EMakeRef);

    //! Copy constructor.
    //! - Makes a new copy of the data.
    CVector(const CVector& Vector);

    //! Copy constructor.
    //! - Makes only reference to the data.
    CVector(const CVector& Vector, EMakeRef);

    //! Constructor.
    //! - Makes a new copy of the data.
    template <tSize M>
    CVector(const CStaticVector<T,M>& Vector);

    //! Constructor.
    //! - Creates reference to a specified matrix row.
    CVector(CMatrix<T>& Matrix, tSize Row, EMakeRowRef);

    //! Constructor.
    //! - Creates reference to a specified matrix column.
    CVector(CMatrix<T>& Matrix, tSize Col, EMakeColRef);

    //! Destructor.
    virtual ~CVector();

    //! Creates a new vector.
    bool create(tSize Size);

    //! Creates a new vector.
    bool create(tSize Size, const T& Value);

    //! Creates a new vector.
    bool create(const CVector& Vector, tSize First, tSize Size);

    //! Creates a new reference to specified data.
    bool create(const CVector& Vector, tSize First, tSize Size, EMakeRef);

    //! Creates a new vector.
    //! - Makes a new copy of the data.
    bool create(const CVector& Vector);

    //! Creates a new vector.
    //! - Makes only reference to the data.
    bool create(const CVector& Vector, EMakeRef);

    //! Creates a new vector.
    //! - Makes a new copy of the data.
    template <tSize M>
    bool create(const CStaticVector<T,M>& Vector);

    //! Creates new reference to a matrix row.
    bool create(CMatrix<T>& Matrix, tSize Row, EMakeRowRef);

    //! Creates new reference to a matrix col.
    bool create(CMatrix<T>& Matrix, tSize Col, EMakeColRef);

    //! Assignment operator.
    CVector& operator =(const CVector& Vector);

    //! Assignment operator.
    template <tSize M>
    CVector& operator =(const CStaticVector<T,M>& Vector);


    //! Returns the vector size.
    tSize getSize() const { return m_Size; }

    //! Returns stride between two neigbouring vector elements.
    tSize getStride() const { return m_Stride; }

    //! Returns reference to the vector data storage.
    tDataStorage& getDataStorage() { return m_DataStorage; }
    const tDataStorage& getDataStorage() const { return m_DataStorage; }

    //! Returns the subscripted element [i].
    T& operator ()(tSize i)
    {
        return *(m_pData + i * m_Stride);
    }
    const T& operator ()(tSize i) const
    {
        return *(m_pData + i * m_Stride);
    }

    //! Returns reference to the element [i].
    T& get(tSize i)
    {
        return *(m_pData + i * m_Stride);
    }
    const T& get(tSize i) const
    {
        return *(m_pData + i * m_Stride);
    }

    //! Returns pointer to the subscripted element.
    T *getPtr(tSize i)
    {
        return (m_pData + i * m_Stride);
    }
    const T *getPtr(tSize i) const
    {
        return (m_pData + i * m_Stride);
    }

    //! Returns pointer to the vector data.
    T *getPtr() { return m_pData; }
    const T *getPtr() const { return m_pData; }

    //! Sets the vector element at the position [i].
    void set(tSize i, const T& Value)
    {
        *(m_pData + i * m_Stride) = Value;
    }


    //! Returns iterator that points at the first vector element.
    tIterator getBegin()
    {
        return tIterator(*this, 0);
    }
    tConstIterator getBegin() const
    {
        return tConstIterator(*this, 0);
    }

    //! Returns iterator that points beyond the last vector element.
    tIterator getEnd()
    {
        return tIterator(*this, m_Size);
    }
    tConstIterator getEnd() const
    {
        return tConstIterator(*this, m_Size);
    }

    //! Returns iterator that points at a specified position.
    tIterator getIterator(tSize i)
    {
        return tIterator(*this, i);
    }
    tConstIterator getIterator(tSize i) const
    {
        return tConstIterator(*this, i);
    }


    //! Miscellaneous combined assignment operators.
    inline CVector& operator +=(const CVector& Vector);
    inline CVector& operator -=(const CVector& Vector);
    inline CVector& operator *=(const CVector& Vector);
    inline CVector& operator /=(const CVector& Vector);

    //! Combined assignment operator.
    template <typename U>
    inline CVector& operator +=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CVector& operator -=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CVector& operator *=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CVector& operator /=(const U& c);


    //! The vector initialization method.
    inline void zeros();

    //! The vector initialization method.
    inline void ones();

    //! Absolute value of all elements.
    inline void abs();

    //! Sets vector elements to a given value.
    inline void fill(const T& Value);

    //! Clip range of elements.
    inline void clip(const T& Lower, const T& Upper);

    //! Subsample the vector.
    inline void subSample(const CVector& Vector, tSize k = 2);

    //! Concatenate two vectors
    inline void concat(const CVector& Vector1, const CVector& Vector2);

    //! Matrix vs. vector multiplication.
    void mult(const CMatrix<T>& Matrix, const CVector& Vector);

    //! Vector vs. matrix multiplication.
    void mult(const CVector& Vector, const CMatrix<T>& Matrix);

protected:
    //! Vector dimension.
    tSize m_Size;

    //! Stride between two neighbouring vector values.
    tSize m_Stride;

    //! Vector data.
    tDataStorage m_DataStorage;

    //! Pointer to data.
    T *m_pData;
};


//=============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsVector.hxx"


//=============================================================================
/*
 * Basic template instances and type definitions.
 */

//! Vector of int numbers
typedef CVector<int>        CIVector;

//! Vector of float numbers
typedef CVector<float>      CFVector;

//! Vector of doubles
typedef CVector<double>     CDVector;


//=============================================================================
/*
 * Basic template instances and type definitions.
 * - Using smart pointer
 */

//! Pointer to a vector of int numbers
typedef CIVector::tSmartPtr     CIVectorPtr;

//! Pointer to a vector of float numbers
typedef CFVector::tSmartPtr     CFVectorPtr;

//! Pointer to a vector of doubles
typedef CDVector::tSmartPtr     CDVectorPtr;


} // namespace math
} // namespace mds

#endif // MDS_VECTOR_H

