//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsStaticVector.hxx                 \n
 * Section: libMath                             \n
 * Date:    2004/06/05                          \n
 *
 * $Id: mdsStaticVector.hxx 95 2006-09-24 20:11:00Z spanel $
 *
 * Description:
 * - Static vector operations.
 */


//==============================================================================
/*
 * Method templates.
 */

// Constructor
template <typename T, tSize M>
inline CStaticVector<T,M>::CStaticVector()
{
}


// Constructor
template <typename T, tSize M>
inline CStaticVector<T,M>::CStaticVector(const T& Value)
    : m_DataStorage(Value)
{
}


// Copy constructor
template <typename T, tSize M>
template <typename U>
inline CStaticVector<T,M>::CStaticVector(const CStaticVector<U,M>& Vector)
{
    T *p = m_DataStorage.getPtr();
    T *pMax = p + SIZE;
    const U *p2 = Vector.getPtr();
    while( p < pMax )
    {
        *(p++) = *(p2++);
    }
}


// Copy constructor
template <typename T, tSize M>
inline CStaticVector<T,M>::CStaticVector(const CStaticVector<T,M>& Vector)
    : m_DataStorage(Vector.m_DataStorage)
{
}


// Assignment operator
template <typename T, tSize M>
template <typename U>
inline CStaticVector<T,M>& CStaticVector<T,M>::operator =(const CStaticVector<U,M>& Vector)
{
    T *p = m_DataStorage.getPtr();
    T *pMax = p + SIZE;
    const U *p2 = Vector.getPtr();
    while( p < pMax )
    {
        *(p++) = *(p2++);
    }

    return *this;
}


// Assignment operator
template <typename T, tSize M>
inline CStaticVector<T,M>& CStaticVector<T,M>::operator =(const CStaticVector<T,M>& Vector)
{
    m_DataStorage = Vector.m_DataStorage;

    return *this;
}


template <typename T, tSize M>
inline CStaticVector<T,M>& CStaticVector<T,M>::operator +=(const CStaticVector<T,M>& Vector)
{
    memVectAdd(m_DataStorage.getPtr(), Vector.getPtr(), SIZE);

    return *this;
}


template <typename T, tSize M>
inline CStaticVector<T,M>& CStaticVector<T,M>::operator -=(const CStaticVector<T,M>& Vector)
{
    memVectSub(m_DataStorage.getPtr(), Vector.getPtr(), SIZE);

    return *this;
}


template <typename T, tSize M>
inline CStaticVector<T,M>& CStaticVector<T,M>::operator *=(const CStaticVector<T,M>& Vector)
{
    memVectMult(m_DataStorage.getPtr(), Vector.getPtr(), SIZE);

    return *this;
}


template <typename T, tSize M>
inline CStaticVector<T,M>& CStaticVector<T,M>::operator /=(const CStaticVector<T,M>& Vector)
{
    memVectDiv(m_DataStorage.getPtr(), Vector.getPtr(), SIZE);

    return *this;
}


template <typename T, tSize M>
template <typename U>
inline CStaticVector<T,M>& CStaticVector<T,M>::operator +=(const U& c)
{
    memAdd(m_DataStorage.getPtr(), c, SIZE);

    return *this;
}


template <typename T, tSize M>
template <typename U>
inline CStaticVector<T,M>& CStaticVector<T,M>::operator -=(const U& c)
{
    memSub(m_DataStorage.getPtr(), c, SIZE);

    return *this;
}


template <typename T, tSize M>
template <typename U>
inline CStaticVector<T,M>& CStaticVector<T,M>::operator *=(const U& c)
{
    memMult(m_DataStorage.getPtr(), c, SIZE);

    return *this;
}


template <typename T, tSize M>
template <typename U>
inline CStaticVector<T,M>& CStaticVector<T,M>::operator /=(const U& c)
{
    memDiv(m_DataStorage.getPtr(), c, SIZE);

    return *this;
}


template <typename T, tSize M>
inline void CStaticVector<T,M>::zeros()
{
    memSet(m_DataStorage.getPtr(), T(0), SIZE);
}


template <typename T, tSize M>
inline void CStaticVector<T,M>::ones()
{
    memSet(m_DataStorage.getPtr(), T(1), SIZE);
}


template <typename T, tSize M>
inline void CStaticVector<T,M>::fill(const T& Value)
{
    memSet(m_DataStorage.getPtr(), Value, SIZE);
}


template <typename T, tSize M>
inline void CStaticVector<T,M>::abs()
{
    T *p = m_DataStorage.getPtr();
    T *pMax = p + SIZE;
    while( p < pMax )
    {
        if( !(*p > T(0)) )
        {
            *p = -(*p);
        }
        ++p;
    }
}


template <typename T, tSize M>
inline void CStaticVector<T,M>::multAdd(const CStaticVector<T,M>& Vector, const T& Scalar)
{
    memVectMultAdd(m_DataStorage.getPtr(), Vector.getPtr(), Scalar, SIZE);
}


// Matrix and vector multiplication
template <typename T, tSize M>
template <tSize K>
inline void CStaticVector<T,M>::mult(const CStaticMatrix<T,M,K>& Matrix,
                                     const CStaticVector<T,K>& Vector
                                    )
{
    for( tSize i = 0; i < SIZE; ++i )
    {
        T *p = m_DataStorage.getPtr(i);

        const T *pV = Vector.getPtr();
        const T *pVMax = pV + K;
        const T *pM = Matrix.getRowPtr(i);

        *p = T(0);
        while (pV < pVMax)
        {
            *p += *(pM++) * *(pV++);
        }
    }
}


// Vector and matrix multiplication
template <typename T, tSize M>
template <tSize K>
inline void CStaticVector<T,M>::mult(const CStaticVector<T,K>& Vector,
                                     const CStaticMatrix<T,K,M>& Matrix
                                    )
{
    for( tSize i = 0; i < SIZE; ++i )
    {
        T *p = m_DataStorage.getPtr(i);

        const T *pV = Vector.getPtr();
        const T *pVMax = pV + K;
        const T *pM = Matrix.getColPtr(i);

        *p = T(0);
        while (pV < pVMax)
        {
            *p += *pM * *(pV++);
            pM += M;
        }
    }
}

