//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsStaticVector.h                   \n
 * Section: libMath                             \n
 * Date:    2004/05/06                          \n
 *
 * $Id: mdsStaticVector.h 345 2007-06-11 13:23:09Z spanel $
 * 
 * Description:
 * - Static vector operations.
 */

#ifndef MDS_STATICVECTOR_H
#define MDS_STATICVECTOR_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsWarning.h>
#include <MDSTk/Base/mdsError.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsStaticData.h>
//#include <MDSTk/Base/mdsSmallObject.h>

#include "mdsBase.h"
#include "mdsVectorBase.h"
#include "mdsVectorIterator.h"
#include "mdsStaticMatrix.h"

// STL
#include <cmath>


namespace mds
{
namespace math
{

//==============================================================================
/*!
 * Template for a static vector.
 * - T is a vector element type.
 * - M is vector size.
 */
template <typename T, tSize M>
class CStaticVector : /*public mds::base::CSmallObject<>,*/ public CVectorBase<CStaticVector<T,M> >
{
public:
    //! Templates that require members of the CStaticVector class can use this
    //! enum to check the existence.
    enum { CLASS_VECTOR };

    //! Vector element type.
    typedef T tElement;

    //! The vector size.
    enum { SIZE = M };

    //! Vector data.
    typedef mds::base::CStaticData<T,SIZE> tDataStorage;

    //! Iterator type.
    //! - Declares types tIterator and tConstIterator.
    MDS_COMPLEX_ITERATOR(CStaticVector, tElement, CVectorIterator);

public:
    //! Constructor.
    inline CStaticVector();

    //! Constructor.
    inline CStaticVector(const T& Value);

    //! Copy constructor.
    template <typename U>
    inline CStaticVector(const CStaticVector<U,M>& Vector);

    //! Specialization of the copy constructor.
    inline CStaticVector(const CStaticVector& Vector);

    //! Destructor.
    ~CStaticVector() {}

    //! Assignment operator.
    template <typename U>
    inline CStaticVector& operator =(const CStaticVector<U,M>& Vector);

    //! Specialization of the assignment operator.
    inline CStaticVector& operator =(const CStaticVector& Vector);


    //! Returns the vector size.
    tSize getSize() const { return SIZE; }

    //! Returns stride between two neigbouring vector elements.
    tSize getStride() const { return 1; }

    //! Returns the subscripted element [i].
    T& operator ()(tSize i)
    {
        return m_DataStorage(i);
    }
    const T& operator ()(tSize i) const
    {
        return m_DataStorage(i);
    }

    //! Gets the element [i].
    T& get(tSize i)
    {
        return m_DataStorage.get(i);
    }
    const T& get(tSize i) const
    {
        return m_DataStorage.get(i);
    }

    //! Returns pointer to the given element.
    T *getPtr(tSize i)
    {
        return m_DataStorage.getPtr(i);
    }
    const T *getPtr(tSize i) const
    {
        return m_DataStorage.getPtr(i);
    }

    //! Returns pointer to the vector data.
    T *getPtr() { return m_DataStorage.getPtr(); }
    const T *getPtr() const { return m_DataStorage.getPtr(); }

    //! Sets the vector element at the position [i].
    void set(tSize i, const T& Value)
    {
        m_DataStorage(i) = Value;
    }


    //! Returns iterator that points at the first vector element.
    tIterator getBegin()
    {
        return tIterator(*this, 0);
    }
    tConstIterator getBegin() const
    {
        return tConstIterator(*this, 0);
    }

    //! Returns iterator that points beyond the last vector element.
    tIterator getEnd()
    {
        return tIterator(*this, SIZE);
    }
    tConstIterator getEnd() const
    {
        return tConstIterator(*this, SIZE);
    }

    //! Returns iterator that points at a specified position.
    tIterator getIterator(tSize i)
    {
        return tIterator(*this, i);
    }
    tConstIterator getIterator(tSize i) const
    {
        return tConstIterator(*this, i);
    }


    //! Combined assignment operator.
    inline CStaticVector& operator +=(const CStaticVector& v);
    inline CStaticVector& operator -=(const CStaticVector& v);
    inline CStaticVector& operator *=(const CStaticVector& v);
    inline CStaticVector& operator /=(const CStaticVector& v);

    //! Combined assignment operator.
    template <typename U>
    inline CStaticVector& operator +=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CStaticVector& operator -=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CStaticVector& operator *=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CStaticVector& operator /=(const U& c);


    //! The vector initialization method.
    inline void zeros();

    //! The vector initialization method.
    inline void ones();

    //! Sets all elements of the vector to a given value.
    inline void fill(const T& Value);

    //! Absolute value of all elements.
    inline void abs();

    //! Provides operation x = x + Vector * Scalar.
    inline void multAdd(const CStaticVector& Vector, const T& Scalar);

    //! Matrix vs. vector multiplication.
    template <tSize K>
    inline void mult(const CStaticMatrix<T,M,K>& Matrix,
                     const CStaticVector<T,K>& Vector
                     );

    //! Vector vs. matrix multiplication.
    template <tSize K>
    inline void mult(const CStaticVector<T,K>& Vector,
                     const CStaticMatrix<T,K,M>& Matrix
                     );

protected:
    //! Vector data.
    tDataStorage m_DataStorage;
};


//==============================================================================
/*
 * Method templates.
 */

// Include the file containing method templates.
#include "mdsStaticVector.hxx"


//==============================================================================
/*
 * Basic template instances and type definitions.
 */

//! Vector of int numbers.
typedef CStaticVector<int, 2>       CIVector2;
typedef CStaticVector<int, 3>       CIVector3;
typedef CStaticVector<int, 4>       CIVector4;

//! Vector of float numbers.
typedef CStaticVector<float, 2>     CFVector2;
typedef CStaticVector<float, 3>     CFVector3;
typedef CStaticVector<float, 4>     CFVector4;

//! Vector of double numbers.
typedef CStaticVector<double, 2>    CDVector2;
typedef CStaticVector<double, 3>    CDVector3;
typedef CStaticVector<double, 4>    CDVector4;


} // namespace math
} // namespace mds

#endif // MDS_STATICVECTOR_H

