//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsStaticMatrix.hxx                 \n
 * Section: libMath                             \n
 * Date:    2004/06/05                          \n
 *
 * $Id: mdsStaticMatrix.hxx 449 2007-09-05 10:50:05Z siler $
 * 
 * Description:
 * - Static matrix template implementation.
 */


//==============================================================================
/*
 * Method templates
 */
template <typename T, tSize M, tSize N>
inline CStaticMatrix<T,M,N>::CStaticMatrix()
{
}


template <typename T, tSize M, tSize N>
inline CStaticMatrix<T,M,N>::CStaticMatrix(const T& Value)
    : m_DataStorage(Value)
{
}


// Copy constructor
template <typename T, tSize M, tSize N>
template <typename U>
inline CStaticMatrix<T,M,N>::CStaticMatrix(const CStaticMatrix<U,M,N>& Matrix)
{
    for( tSize i = 0; i < M; ++i )
    {
        for( tSize j = 0; j < N; ++j )
        {
            m_DataStorage(i * N + j) = Matrix(i, j);
        }
    }
}


// Copy constructor
template <typename T, tSize M, tSize N>
inline CStaticMatrix<T,M,N>::CStaticMatrix(const CStaticMatrix<T,M,N>& Matrix)
    : m_DataStorage(Matrix.m_DataStorage)
{
}


// Assignment operator
template <typename T, tSize M, tSize N>
template <typename U>
inline CStaticMatrix<T,M,N>& CStaticMatrix<T,M,N>::operator =(const CStaticMatrix<U,M,N>& Matrix)
{
    for( tSize i = 0; i < M; ++i )
    {
        for( tSize j = 0; j < N; ++j )
        {
            m_DataStorage(i * N + j) = Matrix(i, j);
        }
    }

    return *this;
}


// Assignment operator
template <typename T, tSize M, tSize N>
inline CStaticMatrix<T,M,N>& CStaticMatrix<T,M,N>::operator =(const CStaticMatrix<T,M,N>& Matrix)
{
    m_DataStorage = Matrix.m_DataStorage;

    return *this;
}


// Combined addition and assignment operator
template <typename T, tSize M, tSize N>
inline CStaticMatrix<T,M,N>& CStaticMatrix<T,M,N>::operator +=(const CStaticMatrix<T,M,N>& Matrix)
{
    memVectAdd(m_DataStorage.getPtr(), Matrix.getPtr(), SIZE);

    return *this;
}


// Combined subtraction and assignment operator
template <typename T, tSize M, tSize N>
inline CStaticMatrix<T,M,N>& CStaticMatrix<T,M,N>::operator -=(const CStaticMatrix<T,M,N>& Matrix)
{
    memVectSub(m_DataStorage.getPtr(), Matrix.getPtr(), SIZE);

    return *this;
}


// Combined multiplication and assignment operator
template <typename T, tSize M, tSize N>
inline CStaticMatrix<T,M,N>& CStaticMatrix<T,M,N>::operator *=(const CStaticMatrix<T,M,N>& Matrix)
{
    memVectMult(m_DataStorage.getPtr(), Matrix.getPtr(), SIZE);

    return *this;
}


// Combined division and assignment operator
template <typename T, tSize M, tSize N>
inline CStaticMatrix<T,M,N>& CStaticMatrix<T,M,N>::operator /=(const CStaticMatrix<T,M,N>& Matrix)
{
    memVectDiv(m_DataStorage.getPtr(), Matrix.getPtr(), SIZE);

    return *this;
}


// Combined scalar multiplication and assignment operator
template <typename T, tSize M, tSize N>
template <typename U>
inline CStaticMatrix<T,M,N>& CStaticMatrix<T,M,N>::operator *=(const U& c)
{
    memMult(m_DataStorage.getPtr(), c, SIZE);

    return *this;
}


// Combined scalar division and assignment operator
template <typename T, tSize M, tSize N>
template <typename U>
inline CStaticMatrix<T,M,N>& CStaticMatrix<T,M,N>::operator /=(const U& c)
{
    memDiv(m_DataStorage.getPtr(), c, SIZE);

    return *this;
}


// Combined scalar addition and assignment operator
template <typename T, tSize M, tSize N>
template <typename U>
inline CStaticMatrix<T,M,N>& CStaticMatrix<T,M,N>::operator +=(const U& c)
{
    memAdd(m_DataStorage.getPtr(), c, SIZE);

    return *this;
}


// Combined scalar subtraction and assignment operator
template <typename T, tSize M, tSize N>
template <typename U>
inline CStaticMatrix<T,M,N>& CStaticMatrix<T,M,N>::operator -=(const U& c)
{
    memSub(m_DataStorage.getPtr(), c, SIZE);

    return *this;
}


// Set all elements of a matrix to zero
template <typename T, tSize M, tSize N>
inline void CStaticMatrix<T,M,N>::zeros()
{
    memSet(m_DataStorage.getPtr(), T(0), SIZE);
}


// Set all elements of a matrix to one
template <typename T, tSize M, tSize N>
inline void CStaticMatrix<T,M,N>::ones()
{
    memSet(m_DataStorage.getPtr(), T(1), SIZE);
}


// Set a matrix to unit
template <typename T, tSize M, tSize N>
inline void CStaticMatrix<T,M,N>::unit()
{
    for( tSize i = 0; i < M; ++i )
    {
        for( tSize j = 0; j < N; ++j )
        {
            m_DataStorage(i * N + j) = (i == j) ? T(1) : T(0);
        }
    }
}


template <typename T, tSize M, tSize N>
inline void CStaticMatrix<T,M,N>::fill(const T& Value)
{
    memSet(m_DataStorage.getPtr(), Value, SIZE);
}


template <typename T, tSize M, tSize N>
inline void CStaticMatrix<T,M,N>::setRow(tSize i, const T *pSrc)
{
    memCopy(m_DataStorage.getPtr(i * N), pSrc, N);
}


template <typename T, tSize M, tSize N>
inline void CStaticMatrix<T,M,N>::copyRow(tSize i, T *pDst) const
{
    memCopy(pDst, m_DataStorage.getPtr(i * N), N);
}


template <typename T, tSize M, tSize N>
inline void CStaticMatrix<T,M,N>::swapRows(tSize i, tSize j)
{
    if( i == j )
    {
        return;
    }

    memSwap(m_DataStorage.getPtr(i * N), m_DataStorage.getPtr(j * N), N);
}


template <typename T, tSize M, tSize N>
inline void CStaticMatrix<T,M,N>::abs()
{
    for( tSize j = 0; j < M; ++j )
    {
        T *p = m_DataStorage.getPtr(j * N);
        T *pMax = p + N;
        while( p < pMax )
        {
            if( !(*p > T(0)) )
            {
                *p = -(*p);
            }
            ++p;
        }
    }
}


template <typename T, tSize M, tSize N>
inline void CStaticMatrix<T,M,N>::multAdd(const CStaticMatrix<T,M,N>& Matrix, const T& Scalar)
{
    memVectAddMult(m_DataStorage.getPtr(), Matrix.getPtr(), Scalar, SIZE);
}


// Matrix multiplication
template <typename T, tSize M, tSize N>
template <tSize K>
inline void CStaticMatrix<T,M,N>::mult(const CStaticMatrix<T,M,K>& Matrix1,
                                       const CStaticMatrix<T,K,N>& Matrix2
                                      )
{
    for( tSize i = 0; i < M; ++i )
    {
        for( tSize j = 0; j < N; ++j )
        {
            T *p = m_DataStorage.getPtr(i * N + j);

            const T *p1 = Matrix1.getRowPtr(i);
            const T *p1Max = p1 + K;
            const T *p2 = Matrix2.getColPtr(j);

            *p = T(0);
            while( p1 < p1Max )
            {
                *p += *(p1++) * *p2;
                p2 += N;
            }
        }
    }
}


// Compute a matrix transpose
template <typename T, tSize M, tSize N>
inline void CStaticMatrix<T,M,N>::transpose(const CStaticMatrix<T,N,M>& Matrix)
{
    for( tSize i = 0; i < M; ++i )
    {
        for( tSize j = 0; j < N; ++j )
        {
            m_DataStorage(i * N + j) = Matrix(j, i);			
        }
    }
}

