//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsStaticMatrix.h                   \n
 * Section: libMath                             \n
 * Date:    2004/05/06                          \n
 *
 * $Id: mdsStaticMatrix.h 345 2007-06-11 13:23:09Z spanel $
 *
 * Description:
 * - Static dense matrix.
 */

#ifndef MDS_STATICMATRIX_H
#define MDS_STATICMATRIX_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsWarning.h>
#include <MDSTk/Base/mdsError.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsStaticData.h>
//#include <MDSTk/Base/mdsSmallObject.h>

#include "mdsBase.h"
#include "mdsMatrixBase.h"
#include "mdsMatrixIterator.h"

// STL
#include <cmath>
#include <ostream>


namespace mds
{
namespace math
{

//==============================================================================
/*!
 * Template for a two dimensional dense matrix.
 * - T is a matrix element type.
 * - M and N are matrix dimensions (the number of rows and columns).
 */
template <typename T, tSize M, tSize N>
class CStaticMatrix : /*public mds::base::CSmallObject<>,*/ public CMatrixBase<CStaticMatrix<T,M,N> >
{
public:
    //! Templates that require members of the CStaticMatrix class can use this
    //! enum to check the existence.
    enum { CLASS_MATRIX };

    //! Size of the matrix data.
    enum { SIZE = M * N };

    //! Matrix element type.
    typedef T tElement;

    //! Matrix data.
    typedef mds::base::CStaticData<T,SIZE> tDataStorage;

    //! Iterator type.
    //! - Declares types tIterator and tConstIterator.
    MDS_COMPLEX_ITERATOR(CStaticMatrix, tElement, CMatrixIterator);

public:
    //! Constructor.
    inline CStaticMatrix();

    //! Constructor.
    inline CStaticMatrix(const T& Value);

    //! Copy constructor.
    template <typename U>
    inline CStaticMatrix(const CStaticMatrix<U,M,N>& Matrix);

    //! Copy constructor specialization.
    inline CStaticMatrix(const CStaticMatrix& Matrix);

    //! Destructor.
    ~CStaticMatrix() {}

    //! Assignment operator.
    template <typename U>
    inline CStaticMatrix& operator =(const CStaticMatrix<U,M,N>& Matrix);

    //! Assignment operator specialization.
    inline CStaticMatrix& operator =(const CStaticMatrix& Matrix);


    //! Returns the matrix dimensions.
    tSize getNumOfRows() const { return M; }
    tSize getNumOfCols() const { return N; }

    //! Returns row offset. In other words, offset between two neighbouring
    //! elements in a matrix column.
    tSize getRowOffset() const { return N; }

    //! Returns column offset. In other words, offset between two neighbouring
    //! elements in a matrix row.
    tSize getColOffset() const { return 1; }

    //! Returns the subscripted element [Row][Col].
    T& operator ()(tSize Row, tSize Col)
    {
        return m_DataStorage(Row * N + Col);
    }
    const T& operator ()(tSize Row, tSize Col) const
    {
        return m_DataStorage(Row * N + Col);
    }

    //! Gets the element [Row][Col].
    T& get(tSize Row, tSize Col)
    {
        return m_DataStorage.get(Row * N + Col);
    }
    const T& get(tSize Row, tSize Col) const
    {
        return m_DataStorage.get(Row * N + Col);
    }

    //! Returns pointer to the given element.
    T *getPtr(tSize Row, tSize Col)
    {
        return m_DataStorage.getPtr(Row * N + Col);
    }
    const T *getPtr(tSize Row, tSize Col) const
    {
        return m_DataStorage.getPtr(Row * N + Col);
    }

    //! Returns pointer to the first element of matrix row.
    T *getRowPtr(tSize Row)
    {
        return m_DataStorage.getPtr(Row * N);
    }
    const T *getRowPtr(tSize Row) const
    {
        return m_DataStorage.getPtr(Row * N);
    }

    //! Returns pointer to the first element of matrix column.
    T *getColPtr(tSize Col)
    {
        return m_DataStorage.getPtr(Col);
    }
    const T *getColPtr(tSize Col) const
    {
        return m_DataStorage.getPtr(Col);
    }

    //! Returns pointer to the matrix data.
    T *getPtr() { return m_DataStorage.getPtr(); }
    const T *getPtr() const { return m_DataStorage.getPtr(); }

    //! Sets the element of matrix at the position [Row][Col].
    void set(tSize Row, tSize Col, const T& Value)
    {
        m_DataStorage(Row * N + Col) = Value;
    }


    //! Returns iterator that points at the first matrix element.
    tIterator getBegin()
    {
        return tIterator(*this, 0, 0);
    }
    tConstIterator getBegin() const
    {
        return tConstIterator(*this, 0, 0);
    }

    //! Returns iterator that points beyond the last matrix element.
    tIterator getEnd()
    {
        return tIterator(*this, M, N);
    }
    tConstIterator getEnd() const
    {
        return tConstIterator(*this, M, N);
    }

    //! Returns iterator that points at a specified position.
    tIterator getIterator(tSize Row, tSize Col)
    {
        return tIterator(*this, Row, Col);
    }
    tConstIterator getIterator(tSize Row, tSize Col) const
    {
        return tConstIterator(*this, Row, Col);
    }


    //! Combined assignment operator.
    inline CStaticMatrix& operator +=(const CStaticMatrix& Matrix);
    inline CStaticMatrix& operator -=(const CStaticMatrix& Matrix);
    inline CStaticMatrix& operator *=(const CStaticMatrix& Matrix);
    inline CStaticMatrix& operator /=(const CStaticMatrix& Matrix);

    //! Combined assignment operator.
    template <typename U>
    inline CStaticMatrix& operator +=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CStaticMatrix& operator -=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CStaticMatrix& operator *=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CStaticMatrix& operator /=(const U& c);


    //! The matrix initialization.
    //! - Zeros the matrix.
    inline void zeros();

    //! The matrix initialization.
    //! - Sets all elements to one.
    inline void ones();

    //! The matrix initialization.
    //! - Makes the matrix unit.
    inline void unit();

    //! Sets all matrix elements to a given value.
    inline void fill(const T& Value);

    //! Makes absolute value of all elements.
    inline void abs();

    //! Fills the i-th matrix row using a given input buffer.
    inline void setRow(tSize i, const T *pSrc);

    //! Copies the i-th matrix row to the output buffer
    inline void copyRow(tSize i, T *pDst) const;

    //! Swaps the i-th and j-th matrix rows.
    inline void swapRows(tSize i, tSize j);

    //! Checks type of the matrix.
    bool isSquare() const { return (M == N); }

    //! Transpose of the matrix.
    inline void transpose(const CStaticMatrix<T,N,M>& Matrix);

    //! Provides operation x = x + Matrix * Scalar.
    inline void multAdd(const CStaticMatrix& Matrix, const T& Scalar);

    //! Matrix multiplication.
    template <tSize K>
    inline void mult(const CStaticMatrix<T,M,K>& Matrix1,
                     const CStaticMatrix<T,K,N>& Matrix2
                     );

protected:
    //! Matrix data.
    tDataStorage m_DataStorage;
};


//==============================================================================
/*
 * Method templates.
 */

// Include the file containing method templates.
#include "mdsStaticMatrix.hxx"


//==============================================================================
/*
 * Basic template instances and type definitions.
 */

//! Matrix of int numbers.
typedef CStaticMatrix<int, 2, 2>        CIMatrix2x2;
typedef CStaticMatrix<int, 3, 3>        CIMatrix3x3;
typedef CStaticMatrix<int, 4, 4>        CIMatrix4x4;

//! Matrix of float numbers.
typedef CStaticMatrix<float, 2, 2>      CFMatrix2x2;
typedef CStaticMatrix<float, 3, 3>      CFMatrix3x3;
typedef CStaticMatrix<float, 4, 4>      CFMatrix4x4;

//! Matrix of double numbers.
typedef CStaticMatrix<double, 2, 2>     CDMatrix2x2;
typedef CStaticMatrix<double, 3, 3>     CDMatrix3x3;
typedef CStaticMatrix<double, 4, 4>     CDMatrix4x4;


} // namespace math
} // namespace mds

#endif // MDS_STATICMATRIX_H

