//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by PGMed@FIT         \n
 *
 * Author:  Miroslav Svub, svub@fit.vutbr.cz    \n
 * File:    mdsSparseSystem.h                   \n
 * Section: libMath                             \n
 * Date:    2007/04/01                          \n
 *
 * $Id: mdsSparseSystem.h 400 2007-06-26 08:20:16Z spanel $
 * 
 * Description:
 * - Sparse system solver.
 */

#ifndef MDS_SPARSESYSTEM_H
#define MDS_SPARSESYSTEM_H

#include "mdsSparseMatrix.h"
#include "mdsUMFpack.h"


namespace mds
{
namespace math
{

//==============================================================================
/*
 * Global functions.
 */

//! This function solves sparse linear system Ax = b.
template <typename T>
void solve(CSparseMatrix<T>& A, CVector<T>& b, CVector<T>& x)
{
    MDS_THROW_WARNING( "Function not implemented for this type" );
}


//! Defined for sparse systems of double values
template <>
void solve(CSparseMatrix<double>& A, CVector<double>& b, CVector<double>& x)
{
    typedef CSparseMatrix<double> tMatrix;

    // test if passed vectors have stride of 1, which allows to directly access storage memory
    if( b.getStride() > 1 || x.getStride() > 1 )
        MDS_THROW_EXCEPTION( "Vectors of stride different from 1 aren't allowed");

    // test whether right hand side is suitable for matrix of the system
    if( b.getSize() != A.getNumOfCols() )
        MDS_THROW_EXCEPTION( "Right hand side b and matrix A do not match");

    // cast non-zero iterator
    tMatrix::tIterator it = A.getBegin();

    // resize solution vector if necessary
    if( x.getSize() != A.getNumOfCols() ) x.create(A.getNumOfCols(), 0.0);

    // create auxiliary arrays to use when calling umfpack routines
    int * Ap     = new int[A.getNumOfCols() + 1];
    int * Ai     = new int[A.getNumOfNonDominant()];
    double * Ax  = new double[A.getNumOfNonDominant()];
    double * rhs = b.getPtr(0);
    double * sol = x.getPtr(0);

    // iterate over matrix and convert it into compressed column form
    int col = -1;
    int i   = 0;
    int j   = 0;

    col = it.getCol();
    Ap[j] = 0;
    while( it != A.getEnd() )
    {
        if( it.getCol() > col )
        {
            col = it.getCol();
            Ap[++j] = i;
        }

        Ai[i] = it.getRow();
        Ax[i] = *it;
        ++i;
        ++it;
    }
    Ap[++j] = i;

    // define some umfpack-ish stuff
    double *null = (double *)NULL;
    void *Symbolic, *Numeric ;

    // call umfpack's fill-up reducing column reordering procedure
    (void)umfpack_di_symbolic(A.getNumOfRows(), A.getNumOfCols(), Ap, Ai,
                              Ax, &Symbolic, null, null
                              );

    // perform LU factorization
    (void)umfpack_di_numeric(Ap, Ai, Ax, Symbolic, &Numeric, null, null);

    // get rid of temporary results
    umfpack_di_free_symbolic(&Symbolic);

    // solve LU factorized system
    (void)umfpack_di_solve(UMFPACK_A, Ap, Ai, Ax, sol, rhs, Numeric, null, null);

    // get rid of everything
    umfpack_di_free_numeric(&Numeric);

    // free everything
    delete[] Ap;
    delete[] Ai;
    delete[] Ax;
}

} // namespace math
} // namespace mds

#endif // MDS_SPARSESYSTEM_H

