//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by PGMed@FIT         \n
 *
 * Author:  Miroslav Svub, svub@fit.vutbr.cz    \n
 * File:    mdsSparseMatrixIterator.h           \n
 * Section: libMath                             \n
 * Date:    2007/04/01                          \n
 *
 * $Id: mdsSparseMatrixIterator.h 391 2007-06-22 11:29:28Z spanel $
 * 
 * Description:
 * - Sparse matrix iterator definition.
 */

#ifndef MDS_SPARSEMATRIXITERATOR_H
#define MDS_SPARSEMATRIXITERATOR_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsIterator.h>


namespace mds
{
namespace math
{

//==============================================================================
/*!
 * Sparse matrix iterator.
 * - Warning semantics of it's methods differ from that of standard 
 *   matrix iterator.
 *   1. It iterates over non-zero elements only.
 *   2. Iteration is column oriented.
 * - Template parameter M is the matrix type.
 * - Parameter T is the matrix element type.
 */
template <class M, typename T>
class CSparseMatrixIterator : public mds::base::CComplexIteratorBase<M,T>
{
public:
    //! Check that M is a matrix. You will see name of this enum somewhere
    //! in compiler error message if the type M is not matrix.
    enum { TEMPLATE_PARAMETER_IS_NOT_SPARSE_MATRIX = M::CLASS_SPARSEMATRIX };

    //! Iterator base.
    typedef mds::base::CComplexIteratorBase<M,T> base;

    //! Matrix type.
    typedef M tMatrix;

    //! Element type.
    typedef T tElement;

    //! Column iterator.
    typedef typename tMatrix::tColIterator tColumnIterator;

public:
    //! Default constructor.
    CSparseMatrixIterator() {}

    //! Constructor.
    CSparseMatrixIterator( tMatrix& Matrix, tSize Row, tSize Col )
        : mds::base::CComplexIteratorBase<M,T>(&Matrix, &(Matrix.getEntry(Row, Col).second) )
        , m_Row(Row)
        , m_Col(Col)
    {
        setUp();
    }

    //! Constructor.
    CSparseMatrixIterator(tMatrix & Matrix, tElement *el)
        : mds::base::CComplexIteratorBase<M,T>(&Matrix, el)
        , m_Row(0)
        , m_Col(0)
    {
        setUp();
    }

    //! Copy constructor.
    CSparseMatrixIterator( const CSparseMatrixIterator & It)
        : mds::base::CComplexIteratorBase<M,T>(It)
        , m_Row(It.m_Row)
        , m_Col(It.m_Col)
    {
        setUp();
    }

    //! Destructor.
    ~CSparseMatrixIterator() {}

    //! Assignment operator.
    CSparseMatrixIterator& operator=(const CSparseMatrixIterator& It)
    {
        base::m_pContainer = It.m_pContainer;
        base::m_pItem = It.m_pItem;
        m_Row = It.m_Row;
        m_Col = It.m_Col;
        m_ColIt = It.m_ColIt;
        return *this;
    }

    //! Operator moves the iterator to the next pixel.
    //! - Prefix notation.
    CSparseMatrixIterator& operator++()
    {
        advance();
        return *this;
    }

    //! Operator moves the iterator to the next pixel.
    //! - Postfix notation.
    CSparseMatrixIterator operator++(int)
    {
        CSparseMatrixIterator Temp(*this);
        advance();
        return Temp;
    }

/*    //! Returns the number of increments between the positions addressed
    //! by two iterators.
    tSize getDistance(const CMatrixIterator& End)
    {
        tSize Count = End.m_Row * End.m_pContainer->getNumOfCols() + End.m_Col;
        Count -= m_Row * base::m_pContainer->getNumOfCols() + m_Col;
        return (Count > 0) ? Count : 0;
    }*/

    //! Returns current iterator position.
    tSize getRow() const { return m_Row; }
    tSize getCol() const { return m_Col; }

/*    //! Sets the iterator position.
    void setPosition(tMatrix& Matrix, tSize Row, tSize Col)
    {
        base::m_pContainer = &Matrix;
        base::m_pItem = Matrix.getPtr(Row, Col);
        m_Row = Row;
        m_Col = Col;
    }*/

protected:
    //! Iterator position.
    tSize m_Row, m_Col;

    //! Iterator true position in sparse column.
    tColumnIterator m_ColIt;

protected:
    //! Moves iterator to the next element.
    void advance()
    {
        if( base::m_pItem != base::m_pContainer->getNull() )
        {
            ++m_ColIt;
            while( m_ColIt == base::m_pContainer->getColEnd(m_Col) )
            {
                ++m_Col;
                if( m_Col >= base::m_pContainer->getNumOfCols() )
                {
                    base::m_pItem = base::m_pContainer->getNull();
                    return;
                }
                m_ColIt = base::m_pContainer->getColBegin(m_Col);
            }
            base::m_pItem = &m_ColIt->second;
            m_Row =  m_ColIt->first;
        }
    }

    //! Initialization.
    void setUp()
    {
        if( !base::m_pContainer->search(m_Row, m_Col, m_ColIt) )
        {
            m_ColIt = base::m_pContainer->getColBegin(0);
        }
    }
};


} // namespace math
} // namespace mds

#endif // MDS_SPARSEMATRIXITERATOR_H

