//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)   \n
 * Copyright (c) 2003-2007 by PGMed@FIT        \n
 *
 * Author:  Miroslav Svub, svub@fit.vutbr.cz   \n
 * File:    mdsSparseMatrix.hxx                \n
 * Section: libMath                            \n
 * Date:    2007/06/12                         \n
 *
 * $Id:$
 *
 * Description:
 * - Template providing basic sparse matrix operations.
 */


//==============================================================================
/*
 * Methods templates.
 */

template <typename T>
inline CSparseMatrix<T>::CSparseMatrix(tSize rows, tSize cols)
    : m_Matrix(cols)
    , m_Dominant(tElement(0))
    , m_NumOfRows(rows)
    , m_NumOfCols(cols)
    , m_NumOfNonDominant(0)
{
}


template <typename T>
inline CSparseMatrix<T>::CSparseMatrix(tSize rows, tSize cols, tElement dom)
    : m_Matrix(cols)
    , m_Dominant(dom)
    , m_NumOfRows(rows)
    , m_NumOfCols(cols)
    , m_NumOfNonDominant(0)
{
}


template <typename T>
CSparseMatrix<T>::~CSparseMatrix()
{
}


template <typename T>
inline void CSparseMatrix<T>::create(tSize rows, tSize cols)
{
    clear();

    m_NumOfCols = cols;
    m_NumOfRows = rows;

    m_Matrix.resize(cols);
}


template <typename T>
inline void CSparseMatrix<T>::create(tSize rows, tSize cols, tElement dominant)
{
    clear();

    m_NumOfCols = cols;
    m_NumOfRows = rows;
    m_Dominant = dominant;

    m_Matrix.resize(cols);
}


template <typename T>
inline void CSparseMatrix<T>::clear()
{
    for ( tSize i = 0; i < m_NumOfCols; i++ )
    {
        m_Matrix[i].erase( m_Matrix[i].begin(), m_Matrix[i].end());
    }
    m_NumOfNonDominant = 0;
}


template <typename T>
inline typename CSparseMatrix<T>::tElement CSparseMatrix<T>::get(tSize row, tSize col)
{
    tColIterator position;
    if( search(row, col, position) ) return position->second;
    else return m_Dominant;
}


template <typename T>
inline void CSparseMatrix<T>::set(tSize row, tSize col, tElement el)
{
    tColIterator position;
    if( search(row, col, position) )
    {
        position->second = el;
    }
    else
    {
        insert(row, col, position, el);
        ++m_NumOfNonDominant;
    }
}


template <typename T>
inline typename CSparseMatrix<T>::tIterator CSparseMatrix<T>::getBegin()
{
    tSize col = 0;
    while( m_Matrix[col].size() == 0 )
    {
        ++col;
    }
    return tIterator(*this, m_Matrix[col][0].first, 0);
}


template <typename T>
inline typename CSparseMatrix<T>::tConstIterator CSparseMatrix<T>::getBegin() const
{
    tSize col = 0;
    while( m_Matrix[col].size() == 0 )
    {
        ++col;
    }
    return tConstIterator(*this, m_Matrix[col][0].first, 0);
}


template <typename T>
inline typename CSparseMatrix<T>::tIterator CSparseMatrix<T>::getEnd()
{
    return tIterator(*this, getNull());
}


template <typename T>
inline typename CSparseMatrix<T>::tConstIterator CSparseMatrix<T>::getEnd() const
{
    return tConstIterator(*this, getNull());
}


template <typename T>
inline void CSparseMatrix<T>::lmult(const CVector<T>& vec, CVector<T>& result)
{
    if( vec.getSize() != m_NumOfRows )
    {
        MDS_THROW_EXCEPTION("Cannot multiply, dimensions do not match" );
    }
    
    result.create( m_NumOfCols, tElement(0));
    
    tColIterator it;
    if( m_Dominant == tElement(0) )
    {
        for( tSize i = 0; i < m_NumOfCols; i++ )
        {
            it = getColBegin( i );
    
            while( it != getColEnd(i) )
            {
                result.get(i) = result.get(i) + it->second * vec.get(it->first);
                ++it;
            }
        }
    }
}


template <typename T>
inline void CSparseMatrix<T>::rmult(const CVector<T>& vec, CVector<T>& result)
{
    if( vec.getSize() != m_NumOfCols )
    {
        MDS_THROW_EXCEPTION("cannot multiply - dimensions do not match" );
    }
    
    result.create( m_NumOfCols, tElement(0));
    
    tColIterator it;
    
    if( m_Dominant == tElement(0) )
    {
        for( tSize i = 0; i < m_NumOfCols; i++ )
        {
            it = getColBegin(i);
    
            while( it != getColEnd(i) )
            {
                result.get(it->first) = result.get(it->first) + vec.get(i) * it->second;
                ++it;
            }
        }
    }
}


template <typename T>
inline void CSparseMatrix<T>::zeros()
{
    clear();
    m_Dominant = tElement(0);
}


template <typename T>
inline void CSparseMatrix<T>::ones()
{
    clear();
    m_Dominant = tElement(1);
}


template <typename T>
inline void CSparseMatrix<T>::unit()
{
    clear();
    for( tSize i = 0; i < m_NumOfCols; i++ )
    {
        m_Matrix[i].push_back(tEntry(i, tElement(1)));
    }
    m_NumOfNonDominant = m_NumOfCols;
}


template <typename T>
inline void CSparseMatrix<T>::fill(const T& Value)
{
    clear();
    m_Dominant = Value;
}


template <typename T>
inline void CSparseMatrix<T>::abs()
{
    tColIterator   it;
    for( tSize i = 0; i < m_NumOfCols; i++ )
    {
        it = getColBegin(i);
        while( it != getColEnd(i) )
        {
            if( it->second < tElement(0) ) it->second = -(it->second);
        }
    }
}


template <typename T>
inline bool CSparseMatrix<T>::search(tSize row, tSize col, tColIterator& it)
{
    typedef std::pair<tColIterator, tColIterator> tBounds;
    
    // perform binary search and return iterator pair marking result
    tBounds bounds = std::equal_range(m_Matrix[col].begin(),
                                      m_Matrix[col].end(),
                                      tEntry(row, tElement(0)),
                                      CEntryCompare()
                                      );
    
    if( bounds.first == bounds.second ) // not found
    {
        it = bounds.second; // it points to first element with row index greater than desired
        return false;
    }
    else
    {
        it = bounds.first; // it points to found element
        return true;
    }
}


template <typename T>
inline void CSparseMatrix<T>::insert(tSize row,
                                     tSize col,
                                     tColIterator& col_pos,
                                     tElement element
                                     )
{
    m_Matrix[col].insert(col_pos, tEntry(row, element));
}


template <typename T>
inline typename CSparseMatrix<T>::tEntry & CSparseMatrix<T>::getEntry(tSize row, tSize col)
{
    tColIterator it;
    if( this->search(row, col, it) ) return *it;
    else return m_NullEntry;
}

