//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)   \n
 * Copyright (c) 2003-2007 by PGMed@FIT        \n
 *
 * Author:  Miroslav Svub, svub@fit.vutbr.cz   \n
 * File:    mdsSparseMatrix.h                  \n
 * Section: libMath                            \n
 * Date:    2007/04/01                         \n
 *
 * $Id: mdsSparseMatrix.h 364 2007-06-15 06:53:51Z svub $
 *
 * Description:
 * - Template providing basic sparse matrix operations.
 */

#ifndef MDS_SPARSEMATRIX_H
#define MDS_SPARSEMATRIX_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsException.h>

#include "mdsSparseMatrixIterator.h"
#include "mdsVector.h"

// STL
#include <vector>
#include <algorithm>
#include <iostream>


namespace mds
{
namespace math
{

/*
 */

//==============================================================================
/*!
 * Template for a two dimensional sparse matrix.
 * - T is a matrix element type.
 * 
 * Operations worst case complexity:
 * - get()                          O(log(column fill factor))  !!!!
 * - set()                          O(column fill factor)       !!!!
 * - rmult() - right vector mult    O(number of columns * column fill factor)
 * - lmult() - right vector mult    O(number of columns * column fill factor)
 * 
 * Notes:
 * - Do not use get for traversal - use tIterator (columnwise).
 */
template <typename T>
class CSparseMatrix : public mds::CObject, public CMatrixBase<CSparseMatrix<T> >
{
public:
    //! Templates that require members of the CMatrix class can use
    //! this enum to check the existence.
    enum { CLASS_SPARSEMATRIX };

    //! Standard method getClassName().
    MDS_CLASS_NAME(CSparseMatrix);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CSparseMatrix);

    //! Matrix element type
    typedef T tElement;

    //! One entry stored in column constists of row index and value.
    typedef std::pair<tSize, tElement> tEntry;

    //! Sparse matrix column is represented by vector of entries.
    typedef std::vector<tEntry> tColumn;

    //! Whole matrix structure is represented by vector of sparse columns.
    typedef std::vector<tColumn> tSparseMatrix;

    //! Iterator over non-dominant values of column.
    typedef typename tColumn::iterator tColIterator;

    //! Iterator type.
    //! - Declares types tIterator and tConstIterator.
    MDS_COMPLEX_ITERATOR(CSparseMatrix, tElement, CSparseMatrixIterator);

public:
    //! Constructor
    inline CSparseMatrix(tSize rows = 0, tSize cols = 0);

    //! Constructor, in addition parametrized by dominant value.
    inline CSparseMatrix(tSize rows, tSize cols, tElement dom);

    //! Destructor
    virtual ~CSparseMatrix();

    //! Creates new matrix of specified dimensions
    inline void create(tSize rows, tSize cols);

    //! Creates new matrix of specified dimensions and dominant entry
    inline void create(tSize rows, tSize cols, tElement dominant);

    //! Clears whole matrix
    inline void clear();


    //! Returns matrix dominant value.
    tElement getDominant() const { return m_Dominant; }

    //! Returns number of matrix rows
    tSize getNumOfRows() const { return m_NumOfRows; }

    //! Returns number of matrix columns
    tSize getNumOfCols() const  { return m_NumOfCols; }

    //! Returns number of non-dominant elements
    tSize getNumOfNonDominant() const { return m_NumOfNonDominant; }

    //! Method returns element at (row, col). If specified element doesn't exist,
    //! returns dominant value (by value).
    inline tElement get(tSize row, tSize col);

    //! Method sets value of element at (row, col). if element doesn't exist,
    //! new one is created.
    inline void set(tSize row, tSize col, tElement el);


    //! Returns iterator to the first nondominant entry of matrix.
    inline tIterator getBegin();

    //! returns constant iterator to the first nondominant entry of matrix
    inline tConstIterator getBegin() const;

    //! Returns end iterator.
    inline tIterator getEnd();

    //! Returns constant end iterator.
    inline tConstIterator getEnd() const;


    //! Matrix vector left multiplication
    inline void lmult(const CVector<T>& vec, CVector<T>& result);

    //! Matrix vector right multiplication
    inline void rmult(const CVector<T>& vec, CVector<T>& result);

    //! Sets all entries in matrix to zero.
    inline void zeros();

    //! The matrix initialization routine.
    //! - Sets all elements to one.
    inline void ones();

    //! The matrix initialization routine.
    //! - Makes the matrix unit.
    inline void unit();

    //! Fills the matrix using a given value.
    inline void fill(const T& Value);

    //! Makes absolute value of all elements.
    inline void abs();

    //! Type of the matrix.
    //! - Returns true if the matrix is square.
    inline bool isSquare() const { return (m_NumOfRows == m_NumOfCols); }


    //! Swaps the i-th and j-th matrix rows.
    inline void swapRows(tSize i, tSize j)
    {
        MDS_THROW_WARNING("Not implemented yet");
    }

    //! Swaps the i-th and j-th matrix columns.
    inline void swapCols(tSize i, tSize j)
    {
        MDS_THROW_WARNING("Not implemented yet");
    }

    //! Transpose of the matrix - to be added (tricky).
    inline void transpose(const CSparseMatrix & Matrix)
    {
        MDS_THROW_WARNING("Not implemented yet");
    }

    //! Stores the sparse matrix as dense.
    inline void asDense(CMatrix<T>& m)
    {
        MDS_THROW_WARNING("Not implemented yet");
    }

    // These methods are somehow controversial and should be protected
    // since used by iterator, this cannot be achieved.
    // Conclusion -> future rethinking of matrix.

public:
    //! Nested function class for comparing two std::pairs according to the first
    //! element instead of lexicographical.
    class CEntryCompare
    {
    public:
        bool operator() (const tEntry& e1, const tEntry& e2)
        {
            return (e1.first < e2.first);
        }
    };

public:
    //! Searches col-th column for index row, returns iterator to that element when
    //! found or iterator to first greater value in case i was not found via it parameter
    inline bool search(tSize row, tSize col, tColIterator& it);

    //! Inserts new element into matrix
    inline void insert(tSize row, tSize col, tColIterator& col_pos, tElement element);

    //! Returns iterator (stl vector iterator) to end of desired column.
    tColIterator getColEnd( tSize col ) { return m_Matrix[col].end(); }

    //! Returns iterator (stl vector iterator) to begining of desired column.
    tColIterator getColBegin( tSize col ) { return m_Matrix[col].begin(); }

    //! Returns reference to matrix internal entry.
    inline tEntry& getEntry(tSize row, tSize col);

    //! Returns null element of matrix
    tElement* getNull() { return &m_NullEntry.second; }

protected:
    //! Internal data storage.
    tSparseMatrix m_Matrix;

    //! Dominant matrix value.
    tElement m_Dominant;

    //! Null matrix entry.
    tEntry m_NullEntry;

    //! Number of rows.
    tSize m_NumOfRows;

    //! Number of columns.
    tSize m_NumOfCols;

    //! Number of nondominant elements.
    tSize m_NumOfNonDominant;
};


//==============================================================================
/*
 * Method templates.
 */

// Include the file containing method templates.
#include "mdsSparseMatrix.hxx"


//==============================================================================
/*
 * Basic template instances and type definitions.
 */

//! Matrix of int numbers.
typedef CSparseMatrix<int>    CISparseMatrix;

//! Matrix of float numbers.
typedef CSparseMatrix<float>  CFSparseMatrix;

//! Matrix of double numbers.
typedef CSparseMatrix<double> CDSparseMatrix;


//==============================================================================
/*
 * Basic template instances and type definitions.
 * - Using smart pointer
 */

//! Matrix of int numbers.
typedef CSparseMatrix<int>::tSmartPtr     CISparseMatrixPtr;

//! Matrix of float numbers.
typedef CSparseMatrix<float>::tSmartPtr     CFSparseMatrixPtr;

//! Matrix of double numbers.
typedef CSparseMatrix<double>::tSmartPtr     CDSparseMatrixPtr;


} // namespace math
} // namespace mds

#endif // MDS_SPARSEMATRIX_H

