//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsMatrixIterator.h                 \n
 * Section: libMath                             \n
 * Date:    2006/10/30                          \n
 *
 * $Id: mdsMatrixIterator.h 345 2007-06-11 13:23:09Z spanel $
 * 
 * Description:
 * - Matrix iterator definition.
 */

#ifndef MDS_MATRIXITERATOR_H
#define MDS_MATRIXITERATOR_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsIterator.h>


namespace mds
{
namespace math
{

//==============================================================================
/*!
 * Iterator used to traverse matrix elements by rows.
 * - Template parameter M is the matrix type.
 * - Parameter T is the matrix element type.
 */
template <class M, typename T>
class CMatrixIterator : public mds::base::CComplexIteratorBase<M,T>
{
public:
    //! Check that M is a matrix. You will see name of this enum somewhere
    //! in compiler error message if the type M is not matrix.
    enum { TEMPLATE_PARAMETER_IS_NOT_MATRIX = M::CLASS_MATRIX };

    //! Iterator base.
    typedef mds::base::CComplexIteratorBase<M,T> base;

    //! Matrix type.
    typedef M tMatrix;

    //! Element type.
    typedef T tElement;

public:
    //! Default constructor.
    CMatrixIterator() {}

    //! Constructor.
    CMatrixIterator(tMatrix& Matrix, tSize Row, tSize Col)
        : mds::base::CComplexIteratorBase<M,T>(&Matrix, Matrix.getPtr(Row, Col))
        , m_Row(Row)
        , m_Col(Col)
    {}

    //! Copy constructor.
    CMatrixIterator(const CMatrixIterator& It)
        : mds::base::CComplexIteratorBase<M,T>(It)
        , m_Row(It.m_Row)
        , m_Col(It.m_Col)
    {}

    //! Destructor.
    ~CMatrixIterator() {}

    //! Assignment operator.
    CMatrixIterator& operator=(const CMatrixIterator& It)
    {
        base::m_pContainer = It.m_pContainer;
        base::m_pItem = It.m_pItem;
        m_Row = It.m_Row;
        m_Col = It.m_Col;
        return *this;
    }


    //! Operator moves the iterator to the next pixel.
    //! - Prefix notation.
    CMatrixIterator& operator++()
    {
        advance();
        return *this;
    }

    //! Operator moves the iterator to the next pixel.
    //! - Postfix notation.
    CMatrixIterator operator++(int)
    {
        CMatrixIterator Temp(*this);
        advance();
        return Temp;
    }

    //! Returns the number of increments between the positions addressed
    //! by two iterators.
    tSize getDistance(const CMatrixIterator& End)
    {
        tSize Count = End.m_Row * End.m_pContainer->getNumOfCols() + End.m_Col;
        Count -= m_Row * base::m_pContainer->getNumOfCols() + m_Col;
        return (Count > 0) ? Count : 0;
    }

    //! Returns current iterator position.
    tSize getRow() const { return m_Row; }
    tSize getCol() const { return m_Col; }

    //! Sets the iterator position.
    void setPosition(tMatrix& Matrix, tSize Row, tSize Col)
    {
        base::m_pContainer = &Matrix;
        base::m_pItem = Matrix.getPtr(Row, Col);
        m_Row = Row;
        m_Col = Col;
    }

protected:
    //! Iterator position.
    tSize m_Row, m_Col;

protected:
    //! Moves iterator to the next element.
    void advance()
    {
        base::m_pItem += base::m_pContainer->getColOffset();
        if( ++m_Col >= base::m_pContainer->getNumOfCols() )
        {
            m_Col = 0;
            ++m_Row;
            base::m_pItem = base::m_pContainer->getPtr(m_Row, m_Col);
        }
    }
};


} // namespace math
} // namespace mds

#endif // MDS_MATRIXITERATOR_H

