//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsMatrix.hxx                       \n
 * Section: libMath                             \n
 * Date:    2003/12/03                          \n
 *
 * $Id: mdsMatrix.hxx 95 2006-09-24 20:11:00Z spanel $
 *
 * Description:
 * - Template providing basic matrix operations.
 */


//==============================================================================
/*
 * Methods templates.
 */

// Constructor
template <typename T>
CMatrix<T>::CMatrix(tSize NumOfRows, tSize NumOfCols)
    : m_NumOfRows(NumOfRows)
    , m_NumOfCols(NumOfCols)
{
    // Allocate the data
    m_RowOffset = m_NumOfCols;
    m_DataStorage.init(m_RowOffset * m_NumOfRows);
    m_pData = m_DataStorage.getPtr();
}


template <typename T>
CMatrix<T>::CMatrix(tSize NumOfRows, tSize NumOfCols, const T& Value)
    : m_NumOfRows(NumOfRows)
    , m_NumOfCols(NumOfCols)
{
    // Allocate the data
    m_RowOffset = m_NumOfCols;
    m_DataStorage.init(m_RowOffset * m_NumOfRows);
    m_pData = m_DataStorage.getPtr();

    // Fill the matrix
    fill(Value);
}


// Constructor
template <typename T>
CMatrix<T>::CMatrix(const CMatrix<T>& Matrix,
                    tSize Row,
                    tSize Col,
                    tSize NumOfRows,
                    tSize NumOfCols
                    )
{
    MDS_ASSERT(Row >= 0 && NumOfRows >= 0 && Row < Matrix.m_NumOfRows
               && Col >= 0 && NumOfCols >= 0 && Col < Matrix.m_NumOfCols);

    // Size of the matrix
    m_NumOfRows = MIN(Row + NumOfRows, Matrix.m_NumOfRows) - Row;
    m_NumOfCols = MIN(Col + NumOfCols, Matrix.m_NumOfCols) - Col;

    // Allocate the data
    m_RowOffset = m_NumOfCols;
    m_DataStorage.init(m_RowOffset * m_NumOfRows);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    T *p = m_pData;
    T *pMax = m_pData + m_NumOfRows * m_RowOffset;
    const T *pSrc = Matrix.getPtr(Row, Col);
    while( p < pMax )
    {
        memCopy(p, pSrc, m_NumOfCols);
        p += m_RowOffset;
        pSrc += Matrix.m_RowOffset;
    }
}


// Constructor
template <typename T>
CMatrix<T>::CMatrix(const CMatrix<T>& Matrix,
                    tSize Row,
                    tSize Col,
                    tSize NumOfRows,
                    tSize NumOfCols,
                    EMakeRef MakeRef
                    )
    : m_RowOffset(Matrix.m_RowOffset)
    , m_DataStorage(Matrix.m_DataStorage, REFERENCE)
    , m_pData(Matrix.m_pData)
{
    MDS_ASSERT(Row >= 0 && NumOfRows >= 0 && Row < Matrix.m_NumOfRows
               && Col >= 0 && NumOfCols >= 0 && Col < Matrix.m_NumOfCols);

    // Matrix size
    m_NumOfRows = MIN(Row + NumOfRows, Matrix.m_NumOfRows) - Row;
    m_NumOfCols = MIN(Col + NumOfCols, Matrix.m_NumOfCols) - Col;

    // Matrix data
    m_pData += Row * m_RowOffset + Col;
}


// Copy constructor
template <typename T>
CMatrix<T>::CMatrix(const CMatrix<T>& Matrix)
    : m_NumOfRows(Matrix.m_NumOfRows)
    , m_NumOfCols(Matrix.m_NumOfCols)
{
    // Allocate the data
    m_RowOffset = m_NumOfCols;
    m_DataStorage.init(m_RowOffset * m_NumOfRows);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    T *p = m_pData;
    T *pMax = p + m_NumOfRows * m_RowOffset;
    const T *pSrc = Matrix.m_pData;
    while( p < pMax )
    {
        memCopy(p, pSrc, m_NumOfCols);
        p += m_RowOffset;
        pSrc += Matrix.m_RowOffset;
    }
}


// Constructor
template <typename T>
template <tSize M, tSize N>
CMatrix<T>::CMatrix(const CStaticMatrix<T,M,N>& Matrix)
    : m_NumOfRows(M)
    , m_NumOfCols(N)
{
    // Allocate the data
    m_RowOffset = m_NumOfCols;
    m_DataStorage.init(m_RowOffset * m_NumOfRows);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    T *p = m_pData;
    T *pMax = p + m_NumOfRows * m_RowOffset;
    const T *pSrc = Matrix.getPtr();
    while( p < pMax )
    {
        memCopy(p, pSrc, m_NumOfCols);
        p += m_RowOffset;
        pSrc += N;
    }
}


// Copy constructor
template <typename T>
CMatrix<T>::CMatrix(const CMatrix<T>& Matrix, EMakeRef MakeRef)
    : m_NumOfRows(Matrix.m_NumOfRows)
    , m_NumOfCols(Matrix.m_NumOfCols)
    , m_RowOffset(Matrix.m_RowOffset)
    , m_DataStorage(Matrix.m_DataStorage, REFERENCE)
    , m_pData(Matrix.m_pData)
{
}


// Destructor
template <typename T>
CMatrix<T>::~CMatrix()
{
}


// Re-allocates memory
template <typename T>
bool CMatrix<T>::create(tSize NumOfRows, tSize NumOfCols)
{
    // Reallocate the data
    m_NumOfRows = NumOfRows;
    m_NumOfCols = NumOfCols;
    m_RowOffset = m_NumOfCols;
    m_DataStorage.resize(m_RowOffset * m_NumOfRows);
    m_pData = m_DataStorage.getPtr();

    // O.K.
    return true;
}


// Creates a new matrix
template <typename T>
bool CMatrix<T>::create(tSize NumOfRows, tSize NumOfCols, const T& Value)
{
    // Reallocate the data
    m_NumOfRows = NumOfRows;
    m_NumOfCols = NumOfCols;
    m_RowOffset = m_NumOfCols;
    m_DataStorage.resize(m_RowOffset * m_NumOfRows);
    m_pData = m_DataStorage.getPtr();

    // Fill the matrix
    fill(Value);

    // O.K.
    return true;
}


// Creates a new matrix
template <typename T>
bool CMatrix<T>::create(const CMatrix<T>& Matrix,
                        tSize Row,
                        tSize Col,
                        tSize NumOfRows,
                        tSize NumOfCols
                        )
{
    // Check the position
    MDS_CHECK(Row >= 0 && NumOfRows >= 0 && Row < Matrix.m_NumOfRows
              && Col >= 0 && NumOfCols >= 0 && Col < Matrix.m_NumOfCols, return false);

    // Matrix size
    m_NumOfRows = MIN(Row + NumOfRows, Matrix.m_NumOfRows) - Row;
    m_NumOfCols = MIN(Col + NumOfCols, Matrix.m_NumOfCols) - Col;

    // Reallocate the data
    m_RowOffset = m_NumOfCols;
    m_DataStorage.resize(m_RowOffset * m_NumOfRows);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    T *p = m_pData;
    T *pMax = m_pData + m_NumOfRows * m_RowOffset;
    const T *pSrc = Matrix.getPtr(Row, Col);
    while( p < pMax )
    {
        memCopy(p, pSrc, m_NumOfCols);
        p += m_RowOffset;
        pSrc += Matrix.m_RowOffset;
    }

    // O.K.
    return true;
}


// Creates a new matrix
template <typename T>
bool CMatrix<T>::create(const CMatrix<T>& Matrix,
                        tSize Row,
                        tSize Col,
                        tSize NumOfRows,
                        tSize NumOfCols,
                        EMakeRef MakeRef
                        )
{
    // Check the position
    MDS_CHECK(Row >= 0 && NumOfRows >= 0 && Row < Matrix.m_NumOfRows
              && Col >= 0 && NumOfCols >= 0 && Col < Matrix.m_NumOfCols, return false);

    // Matrix size
    m_NumOfRows = MIN(Row + NumOfRows, Matrix.m_NumOfRows) - Row;
    m_NumOfCols = MIN(Col + NumOfCols, Matrix.m_NumOfCols) - Col;

    // Create a reference to the image data
    m_RowOffset = Matrix.m_RowOffset;
    m_DataStorage.create(Matrix.m_DataStorage, REFERENCE);
    m_pData = Matrix.m_pData + Row * m_RowOffset + Col;

    // O.K.
    return true;
}


// Creates a new matrix
template <typename T>
bool CMatrix<T>::create(const CMatrix<T>& Matrix)
{
    // Reallocate the data
    m_NumOfRows = Matrix.m_NumOfRows;
    m_NumOfCols = Matrix.m_NumOfCols;
    m_RowOffset = m_NumOfCols;
    m_DataStorage.resize(m_RowOffset * m_NumOfRows);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    T *p = m_pData;
    T *pMax = p + m_NumOfRows * m_RowOffset;
    const T *pSrc = Matrix.m_pData;
    while( p < pMax )
    {
        memCopy(p, pSrc, m_NumOfCols);
        p += m_RowOffset;
        pSrc += Matrix.m_RowOffset;
    }

    // O.K.
    return true;
}


// Creates a new matrix
template <typename T>
template <tSize M, tSize N>
bool CMatrix<T>::create(const CStaticMatrix<T,M,N>& Matrix)
{
    // Reallocate the data
    m_NumOfRows = M;
    m_NumOfCols = N;
    m_RowOffset = m_NumOfCols;
    m_DataStorage.resize(m_RowOffset * m_NumOfRows);
    m_pData = m_DataStorage.getPtr();

    // Copy the data
    T *p = m_pData;
    T *pMax = p + m_NumOfRows * m_RowOffset;
    const T *pSrc = Matrix.getPtr();
    while( p < pMax )
    {
        memCopy(p, pSrc, m_NumOfCols);
        p += m_RowOffset;
        pSrc += N;
    }

    // O.K.
    return true;
}


// Creates a new matrix
template <typename T>
bool CMatrix<T>::create(const CMatrix<T>& Matrix, EMakeRef MakeRef)
{
    // Make a reference to the data
    m_NumOfRows = Matrix.m_NumOfRows;
    m_NumOfCols = Matrix.m_NumOfCols;
    m_RowOffset = m_NumOfCols;
    m_DataStorage.create(Matrix, REFERENCE);
    m_pData = Matrix.m_pData;

    // O.K.
    return true;
}


// Assignment operator
template <typename T>
CMatrix<T>& CMatrix<T>::operator =(const CMatrix<T>& Matrix)
{
    if( &Matrix == this )
    {
        return *this;
    }

    tSize NumOfRows = MIN(m_NumOfRows, Matrix.m_NumOfRows);
    tSize NumOfCols = MIN(m_NumOfCols, Matrix.m_NumOfCols);

    for( tSize j = 0; j < NumOfRows; ++j )
    {
        memCopy(m_pData + j * m_RowOffset,
                Matrix.m_pData + j * Matrix.m_RowOffset,
                NumOfCols
               );
    }

    return *this;
}


// Assignment operator
template <typename T>
template <tSize M, tSize N>
CMatrix<T>& CMatrix<T>::operator =(const CStaticMatrix<T,M,N>& Matrix)
{
    tSize NumOfRows = MIN(m_NumOfRows, M);
    tSize NumOfCols = MIN(m_NumOfCols, N);

    const T *pSrc = Matrix.getPtr();
    for( tSize j = 0; j < NumOfRows; ++j )
    {
        memCopy(m_pData + j * m_RowOffset,
                pSrc + j * N,
                NumOfCols
               );
    }

    return *this;
}


// Combined addition and assignment operator
template <typename T>
inline CMatrix<T>& CMatrix<T>::operator +=(const CMatrix<T>& Matrix)
{
    tSize NumOfRows = MIN(m_NumOfRows, Matrix.m_NumOfRows);
    tSize NumOfCols = MIN(m_NumOfCols, Matrix.m_NumOfCols);

    for( tSize j = 0; j < NumOfRows; ++j )
    {
        memVectAdd(m_pData + j * m_RowOffset,
                   Matrix.m_pData + j * Matrix.m_RowOffset,
                   NumOfCols
                   );
    }

    return *this;
}


// Combined subtraction and assignment operator
template <typename T>
inline CMatrix<T>& CMatrix<T>::operator -=(const CMatrix<T>& Matrix)
{
    tSize NumOfRows = MIN(m_NumOfRows, Matrix.m_NumOfRows);
    tSize NumOfCols = MIN(m_NumOfCols, Matrix.m_NumOfCols);

    for( tSize j = 0; j < NumOfRows; ++j )
    {
        memVectSub(m_pData + j * m_RowOffset,
                   Matrix.m_pData + j * Matrix.m_RowOffset,
                   NumOfCols
                   );
    }

    return *this;
}


// Combined scalar multiplication and assignment operator
template <typename T>
inline CMatrix<T>& CMatrix<T>::operator *=(const CMatrix<T>& Matrix)
{
    tSize NumOfRows = MIN(m_NumOfRows, Matrix.m_NumOfRows);
    tSize NumOfCols = MIN(m_NumOfCols, Matrix.m_NumOfCols);

    for( tSize j = 0; j < NumOfRows; ++j )
    {
        memVectMult(m_pData + j * m_RowOffset,
                    Matrix.m_pData + j * Matrix.m_RowOffset,
                    NumOfCols
                    );
    }

    return *this;
}


// Combined division and assignment operator
template <typename T>
inline CMatrix<T>& CMatrix<T>::operator /=(const CMatrix<T>& Matrix)
{
    tSize NumOfRows = MIN(m_NumOfRows, Matrix.m_NumOfRows);
    tSize NumOfCols = MIN(m_NumOfCols, Matrix.m_NumOfCols);

    for( tSize j = 0; j < NumOfRows; ++j )
    {
        memVectDiv(m_pData + j * m_RowOffset,
                   Matrix.m_pData + j * Matrix.m_RowOffset,
                   NumOfCols
                   );
    }

    return *this;
}


// Combined scalar multiplication and assignment operator
template <typename T>
template <typename U>
inline CMatrix<T>& CMatrix<T>::operator *=(const U& c)
{
    for( tSize j = 0; j < m_NumOfRows; ++j )
    {
        memMult(m_pData + j * m_RowOffset, c, m_NumOfCols);
    }

    return *this;
}


// Combined scalar division and assignment operator
template <typename T>
template <typename U>
inline CMatrix<T>& CMatrix<T>::operator /=(const U& c)
{
    for( tSize j = 0; j < m_NumOfRows; ++j )
    {
        memDiv(m_pData + j * m_RowOffset, c, m_NumOfCols);
    }

    return *this;
}


// Combined scalar addition and assignment operator
template <typename T>
template <typename U>
inline CMatrix<T>& CMatrix<T>::operator +=(const U& c)
{
    for( tSize j = 0; j < m_NumOfRows; ++j )
    {
        memAdd(m_pData + j * m_RowOffset, c, m_NumOfCols);
    }

    return *this;
}


// Combined scalar subtraction and assignment operator
template <typename T>
template <typename U>
inline CMatrix<T>& CMatrix<T>::operator -=(const U& c)
{
    for( tSize j = 0; j < m_NumOfRows; ++j )
    {
        memSub(m_pData + j * m_RowOffset, c, m_NumOfCols);
    }

    return *this;
}


// Set all elements of a matrix to zero
template <typename T>
inline void CMatrix<T>::zeros()
{
    for( tSize j = 0; j < m_NumOfRows; ++j )
    {
        memSet(m_pData + j * m_RowOffset, T(0), m_NumOfCols);
    }
}


// Set all elements of a matrix to one
template <typename T>
inline void CMatrix<T>::ones()
{
    for( tSize j = 0; j < m_NumOfRows; ++j )
    {
        memSet(m_pData + j * m_RowOffset, T(1), m_NumOfCols);
    }
}


// Set a matrix to unit
template <typename T>
inline void CMatrix<T>::unit()
{
    for( tSize i = 0; i < m_NumOfRows; ++i )
    {
        for( tSize j = 0; j < m_NumOfCols; ++j )
        {
            *(m_pData + i * m_RowOffset + j) = (i == j) ? T(1) : T(0);
        }
    }
}


template <typename T>
inline void CMatrix<T>::fill(const T& Value)
{
    for( tSize j = 0; j < m_NumOfRows; ++j )
    {
        memSet(m_pData + j * m_RowOffset, Value, m_NumOfCols);
    }
}


template <typename T>
inline void CMatrix<T>::setRow(tSize i, const T *pSrc)
{
    memCopy(m_pData + i * m_RowOffset, pSrc, m_NumOfCols);
}


template <typename T>
inline void CMatrix<T>::copyRow(tSize i, T *pDst) const
{
    memCopy(pDst, m_pData + i * m_RowOffset, m_NumOfCols);
}


template <typename T>
inline void CMatrix<T>::swapRows(tSize i, tSize j)
{
    if( i == j )
    {
        return;
    }

    memSwap(m_pData + i * m_RowOffset, m_pData + j * m_RowOffset, m_NumOfCols);
}


template <typename T>
inline void CMatrix<T>::abs()
{
    for( tSize j = 0; j < m_NumOfRows; ++j )
    {
        T *p = m_pData + j * m_RowOffset;
        T *pMax = p + m_NumOfCols;
        while( p < pMax )
        {
            if( !(*p > T(0)) )
            {
                *p = -(*p);
            }
            ++p;
        }
    }
}


// Matrix multiplication
template <typename T>
inline void CMatrix<T>::mult(const CMatrix<T>& m1, const CMatrix<T>& m2)
{
    if( m1.m_NumOfCols != m2.m_NumOfRows || m_NumOfRows != m1.m_NumOfRows || m_NumOfCols != m2.m_NumOfCols )
    {
        MDS_THROW_WARNING("Method CMatrix::mult() failed");
    }

    for( tSize i = 0; i < m_NumOfRows; ++i )
    {
        for( tSize j = 0; j < m_NumOfCols; ++j )
        {
            T *p = getPtr(i, j);

            const T *p1 = m1.getRowPtr(i);
            const T *p1Max = p1 + m1.m_NumOfCols;
            const T *p2 = m2.getColPtr(j);

            *p = T(0);
            while( p1 < p1Max )
            {
                *p += *(p1++) * *p2;
                p2 += m2.m_RowOffset;
            }
        }
    }
}


// Compute a matrix transpose
template <typename T>
inline void CMatrix<T>::transpose(const CMatrix<T>& Matrix)
{
    if( m_NumOfCols != Matrix.m_NumOfRows || m_NumOfRows != Matrix.m_NumOfCols )
    {
        MDS_THROW_WARNING("Method CMatrix::transpose() failed");
    }

    for( tSize i = 0; i < m_NumOfRows; ++i )
    {
        for( tSize j = 0; j < m_NumOfCols; ++j )
        {
            *(m_pData + i * m_RowOffset + j) = *(Matrix.m_pData + j * Matrix.m_RowOffset + i);
        }
    }
}

