//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)   \n
 * Copyright (c) 2003-2005 by Michal Spanel    \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz \n
 * File:    mdsMatrix.h                        \n
 * Section: libMath                            \n
 * Date:    2003/12/03                         \n
 *
 * $Id: mdsMatrix.h 345 2007-06-11 13:23:09Z spanel $
 *
 * Description:
 * - Template providing basic matrix operations.
 */

#ifndef MDS_MATRIX_H
#define MDS_MATRIX_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsWarning.h>
#include <MDSTk/Base/mdsError.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsMemory.h>
#include <MDSTk/Base/mdsRefData.h>
#include <MDSTk/Base/mdsSharedPtr.h>

#include "mdsBase.h"
#include "mdsMatrixBase.h"
#include "mdsMatrixIterator.h"
#include "mdsStaticMatrix.h"

// STL
#include <cmath>


namespace mds
{
namespace math
{

//==============================================================================
/*!
 * Template for a two dimensional dense matrix stored in row-.
 * - T is a matrix element type.
 */
template <typename T>
class CMatrix : public mds::CObject, public CMatrixBase<CMatrix<T> >
{
public:
    //! Templates that require members of the CMatrix class can use
    //! this enum to check the existence.
    enum { CLASS_MATRIX };

    //! Standard method getClassName().
    MDS_CLASS_NAME(CMatrix);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CMatrix);

    //! Matrix element type.
    typedef T tElement;

    //! Matrix data.
    typedef mds::base::CRefData<T> tDataStorage;

    //! Iterator type.
    //! - Declares types tIterator and tConstIterator.
    MDS_COMPLEX_ITERATOR(CMatrix, tElement, CMatrixIterator);

public:
    //! Constructor.
    //! - Parameters involve the matrix dimensions.
    CMatrix(tSize NumOfRows = 0, tSize NumOfCols = 0);

    //! Constructor.
    //! - Matrix dimensions and initial value.
    CMatrix(tSize NumOfRows, tSize NumOfCols, const T& Value);

    //! Constructor.
    //! - Creates a submatrix of a given matrix.
    CMatrix(const CMatrix& Matrix,
            tSize Row,
            tSize Col,
            tSize NumOfRows,
            tSize NumOfCols
            );

    //! Constructor.
    //! - Creates a submatrix of a given matrix.
    //! - Makes only reference to the data.
    CMatrix(const CMatrix& Matrix,
            tSize Row,
            tSize Col,
            tSize NumOfRows,
            tSize NumOfCols,
            EMakeRef
            );

    //! Copy constructor.
    //! - Makes a new copy of the data.
    CMatrix(const CMatrix& Matrix);

    //! Copy constructor.
    //! - Makes only reference to the data of a given matrix.
    CMatrix(const CMatrix& Matrix, EMakeRef);

    //! Constructor.
    //! - Makes a new copy of the data.
    template <tSize M, tSize N>
    CMatrix(const CStaticMatrix<T,M,N>& Matrix);

    //! Virtual destructor.
    virtual ~CMatrix();

    //! Re-allocates memory for the matrix if needed.
    bool create(tSize NumOfRows, tSize NumOfCols);

    //! Creates a new matrix.
    bool create(tSize NumOfRows, tSize NumOfCols, const T& Value);

    //! Creates a new matrix.
    bool create(const CMatrix& Matrix,
                tSize Row,
                tSize Col,
                tSize NumOfRows,
                tSize NumOfCols
                );

    //! Creates a new matrix.
    bool create(const CMatrix& Matrix,
                tSize Row,
                tSize Col,
                tSize NumOfRows,
                tSize NumOfCols,
                EMakeRef
                );

    //! Creates a new matrix.
    //! - Makes a new copy of the data.
    bool create(const CMatrix& Matrix);

    //! Creates a new matrix.
    //! - Makes reference to the data of a given matrix.
    bool create(const CMatrix& Matrix, EMakeRef);

    //! Creates a new matrix.
    //! - Makes a new copy of the data.
    template <tSize M, tSize N>
    bool create(const CStaticMatrix<T,M,N>& Matrix);

    //! Assignment operator.
    CMatrix& operator =(const CMatrix& Matrix);

    //! Assignment operator.
    template <tSize M, tSize N>
    CMatrix& operator =(const CStaticMatrix<T,M,N>& Matrix);


    //! Returns the matrix dimensions.
    tSize getNumOfRows() const { return m_NumOfRows; }
    tSize getNumOfCols() const { return m_NumOfCols; }

    //! Returns row offset. In other words, offset between two neighbouring
    //! elements in a matrix column.
    tSize getRowOffset() const { return m_RowOffset; }

    //! Returns column offset. In other words, offset between two neighbouring
    //! elements in a matrix row.
    tSize getColOffset() const { return 1; }

    //! Returns reference to the matrix data storage.
    tDataStorage& getDataStorage() { return m_DataStorage; }
    const tDataStorage& getDataStorage() const { return m_DataStorage; }

    //! Returns the subscripted element [Row][Col].
    T& operator ()(tSize Row, tSize Col)
    {
        return *(m_pData + Row * m_RowOffset + Col);
    }
    const T& operator ()(tSize Row, tSize Col) const
    {
        return *(m_pData + Row * m_RowOffset + Col);
    }

    //! Gets the element [Row][Col].
    T& get(tSize Row, tSize Col)
    {
        return *(m_pData + Row * m_RowOffset + Col);
    }
    const T& get(tSize Row, tSize Col) const
    {
        return *(m_pData + Row * m_RowOffset + Col);
    }

    //! Returns pointer to the given element.
    T *getPtr(tSize Row, tSize Col)
    {
        return (m_pData + Row * m_RowOffset + Col);
    }
    const T *getPtr(tSize Row, tSize Col) const
    {
        return (m_pData + Row * m_RowOffset + Col);
    }

    //! Returns pointer to the first element of matrix row.
    T *getRowPtr(tSize Row)
    {
        return (m_pData + Row * m_RowOffset);
    }
    const T *getRowPtr(tSize Row) const
    {
        return (m_pData + Row * m_RowOffset);
    }

    //! Returns pointer to the first element of matrix column.
    T *getColPtr(tSize Col)
    {
        return (m_pData + Col);
    }
    const T *getColPtr(tSize Col) const
    {
        return (m_pData + Col);
    }

    //! Returns pointer to the matrix data.
    T *getPtr() { return m_pData; }
    const T *getPtr() const { return m_pData; }

    //! Sets the element of matrix at the position [row][col].
    void set(tSize Row, tSize Col, const T& Value)
    {
        *(m_pData + Row * m_RowOffset + Col) = Value;
    }


    //! Returns iterator that points at the first matrix element.
    tIterator getBegin()
    {
        return tIterator(*this, 0, 0);
    }
    tConstIterator getBegin() const
    {
        return tConstIterator(*this, 0, 0);
    }

    //! Returns iterator that points beyond the last matrix element.
    tIterator getEnd()
    {
        return tIterator(*this, m_NumOfRows, m_NumOfCols);
    }
    tConstIterator getEnd() const
    {
        return tConstIterator(*this, m_NumOfRows, m_NumOfCols);
    }

    //! Returns iterator that points at a specified position.
    tIterator getIterator(tSize Row, tSize Col)
    {
        return tIterator(*this, Row, Col);
    }
    tConstIterator getIterator(tSize Row, tSize Col) const
    {
        return tConstIterator(*this, Row, Col);
    }


    //! Miscellaneous combined assignment operators.
    inline CMatrix& operator +=(const CMatrix& Matrix);
    inline CMatrix& operator -=(const CMatrix& Matrix);
    inline CMatrix& operator *=(const CMatrix& Matrix);
    inline CMatrix& operator /=(const CMatrix& Matrix);

    //! Combined assignment operator.
    template <typename U>
    inline CMatrix& operator +=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CMatrix& operator -=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CMatrix& operator *=(const U& c);

    //! Combined assignment operator.
    template <typename U>
    inline CMatrix& operator /=(const U& c);


    //! The matrix initialization routine.
    //! - Zeros the matrix.
    inline void zeros();

    //! The matrix initialization routine.
    //! - Sets all elements to one.
    inline void ones();

    //! The matrix initialization routine.
    //! - Makes the matrix unit.
    inline void unit();

    //! Fills the matrix using a given value.
    inline void fill(const T& Value);

    //! Makes absolute value of all elements.
    inline void abs();

    //! Fills the i-th matrix row using a given input buffer.
    inline void setRow(tSize i, const T *pSrc);

    //! Copies the i-th matrix row to the output buffer.
    inline void copyRow(tSize i, T *pDst) const;

    //! Swaps the i-th and j-th matrix rows.
    inline void swapRows(tSize i, tSize j);

    //! Type of the matrix.
    //! - Returns true if the matrix is square.
    inline bool isSquare() const { return (m_NumOfRows == m_NumOfCols); }

    //! Transpose of the matrix.
    inline void transpose(const CMatrix& Matrix);

    //! Matrix multiplication.
    inline void mult(const CMatrix& m1, const CMatrix& m2);

protected:
    //! The matrix dimensions.
    tSize m_NumOfRows, m_NumOfCols;

    //! Offset used by subscripting functions.
    tSize m_RowOffset;

    //! Matrix data.
    tDataStorage m_DataStorage;

    //! Pointer to the data.
    T *m_pData;
};


//==============================================================================
/*
 * Method templates.
 */

// Include the file containing method templates.
#include "mdsMatrix.hxx"


//==============================================================================
/*
 * Basic template instances and type definitions.
 */

//! Matrix of int numbers.
typedef CMatrix<int>    CIMatrix;

//! Matrix of float numbers.
typedef CMatrix<float>  CFMatrix;

//! Matrix of double numbers.
typedef CMatrix<double> CDMatrix;


//==============================================================================
/*
 * Basic template instances and type definitions.
 * - Using smart pointer
 */

//! Matrix of int numbers.
typedef CIMatrix::tSmartPtr     CIMatrixPtr;

//! Matrix of float numbers.
typedef CFMatrix::tSmartPtr     CFMatrixPtr;

//! Matrix of double numbers.
typedef CDMatrix::tSmartPtr     CDMatrixPtr;


} // namespace math
} // namespace mds

#endif // MDS_MATRIX_H

