//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsLapack.h                         \n
 * Section: libMath                             \n
 * Date:    2006/02/16                          \n
 *
 * $Id: mdsLapack.h 119 2006-10-30 18:23:00Z spanel $
 *
 * Description:
 * - BLAS and LAPACK functions.
 */

#ifndef MDS_LAPACK_H
#define MDS_LAPACK_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsWarning.h>
#include <MDSTk/Base/mdsMemory.h>
#include <MDSTk/Base/mdsLock.h>
#include <MDSTk/Base/mdsSingleton.h>

// Include BLAS and LAPACK
extern "C"
{
// Microsoft Visual C++
#ifdef _MSC_VER
#    include "../lib/vc/atlas/clapack.h"
#    include "../lib/vc/atlas/clapack2.h"
#endif // _MSC_VER

// Windows version - MinGW32
#if defined(_WIN32) && !defined(_MSC_VER)
#    include "../lib/win/atlas/clapack.h"
#    include "../lib/win/atlas/clapack2.h"
#endif // _WIN32

// Linux version
#ifdef _LINUX
#    include "../lib/linux/atlas/clapack.h"
#    include "../lib/linux/atlas/clapack2.h"
#endif // _LINUX
}


namespace mds
{
namespace math
{

//==============================================================================
/*!
 * Class used for locking BLAS and LAPACK interface.
 */
class CLapack : public mds::base::CLibrarySingleton, public mds::base::CLibraryLockableClass<CLapack>
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CLapack);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CLapack);

    //! Lock type.
    typedef mds::base::CLibraryLockableClass<CLapack>::CLock tLock;

public:
    //! Virtual destructor.
    virtual ~CLapack() {}


    //! ATLAS LAPACK function sgetrf().
    //! - AP = LU; A <- LU, ipiv <- P
    inline void getrf(enum CBLAS_ORDER Order, int M, int N, float *A, int lda, int *ipiv)
    {
        tLock Lock;
        (void)Lock;

        int iResult = clapack_sgetrf(Order, M, N, A, lda, ipiv);
        if( iResult != 0 )
        {
            MDS_THROW_WARNING("Method CLapack::getrf() failed");
        }
    }

    //! ATLAS LAPACK function dgetrf().
    //! - AP = LU; A <- LU, ipiv <- P
    inline void getrf(enum CBLAS_ORDER Order, int M, int N, double *A, int lda, int *ipiv)
    {
        tLock Lock;
        (void)Lock;

        int iResult = clapack_dgetrf(Order, M, N, A, lda, ipiv);
        if( iResult != 0 )
        {
            MDS_THROW_WARNING("Method CLapack::getrf() failed");
        }
    }


    //! ATLAS LAPACK function sgetri().
    //! - A = A^-1; assuming on entry A = LU, ipiv = P
    inline void getri(enum CBLAS_ORDER Order, int N, float *A, int lda, int *ipiv)
    {
        tLock Lock;
        (void)Lock;

        int iResult = clapack_sgetri(Order, N, A, lda, ipiv);
        if( iResult != 0 )
        {
            MDS_THROW_WARNING("Method CLapack::getri() failed");
        }
    }

    //! ATLAS LAPACK function dgetri().
    //! - A = A^-1; assuming on entry A = LU, ipiv = P
    inline void getri(enum CBLAS_ORDER Order, int N, double *A, int lda, int *ipiv)
    {
        tLock Lock;
        (void)Lock;

        int iResult = clapack_dgetri(Order, N, A, lda, ipiv);
        if( iResult != 0 )
        {
            MDS_THROW_WARNING("Method CLapack::getri() failed");
        }
    }

    //! LAPACK function ssyev().
    //! - Computes all eigenvalues (and eigenvectors) of a symmetric matrix.
    inline void syev(char jobz, char uplo, integer n, float *a, integer lda, float *w, float *work, integer lwork)
    {
        tLock Lock;
        (void)Lock;

        integer Info;
        ssyev_(&jobz, &uplo, &n, a, &lda, w, work, &lwork, &Info);
        if( Info != 0 )
        {
            MDS_THROW_WARNING("Method CLapack::syev() failed");
        }
    }

    //! LAPACK function dsyev().
    //! - Computes all eigenvalues (and eigenvectors) of a symmetric matrix.
    inline void syev(char jobz, char uplo, integer n, double *a, integer lda, double *w, double *work, integer lwork)
    {
        tLock Lock;
        (void)Lock;

        integer Info;
        dsyev_(&jobz, &uplo, &n, a, &lda, w, work, &lwork, &Info);
        if( Info != 0 )
        {
            MDS_THROW_WARNING("Method CLapack::syev() failed");
        }
    }

private:
    //! Private constructor.
    CLapack() {}

    //! Allow main log instantiation using the singleton holder.
    MDS_PRIVATE_SINGLETON(CLapack);
};


//==============================================================================
/*
 * Certain logging macros.
 */

//! Returns reference to the global application log.
#define MDS_LAPACK          MDS_SINGLETON(mds::math::CLapack)

//! Locks lapack interface.
#define MDS_LAPACK_LOCK     MDS_LAPACK.lock()

//! Leaves logging mutual section.
#define MDS_LAPACK_UNLOCK   MDS_LAPACK.unlock()


} // namespace math
} // namespace mds

#endif // MDS_BASE_H

