//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsComplex.h                        \n
 * Section: libMath                             \n
 * Date:    2006/04/20                          \n
 *
 * $Id: mdsComplex.h 116 2006-10-20 08:51:07Z spanel $
 * 
 * Description:
 * - Complex number definition.
 */

#ifndef MDS_COMPLEX_H
#define MDS_COMPLEX_H

#include <MDSTk/Base/mdsSetup.h>

#include "mdsBase.h"

// STL
#include <cmath>
//#include <complex>
#include <iostream>


namespace mds
{
namespace math
{

//==============================================================================
/*!
 * Class representing a complex number.
 * - T is type of the complex number real and imaginary part.
 * - TODO: Get round the VC7 implementation of the STL std::complex<> class
 *   which is not functional and fully compatible with some parts of the
 *   code in this class.
 */
template <typename T>
class CComplex
{
public:
    //! Type of the complex number component.
    typedef T tComponent;

public:
    //! Real and imaginary part of the complex number.
    T re, im;

public:
    //! Default constructor.
    CComplex() : re(0), im(0) {}

    //! Constructor.
    CComplex(tComponent r, tComponent i) : re(r), im(i) {}

    //! Constructor.
    //! - Creates complex number from a given real number.
    CComplex(tComponent r) : re(r), im(0) {}

    //! Constructor.
/*  template <typename U>
    CComplex(const std::complex<U>& z) : re(z.real()) , im(z.imag()) {}*/

    //! Destructor.
    ~CComplex() {}

    //! Copy constructor.
    template <typename U>
    CComplex(const CComplex<U>& z) : re(z.re), im(z.im) {}

    //! Assignment operator.
    template <typename U>
    CComplex& operator =(const CComplex<U>& z)
    {
        re = z.re;
        im = z.im;
        return *this;
    }

    //! Assignment operator.
/*  template <typename U>
    CComplex& operator =(const std::complex<U>& z)
    {
        re = z.real();
        im = z.imag();
        return *this;
    }*/

    //! Assignment operator.
    CComplex& operator =(tComponent r)
    {
        re = r;
        im = 0;
        return *this;
    }

    //! Conversion to the STL complex number.
/*  operator std::complex<T>()
    {
        return std::complex<T>(re, im);
    }*/


    //! Returns real part of the complex number.
    tComponent& getReal() { return re; }
    const tComponent& getReal() const { return re; }

    //! Returns imaginary part of the complex number.
    tComponent& getImag() { return im; }
    const tComponent& getImag() const { return im; }

    //! Returns both the real and the imaginary part of the complex number.
    void get(tComponent& r, tComponent& i) const
    {
        r = re;
        i = im;
    }

    //! Sets real part of the complex number.
    void setReal(tComponent r) { re = r; }

    //! Sets imaginary part of the complex number.
    void setImag(tComponent i) { im = i; }

    //! Sets real and imaginary part of the complex number.
    void set(tComponent r, tComponent i)
    {
        re = r;
        im = i;
    }


    //! Combined assignment operators.
    inline CComplex& operator +=(const CComplex& z);
    inline CComplex& operator -=(const CComplex& z);
    inline CComplex& operator *=(const CComplex& z);
    inline CComplex& operator /=(const CComplex& z);
    
    //! Others combined assignment operators.
    inline CComplex& operator +=(const T& c);
    inline CComplex& operator -=(const T& c);
    inline CComplex& operator *=(const T& c);
    inline CComplex& operator /=(const T& c);


    //! Returns absolute value (magnitude) of the complex number.
    tComponent getAbs() const
    {
        tComponent x = re;
        tComponent y = im;
        const tComponent s = mds::math::getMax(mds::math::getAbs(x), mds::math::getAbs(y));
        if( s == 0 )
        {
            return s;
        }
        x /= s;
        y /= s;
        return s * sqrt(x * x + y * y);
    }

    //! Returns squared absolute value of the complex number.
    tComponent getNorm() const
    {
        return re * re + im * im;
    }

    //! Returns argument (angle) of the complex number.
    tComponent getArg() const
    {
        return (T)atan2(im, re);
    }

    //! Returns complex conjugate number.
    CComplex getConj() const
    {
        return CComplex(re, -im);
    }

    //! Normalizes the complex number so that the absolute value
    //! is equal to one.
    CComplex& normalize()
    {
        tComponent Temp = getAbs();
        if( Temp != 0 )
        {
            tComponent InvTemp = 1 / Temp;
            re *= InvTemp;
            im *= InvTemp;
        }
        return *this;
    }
};


//==============================================================================
/*
 * Combined assignment operators.
 */

template <typename T>
inline CComplex<T>& CComplex<T>::operator +=(const CComplex<T>& z)
{
    re += z.re;
    im += z.im;
    return *this;
}


template <typename T>
inline CComplex<T>& CComplex<T>::operator -=(const CComplex<T>& z)
{
    re -= z.re;
    im -= z.im;
    return *this;
}


template <typename T>
inline CComplex<T>& CComplex<T>::operator *=(const CComplex<T>& z)
{
    T Temp = re * z.re - im * z.im;
    im = re * z.im + im * z.re;
    re = Temp;
    return *this;
}


template <typename T>
inline CComplex<T>& CComplex<T>::operator /=(const CComplex<T>& z)
{
    T ar = getAbs(z.re);
    T ai = getAbs(z.im);
    T nr, ni;
    if (ar <= ai)
    {
        T t = z.re / z.im;
        T d = z.im * (t * t + 1L);
        nr = (re * t + im) / d;
        ni = (im * t - re) / d;
    }
    else
    {
        T t = z.im / z.re;
        T d = z.re * (t * t + 1L);
        nr = (re + im * t) / d;
        ni = (im - re * t) / d;
    }
    re = nr;
    im = ni;
    return *this;
}


template <typename T>
inline CComplex<T>& CComplex<T>::operator +=(const T& c)
{
    re += c;
    return *this;
}


template <typename T>
inline CComplex<T>& CComplex<T>::operator -=(const T& c)
{
    re -= c;
    return *this;
}


template <typename T>
inline CComplex<T>& CComplex<T>::operator *=(const T& c)
{
    re *= c;
    im *= c;
    return *this;
}


template <typename T>
inline CComplex<T>& CComplex<T>::operator /=(const T& c)
{
    re /= c;
    im /= c;
    return *this;
}


//==============================================================================
/*
 * Others operators.
 */

template <typename T>
inline CComplex<T> operator +(const CComplex<T>& x, const CComplex<T>& y)
{
    return CComplex<T>(x.re + y.re, x.im + y.im);
}


template <typename T>
inline CComplex<T> operator +(const CComplex<T>& x, T y)
{
    return CComplex<T>(x.re + y, x.im);
}


template <typename T>
inline CComplex<T> operator +(T x, const CComplex<T>& y)
{
    return CComplex<T>(x + y.re, y.im);
}


template <typename T>
inline CComplex<T> operator -(const CComplex<T>& x, const CComplex<T>& y)
{
    return CComplex<T>(x.re - y.re, x.im - y.im);
}


template <typename T>
inline CComplex<T> operator -(const CComplex<T>& x, T y)
{
    return CComplex<T>(x.re - y, x.im);
}


template <typename T>
inline CComplex<T> operator -(T x, const CComplex<T>& y)
{
    return CComplex<T>(x - y.re, - y.im);
}


template <typename T>
inline CComplex<T> operator *(const CComplex<T>& x, const CComplex<T>& y)
{
    return CComplex<T>(x.re * y.re - x.im * y.im, x.re * y.im + x.im * y.re);
}


template <typename T>
inline CComplex<T> operator *(const CComplex<T>& x, T y)
{
    return CComplex<T>(x.re * y, x.im * y);
}


template <typename T>
inline CComplex<T> operator *(T x, const CComplex<T>& y)
{
    return CComplex<T>(x * y.re, x * y.im);
}


template <typename T>
inline CComplex<T> operator /(const CComplex<T>& x, const CComplex<T>& y)
{
    T ar = getAbs(y.re);
    T ai = getAbs(y.im);
    T nr, ni;
    T t, d;
    if( ar <= ai )
    {
        t = y.re / y.im;
        d = y.im * (t*t + 1L);
        nr = (x.re * t + x.im) / d;
        ni = (x.im * t - x.re) / d;
    }
    else
    {
        t = y.im / y.re;
        d = y.re * (t*t + 1L);
        nr = (x.re + x.im * t) / d;
        ni = (x.im - x.re * t) / d;
    }
    return CComplex<T>(nr, ni);
}


template <typename T>
inline CComplex<T> operator /(T x, const CComplex<T>& y)
{
    T ar = getAbs(y.re);
    T ai = getAbs(y.im);
    T nr, ni;
    T t, d;
    if( ar <= ai )
    {
        t = y.re / y.im;
        d = y.im * (1 + t*t);
        nr = x * t / d;
        ni = -x / d;
    }
    else
    {
        t = y.im / y.re;
        d = y.re * (1 + t*t);
        nr = x / d;
        ni = -x * t / d;
    }
    return CComplex<T>(nr, ni);
}


template <typename T>
inline CComplex<T> operator /(const CComplex<T>& x, T y)
{
    return CComplex<T>(x.re / y, x.im / y);
}


template <typename T>
inline CComplex<T> operator +(const CComplex<T>& x)
{
    return x;
}


template <typename T>
inline CComplex<T> operator -(const CComplex<T>& x)
{
    return CComplex<T>(-x.re, -x.im);
}


template <typename T>
inline bool operator ==(const CComplex<T>& x, const CComplex<T>& y)
{
    return x.re == y.re && x.im == y.im;
}


template <typename T>
inline bool operator ==(const CComplex<T>& x, T y)
{
    return x.re == y && x.im == 0;
}


template <typename T>
inline bool operator ==(T x, const CComplex<T>& y)
{
    return x == y.re && y.im == 0;
}


template <typename T>
inline bool operator !=(const CComplex<T>& x, const CComplex<T>& y)
{
    return x.re != y.re || x.im != y.im;
}


template <typename T>
inline bool operator !=(const CComplex<T>& x, T y)
{
    return x.re != y || x.im != 0;
}


template <typename T>
inline bool operator != (T x, const CComplex<T>& y)
{
    return x != y.re || y.im != 0;
}


//==============================================================================
/*
 * Global functions.
 */

//! Returns real part of a complex number.
template <typename T>
inline T getReal(const CComplex<T>& z)
{
    return z.getReal();
}


//! Returns imaginary part of a complex number.
template <typename T>
inline T getImag(const CComplex<T>& z)
{
    return z.getImag();
}


//! Creates a temporary complex number from polar coordinates.
//! - Angle is equal to zero.
template <typename T>
inline CComplex<T> polar(const T& x)
{
    return CComplex<T>(x, 0);
}


//! Creates a temporary complex number from polar coordinates.
template <typename T>
inline CComplex<T> polar(const T& x, const T& y)
{
    return CComplex<T>(x * cos(y), x * sin(y));
}


//! Returns absolute value (magnitude) of the complex number.
template <typename T>
inline T getAbs(const CComplex<T>& z)
{
    return z.getAbs();
}


//! Returns squared absolute value of the complex number.
template <typename T>
inline T getNorm(const CComplex<T>& z)
{
    return z.getNorm();
}


//! Returns argument (phase) of the complex number.
template <typename T>
inline T getArg(const CComplex<T>& z)
{
    return z.getArg();
}


//! Returns complex conjugate number.
template <typename T>
inline CComplex<T> getConj(const CComplex<T>& z)
{
    return z.getConj();
}


//! Returns square root of the complex number.
/*template <typename T>
inline CComplex<T> getSqrt(const CComplex<T>& z)
{
    std::complex<T> s = std::sqrt(std::complex<T>(z.re, z.im));
    
    return CComplex<T>(s.real(), s.imag());
}*/


//! Returns exponential of the complex number.
template <typename T>
inline CComplex<T> getExp(const CComplex<T>& z)
{
    return polar(T(exp(z.getReal())), z.getImag());
}


//! Returns natural logarithm with base e of the complex number.
/*template <typename T>
inline CComplex<T> getLog(const CComplex<T>& z)
{
    std::complex<T> s = std::log(std::complex<T>(z.re, z.im));
    
    return CComplex<T>(s.real(), s.imag());
}
 
 
//! Returns logarithm with base 10 of the complex number.
template <typename T>
inline CComplex<T> getLog10(const CComplex<T>& z)
{
    std::complex<T> s = std::log10(std::complex<T>(z.re, z.im));
    
    return CComplex<T>(s.real(), s.imag());
}
 
 
//! Returns complex power.
template <typename T>
inline CComplex<T> getPow(const CComplex<T>& x, const CComplex<T>& y)
{
    std::complex<T> s = std::pow(std::complex<T>(x), std::complex<T>(y));
    
    return CComplex<T>(s.real(), s.imag());
}
 
 
//! Returns complex power.
template <typename T>
inline CComplex<T> getPow(const CComplex<T>& x, const T& y)
{
    std::complex<T> s = std::pow(std::complex<T>(x), y);
    
    return CComplex<T>(s.real(), s.imag());
}
 
 
//! Returns complex power.
template <typename T>
inline CComplex<T> getPow(const T& x, const CComplex<T>& y)
{
    std::complex<T> s = std::pow(x, std::complex<T>(y));
    
    return CComplex<T>(s.real(), s.imag());
}
 
 
//! Returns complex power of a complex number.
template <typename T>
inline CComplex<T> getPow(const CComplex<T>& x, int y)
{
    std::complex<T> s = std::pow(std::complex<T>(x), y);
    
    return CComplex<T>(s.real(), s.imag());
}*/


//! Returns cosine of the complex number.
template <typename T>
inline CComplex<T> getCos(const CComplex<T>& z)
{
    return CComplex<T>(cos(z.re) * cosh(z.im), -sin(z.re) * sinh(z.im));
}


//! Returns hyperbolic cosine of the complex number.
template <typename T>
inline CComplex<T> getCosh(const CComplex<T>& z)
{
    return CComplex<T>(cosh(z.re) * cos(z.im), sinh(z.re) * sin(z.im));
}


//! Returns sine of a complex number.
template <typename T>
inline CComplex<T> getSin(const CComplex<T>& z)
{
    return CComplex<T>(sin(z.re) * cosh(z.im), cos(z.re) * sinh(z.im));
}


//! Returns hyperbolic sine of a complex number.
template <typename T>
inline CComplex<T> getSinh(const CComplex<T>& z)
{
    return CComplex<T>(sinh(z.re) * cos(z.im), cosh(z.re) * sin(z.im));
}


//! Returns tangent of a complex number.
/*template <typename T>
inline CComplex<T> getTan(const CComplex<T>& z)
{
    std::complex<T> s = std::tan(std::complex<T>(z.re, z.im));
    
    return CComplex<T>(s.real(), s.imag());
}
 
 
//! Returns hyperbolic tangent of a complex number.
template <typename T> 
inline CComplex<T> getTanh(const CComplex<T>& z)
{
    std::complex<T> s = std::tanh(std::complex<T>(z.re, z.im));
    
    return CComplex<T>(s.real(), s.imag());
}*/


//! Writes complex number to an output stream.
template <typename T>
inline std::ostream& operator <<(std::ostream& Stream, const CComplex<T>& z)
{
    Stream << z.re;
    if( z.im )
    {
        if( z.im > 0 )
        {
            Stream << '+';
        }
        Stream << z.im << 'i';
    }

    return Stream;
}


//! Reads complex number from an input stream.
template <typename T>
inline std::istream& operator >> (std::istream& Stream, CComplex<T>& z)
{
    Stream >> z.re >> z.im;

    return Stream;
}


//=============================================================================
/*
 * Basic template instances and type definitions.
 */

//! Float complex number.
typedef CComplex<float>     CFComplex;

//! Double complex number.
typedef CComplex<double>    CDComplex;


//! Float complex number.
typedef CFComplex           tFComplex;

//! Double complex number.
typedef CDComplex           tDComplex;


} // namespace math
} // namespace mds

#endif // MDS_COMPLEX_H

