//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsBase.h                           \n
 * Section: libMath                             \n
 * Date:    2003/10/16                          \n
 *
 * $Id: mdsBase.h 414 2007-07-03 14:06:55Z spanel $
 * 
 * Description:
 * - Definition of various useful constants, functions and macros.
 */

#ifndef MDS_BASE_H
#define MDS_BASE_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsStaticAssert.h>
#include <MDSTk/Base/mdsTypeTraits.h>


namespace mds
{
namespace math
{

//=============================================================================
/*
 * Definitions of various constants.
 */

//! Euler's constant.
const double E              = 2.718281828459045;

//! Constant pi.
const double PI             = 3.141592653589793;

//! Constant 2*pi.
const double TWO_PI         = 2.0 * PI;

//! Constant pi/2.
const double HALF_PI        = 0.5 * PI;

//! Constant sqrt(2*pi).
const double SQRT_TWO_PI    = 2.506628274631001;

//! Square root of the number 2.
const double SQRT2          = 1.414213562373095;

//! Square root of the number 3.
const double SQRT3          = 1.732050807568877;

//! Square root of the number 6.
const double SQRT6          = 2.449489742783178;


//=============================================================================
/*
 * Several useful template functions.
 */

//! Returns maximum of two values.
template <typename T>
inline const T& getMax(const T& a, const T& b)
{
    return ((a > b) ? a : b);
}


//! Returns maximum of three values.
template <typename T>
inline const T& getMax(const T& a, const T& b, const T& c)
{
    return getMax(getMax(a, b), c);
}


//! Returns maximum of four values.
template <typename T>
inline const T& getMax(const T& a, const T& b, const T& c, const T& d)
{
    return getMax(getMax(getMax(a, b), c), d);
}


//! Returns minimum of two values.
template <typename T>
inline const T& getMin(const T& a, const T& b)
{
    return ((a < b) ? a : b);
}


//! Returns minimum of three values.
template <typename T>
inline const T& getMin(const T& a, const T& b, const T& c)
{
    return getMin(getMin(a, b), c);
}


//! Returns minimum of four values.
template <typename T>
inline const T& getMin(const T& a, const T& b, const T& c, const T& d)
{
    return getMin(getMin(getMin(a, b), c), d);
}


//! Absolute value.
template <typename T>
inline T getAbs(const T& a)
{
    return ((a > 0) ? a : -a);
}


//! Exchanges two values.
template <typename T>
inline void swap(T& a, T& b)
{
    T Temp = a;
    a = b;
    b = Temp;
}


//! Rounds a given floating point number to integer.
template <typename T>
inline int round2Int(const T& x)
{
    MDS_STATIC_ASSERT(CTypeTraits<T>::isFloat, Bad_template_parameter_type);
    
    static const T Half = 0.5f;

    return int(x + Half);
}


//! Conversion of a given number to bool.
template <typename T>
inline bool conv2Bool(const T& x)
{
    static const T Zero = 0;

    return ((x == Zero) ? false : true);
}


//! Value x is mirrored into a given interval <a, b>.
template <typename T>
inline void mirror(T& x, const T& a, const T& b)
{
    while( (x < a) || (x > b) )
    {
        if( x < a )
        {
            x = 2 * a - x;
        }
        if( x > b )
        {
            x = 2 * b - x;
        }
    }
}


//! Value x is limited into an interval <a, b>.
template <typename T>
inline void limit(T& x, const T& a, const T& b)
{
    if( x < a )
    {
        x = a;
    }
    else if( x > b )
    {
        x = b;
    }
}


//! Returns true if a given number is odd.
template <typename T>
inline bool isOdd(const T& x)
{
    MDS_STATIC_ASSERT(CTypeTraits<T>::isIntegral, Bad_template_parameter_type);

    return (x & 1) == 1;
}


//! Returns true if a given number is even.
template <typename T>
inline bool isEven(const T& x)
{
    MDS_STATIC_ASSERT(CTypeTraits<T>::isIntegral, Bad_template_parameter_type);

    return (x & 1) == 0;
}


//=============================================================================
/*
 * Several obsolete macros.
 */

//! Maximum of a and b.
//! - Obsolete!
#define MAX(a, b)        (((a) > (b)) ? (a) : (b))

//! Minimum of a and b.
//! - Obsolete!
#define MIN(a, b)        (((a) < (b)) ? (a) : (b))

//! Absolute value of a.
//! - Obsolete!
#define ABS(a)            (((a) > 0) ? (a) : (-(a)))

//! Exchanges integer numbers a and b.
//! - Obsolete!
#define SWAP(a, b)        { (a) ^= (b); (b) ^= (a); (a) ^= (b); }

//! Rounds double to int.
//! - Obsolete!
#define ROUND2INT(x)    (int(x + 0.5))

//! Integer to bool conversion.
//! - Obsolete!
#define CONV2BOOL(x)    (((x) == 0) ? false : true)

//! Value x is mirrored into a given interval <a,b>.
//! - Obsolete!
#define MIRROR(x, a, b) \
{ \
    while( ((x) < (a)) || ((x) > (b)) ) \
    { \
        if( (x) < (a) ) (x) = 2 * (a) - (x); \
        if( (x) > (b) ) (x) = 2 * (b) - (x); \
    } \
}

//! Value x is limited into a given interval <a,b>.
//! - Obsolete!
#define LIMIT(x, a, b) \
{ \
    if( (x) < (a) ) (x) = (a); \
    else if( (x) > (b) ) (x) = (b); \
}


} // namespace math
} // namespace mds

#endif // MDS_BASE_H

