//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VectorFunctions/mdsGeneral.hxx      \n
 * Section: libMath                             \n
 * Date:    2006/02/15                          \n
 *
 * $Id: mdsGeneral.hxx 95 2006-09-24 20:11:00Z spanel $
 *
 * Description:
 * - General vector functions.
 */


//==============================================================================
/*
 * Functions templates.
 */

template <typename R, class V1, class V2>
inline R getProd(const CVectorBase<V1>& Vector1, const CVectorBase<V2>& Vector2)
{
    typedef typename V1::tElement tElement1;
    typedef typename V2::tElement tElement2;
    const V1& Vector1Impl = Vector1.getImpl();
    const V2& Vector2Impl = Vector2.getImpl();

    tSize Count = mds::math::getMin(Vector1Impl.getSize(), Vector2Impl.getSize());

    const tElement1 *p1 = Vector1Impl.getPtr();
    const tElement1 *p1Max = Vector1Impl.getPtr(Count);
    const tElement2 *p2 = Vector2Impl.getPtr();

    R Total = R(0);
    while( p1 < p1Max )
    {
        Total += R(*p1) * R(*p2);
        p1 += Vector1Impl.getStride();
        p2 += Vector2Impl.getStride();
    }
    return Total;
}


template <typename R, class V>
inline R getSum(const CVectorBase<V>& Vector)
{
    typedef typename V::tElement tElement;
    const V& VectorImpl = Vector.getImpl();

    const tElement *p = VectorImpl.getPtr();
    const tElement *pMax = VectorImpl.getPtr(VectorImpl.getSize());

    R Total = R(0);
    while( p < pMax )
    {
        Total += R(*p);
        p += VectorImpl.getStride();
    }
    return Total;
}


template <typename R, class V>
inline R getMult(const CVectorBase<V>& Vector)
{
    typedef typename V::tElement tElement;
    const V& VectorImpl = Vector.getImpl();

    const tElement *p = VectorImpl.getPtr();
    const tElement *pMax = VectorImpl.getPtr(VectorImpl.getSize());

    R Total = R(1);
    while( p < pMax )
    {
        Total *= R(*p);
        p += VectorImpl.getStride();
    }
    return Total;
}


template <typename R, class V>
inline R getSumOfSquares(const CVectorBase<V>& Vector)
{
    typedef typename V::tElement tElement;
    const V& VectorImpl = Vector.getImpl();

    const tElement *p = VectorImpl.getPtr();
    const tElement *pMax = VectorImpl.getPtr(VectorImpl.getSize());

    R Total = R(0);
    while( p < pMax )
    {
        R Temp = R(*p);
        Total += Temp * Temp;
        p += VectorImpl.getStride();
    }
    return Total;
}


template <typename R, class V1, class V2>
inline R getSquareError(const CVectorBase<V1>& Vector1, const CVectorBase<V2>& Vector2)
{
    typedef typename V1::tElement tElement1;
    typedef typename V2::tElement tElement2;
    const V1& Vector1Impl = Vector1.getImpl();
    const V2& Vector2Impl = Vector2.getImpl();

    tSize Count = mds::math::getMin(Vector1Impl.getSize(), Vector2Impl.getSize());
    MDS_CHECK(Count > 0, return R(0));

    const tElement1 *p1 = Vector1Impl.getPtr();
    const tElement1 *p1Max = Vector1Impl.getPtr(Count);
    const tElement2 *p2 = Vector2Impl.getPtr();

    R Total = R(0);
    while( p1 < p1Max )
    {
        R Temp = R(*p1) - R(*p2);
        Total += Temp * Temp;
        p1 += Vector1Impl.getStride();
        p2 += Vector2Impl.getStride();
    }
    return Total / R(Count);
}


template <typename R, class V1, class V2>
inline R getError(const CVectorBase<V1>& Vector1, const CVectorBase<V2>& Vector2)
{
    typedef typename V1::tElement tElement1;
    typedef typename V2::tElement tElement2;
    const V1& Vector1Impl = Vector1.getImpl();
    const V2& Vector2Impl = Vector2.getImpl();

    tSize Count = mds::math::getMin(Vector1Impl.getSize(), Vector2Impl.getSize());
    MDS_CHECK(Count > 0, return R(0));

    const tElement1 *p1 = Vector1Impl.getPtr();
    const tElement1 *p1Max = Vector1Impl.getPtr(Count);
    const tElement2 *p2 = Vector2Impl.getPtr();

    R Total = R(0);
    while( p1 < p1Max )
    {
        R Temp = R(*p1) - R(*p2);
        Total += Temp * Temp;
        p1 += Vector1Impl.getStride();
        p2 += Vector2Impl.getStride();
    }
    return std::sqrt(Total) / R(Count);
}


template <typename R, class V>
inline R getMean(const CVectorBase<V>& Vector)
{
    typedef typename V::tElement tElement;
    const V& VectorImpl = Vector.getImpl();

    tSize Count = VectorImpl.getSize();
    MDS_CHECK(Count > 0, return R(0));

    const tElement *p = VectorImpl.getPtr();
    const tElement *pMax = VectorImpl.getPtr(Count);

    R Total = R(0);
    while( p < pMax )
    {
        Total += R(*p);
        p += VectorImpl.getStride();
    }
    return Total / R(Count);
}


template <typename R, class V>
inline R getVariance(const CVectorBase<V>& Vector)
{
    typedef typename V::tElement tElement;
    const V& VectorImpl = Vector.getImpl();

    tSize Count = VectorImpl.getSize();
    MDS_CHECK(Count > 0, return R(0));

    const tElement *p = VectorImpl.getPtr();
    const tElement *pMax = VectorImpl.getPtr(Count);

    R Sum = R(0), SumSqr = R(0);
    while( p < pMax )
    {
        R Temp = R(*p);
        Sum += Temp;
        SumSqr += Temp * Temp;
        p += VectorImpl.getStride();
    }
    R Temp = R(1) / R(Count);
    return SumSqr * Temp - (Sum * Temp * Sum * Temp);
}


template <typename R, class V>
inline R getMin(const CVectorBase<V>& Vector)
{
    typedef typename V::tElement tElement;
    const V& VectorImpl = Vector.getImpl();

    tSize Count = VectorImpl.getSize();
    MDS_CHECK(Count > 0, return R(0));

    const tElement *p = VectorImpl.getPtr(1);
    const tElement *pMax = VectorImpl.getPtr(Count);

    tElement MinVal = VectorImpl.get(0);
    while( p < pMax )
    {
        if( *p < MinVal )
        {
            MinVal = *p;
        }
        p += VectorImpl.getStride();
    }
    return MinVal;
}


template <typename R, class V>
inline R getMax(const CVectorBase<V>& Vector)
{
    typedef typename V::tElement tElement;
    const V& VectorImpl = Vector.getImpl();

    tSize Count = VectorImpl.getSize();
    MDS_CHECK(Count > 0, return R(0));

    const tElement *p = VectorImpl.getPtr(1);
    const tElement *pMax = VectorImpl.getPtr(Count);

    tElement MaxVal = VectorImpl.get(0);
    while( p < pMax )
    {
        if( *p > MaxVal )
        {
            MaxVal = *p;
        }
        p += VectorImpl.getStride();
    }
    return MaxVal;
}


template <class V>
std::ostream& operator <<(std::ostream& Stream, const CVectorBase<V>& Vector)
{
    const V& VectorImpl = Vector.getImpl();

//  Stream.setf(std::ios_base::fixed);
    for( mds::tSize i = 0; i < VectorImpl.getSize(); i++ )
    {
        Stream << VectorImpl(i) << "  ";
    }

    return Stream;
}


template <class V1, class V2>
inline void inplaceAdd(CVectorBase<V2>& R, const CVectorBase<V1>& A)
{
    typedef typename V1::tElement tElement1;
    typedef typename V2::tElement tElement2;
    const V1& AImpl = A.getImpl();
    V2& RImpl = R.getImpl();

    tSize Count = mds::math::getMin(AImpl.getSize(), RImpl.getSize());

    const tElement1 *pA = AImpl.getPtr();
    const tElement1 *pAMax = AImpl.getPtr(Count);
    tElement2 *pR = RImpl.getPtr();

    while( pA < pAMax )
    {
        *pR += *pA;
        pA += AImpl.getStride();
        pR += RImpl.getStride();
    }
}


template <class V1, class V2>
inline void inplaceSub(CVectorBase<V2>& R, const CVectorBase<V1>& A)
{
    typedef typename V1::tElement tElement1;
    typedef typename V2::tElement tElement2;
    const V1& AImpl = A.getImpl();
    V2& RImpl = R.getImpl();

    tSize Count = mds::math::getMin(AImpl.getSize(), RImpl.getSize());

    const tElement1 *pA = AImpl.getPtr();
    const tElement1 *pAMax = AImpl.getPtr(Count);
    tElement2 *pR = RImpl.getPtr();

    while( pA < pAMax )
    {
        *pR -= *pA;
        pA += AImpl.getStride();
        pR += RImpl.getStride();
    }
}


template <typename T, class V1, class V2>
inline void inplaceMultAdd(CVectorBase<V2>& R, const CVectorBase<V1>& A, const T& b)
{
    typedef typename V1::tElement tElement1;
    typedef typename V2::tElement tElement2;
    const V1& AImpl = A.getImpl();
    V2& RImpl = R.getImpl();

    tSize Count = mds::math::getMin(AImpl.getSize(), RImpl.getSize());

    const tElement1 *pA = AImpl.getPtr();
    const tElement1 *pAMax = AImpl.getPtr(Count);
    tElement2 *pR = RImpl.getPtr();

    while( pA < pAMax )
    {
        *pR += *pA * b;
        pA += AImpl.getStride();
        pR += RImpl.getStride();
    }
}

