//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    MatrixFunctions/mdsInverse.hxx      \n
 * Section: libMath                             \n
 * Date:    2006/02/16                          \n
 *
 * $Id: mdsInverse.hxx 110 2006-10-12 21:25:52Z spanel $
 *
 * Description:
 * - Computes inverse of a square matrix.
 */


//==============================================================================
/*
 * Functions templates.
 */

template <class M>
inline void inverse(CMatrixBase<M>& Matrix)
{
    typedef typename M::tElement tElement;
    M& MatrixImpl = Matrix.getImpl();

    // Check that the matrix is square
    if( !MatrixImpl.isSquare() || MatrixImpl.getColOffset() != 1 )
    {
        MDS_THROW_WARNING("Function inverse() failed");
    }

    // Size of the matrix
    tSize Size = MatrixImpl.getNumOfRows();

    switch( Size )
    {
        // The matrix 1x1
        case 1: {
            // Determinant of the matrix 1x1
            double dDet = double(MatrixImpl(0,0));
            if (dDet == 0.0)
            {
                MDS_THROW_WARNING("Function inverse() failed");
            }
        
            // Inverse of the matrix 1x1
            MatrixImpl(0,0) = tElement(1.0 / dDet);
            } break;

        // The matrix 2x2
        case 2: {
            // Matrix elements
            tElement a = MatrixImpl(0,0);
            tElement b = MatrixImpl(0,1);
            tElement c = MatrixImpl(1,0);
            tElement d = MatrixImpl(1,1);
    
            // Determinant of the matrix 2x2
            double dDet = double(a * d) - double(b * c);
            if (dDet == 0.0)
            {
                MDS_THROW_WARNING("Function inverse() failed");
            }
    
            // Inverse of the matrix 1x1
            dDet = 1.0 / dDet;
            MatrixImpl(0,0) = tElement(d * dDet);
            MatrixImpl(0,1) = tElement(-b * dDet);
            MatrixImpl(1,0) = tElement(-c * dDet);
            MatrixImpl(1,1) = tElement(a * dDet);
            } break;

        // The matrix nxn
        default: {
            // Helper vector
            CVector<int> Pivots(Size);
    
            // LU factorization of the matrix
            MDS_LAPACK.getrf(CblasRowMajor,
                             Size, Size,
                             MatrixImpl.getPtr(),
                             MatrixImpl.getRowOffset(),
                             Pivots.getPtr()
                             );
    
            // Inverse of the matrix
            MDS_LAPACK.getri(CblasRowMajor,
                             Size,
                             MatrixImpl.getPtr(),
                             MatrixImpl.getRowOffset(),
                             Pivots.getPtr()
                             );
            } break;
    } // end of switch
}

