//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    MatrixFunctions/mdsGeneral.hxx      \n
 * Section: libMath                             \n
 * Date:    2006/02/15                          \n
 *
 * $Id: mdsGeneral.hxx 91 2006-09-15 21:41:18Z spanel $
 *
 * Description:
 * - General matrix functions.
 */


//==============================================================================
/*
 * Functions templates.
 */

template <typename R, class M1, class M2>
inline R getProduct(const CMatrixBase<M1>& Matrix1, const CMatrixBase<M2>& Matrix2)
{
    typedef typename M1::tElement tElement1;
    typedef typename M2::tElement tElement2;
    const M1& Matrix1Impl = Matrix1.getImpl();
    const M2& Matrix2Impl = Matrix2.getImpl();

    tSize NumOfCols = mds::math::getMin(Matrix1Impl.getColSize(), Matrix2Impl.getColSize());
    tSize NumOfRows = mds::math::getMin(Matrix1Impl.getRowSize(), Matrix2Impl.getRowSize());

    tSize Count = NumOfCols * NumOfRows;
    MDS_CHECK(Count > 0, return R(0));

    R Total = R(0);
    for( tSize j = 0; j < NumOfRows; ++j )
    {
        const tElement1 *p1 = Matrix1Impl.getRowPtr(j);
        const tElement1 *p1Max = p1 + NumOfCols;
        const tElement2 *p2 = Matrix2Impl.getRowPtr(j);
        while( p1 < p1Max )
        {
            Total += R(*(p2++)) * R(*(p1++));
        }
    }
    return Total;
}


template <typename R, class M1, class M2>
inline R getSquareError(const CMatrixBase<M1>& Matrix1, const CMatrixBase<M2>& Matrix2)
{
    typedef typename M1::tElement tElement1;
    typedef typename M2::tElement tElement2;
    const M1& Matrix1Impl = Matrix1.getImpl();
    const M2& Matrix2Impl = Matrix2.getImpl();

    tSize NumOfCols = mds::math::getMin(Matrix1Impl.getColSize(), Matrix2Impl.getColSize());
    tSize NumOfRows = mds::math::getMin(Matrix1Impl.getRowSize(), Matrix2Impl.getRowSize());

    tSize Count = NumOfCols * NumOfRows;
    MDS_CHECK(Count > 0, return R(0));

    R Total = R(0);
    for( tSize j = 0; j < NumOfRows; ++j )
    {
        const tElement1 *p1 = Matrix1Impl.getRowPtr(j);
        const tElement1 *p1Max = p1 + NumOfCols;
        const tElement2 *p2 = Matrix2Impl.getRowPtr(j);
        while( p1 < p1Max )
        {
            R Temp = R(*(p2++)) - R(*(p1++));
            Total += Temp * Temp;
        }
    }
    return (Total / R(Count));
}


template <typename R, class M>
inline R getSum(const CMatrixBase<M>& Matrix)
{
    typedef typename M::tElement tElement;
    const M& MatrixImpl = Matrix.getImpl();

    R Total = R(0);
    for( tSize j = 0; j < MatrixImpl.getNumOfRows(); ++j )
    {
        const tElement *p = MatrixImpl.getRowPtr(j);
        const tElement *pMax = p + MatrixImpl.getNumOfCols();
        while( p < pMax )
        {
            Total += R(*(p++));
        }
    }
    return Total;
}


template <typename R, class M>
inline R getMult(const CMatrixBase<M>& Matrix)
{
    typedef typename M::tElement tElement;
    const M& MatrixImpl = Matrix.getImpl();

    R Total = R(1);
    for( tSize j = 0; j < MatrixImpl.getNumOfRows(); ++j )
    {
        const tElement *p = MatrixImpl.getRowPtr(j);
        const tElement *pMax = p + MatrixImpl.getNumOfCols();
        while( p < pMax )
        {
            Total *= R(*(p++));
        }
    }
    return Total;
}


template <typename R, class M>
inline R getSumOfSquares(const CMatrixBase<M>& Matrix)
{
    typedef typename M::tElement tElement;
    const M& MatrixImpl = Matrix.getImpl();

    R Total = R(0);
    for( tSize j = 0; j < MatrixImpl.getNumOfRows(); ++j )
    {
        const tElement *p = MatrixImpl.getRowPtr(j);
        const tElement *pMax = p + MatrixImpl.getNumOfCols();
        while( p < pMax )
        {
            R Sample = R(*(p++));
            Total += Sample * Sample;
        }
    }
    return Total;
}


template <typename R, class M>
inline R getTrace(const CMatrixBase<M>& Matrix)
{
    typedef typename M::tElement tElement;
    const M& MatrixImpl = Matrix.getImpl();

    tSize Size = mds::math::getMin(MatrixImpl.getNumOfRows(), MatrixImpl.getNumOfCols());

    const tElement *p = MatrixImpl.getPtr();
    const tElement *pMax = MatrixImpl.getRowPtr(Size);

    R Total = R(0);
    while( p < pMax )
    {
        Total += R(*p);
        p += MatrixImpl.getRowOffset() + 1;
    }
    return Total;
}


template <typename R, class M>
inline R getMean(const CMatrixBase<M>& Matrix)
{
    typedef typename M::tElement tElement;
    const M& MatrixImpl = Matrix.getImpl();

    tSize Count = MatrixImpl.getNumOfRows() * MatrixImpl.getNumOfCols();
    MDS_CHECK(Count > 0, return R(0));

    R Total = R(0);
    for( tSize j = 0; j < MatrixImpl.getNumOfRows(); j++ )
    {
        const tElement *p = MatrixImpl.getRowPtr(j);
        const tElement *pMax = p + MatrixImpl.getNumOfCols();
        while( p < pMax )
        {
            Total += R(*(p++));
        }
    }
    return Total / R(Count);
}


template <typename R, class M>
inline R getVariance(const CMatrixBase<M>& Matrix)
{
    typedef typename M::tElement tElement;
    const M& MatrixImpl = Matrix.getImpl();

    tSize Count = MatrixImpl.getNumOfRows() * MatrixImpl.getNumOfCols();
    MDS_CHECK(Count > 0, return R(0));

    R Sum = R(0), SumSqr = R(0);
    for( tSize j = 0; j < MatrixImpl.getNumOfRows(); ++j )
    {
        const tElement *p = MatrixImpl.getRowPtr(j);
        const tElement *pMax = p + MatrixImpl.getNumOfCols();
        while( p < pMax )
        {
            R Temp = R(*(p++));
            Sum += Temp;
            SumSqr += Temp * Temp;
        }
    }
    R Temp = R(1) / R(Count);
    return SumSqr * Temp - (Sum * Temp * Sum * Temp);
}


template <typename R, class M>
inline R getMin(const CMatrixBase<M>& Matrix)
{
    typedef typename M::tElement tElement;
    const M& MatrixImpl = Matrix.getImpl();

    tSize Count = MatrixImpl.getNumOfRows() * MatrixImpl.getNumOfCols();
    MDS_CHECK(Count > 0, return R(0));

    tElement MinVal = *MatrixImpl.getPtr();
    for( tSize j = 0; j < MatrixImpl.getNumOfRows(); ++j )
    {
        const tElement *p = MatrixImpl.getRowPtr(j);
        const tElement *pMax = p + MatrixImpl.getNumOfCols();
        while( p < pMax )
        {
            if( *p < MinVal )
            {
                MinVal = *p;
            }
            ++p;
        }
    }
    return R(MinVal);
}


template <typename R, class M>
inline R getMax(const CMatrixBase<M>& Matrix)
{
    typedef typename M::tElement tElement;
    const M& MatrixImpl = Matrix.getImpl();

    tSize Count = MatrixImpl.getNumOfRows() * MatrixImpl.getNumOfCols();
    MDS_CHECK(Count > 0, return R(0));

    tElement MaxVal = *MatrixImpl.getPtr();
    for( tSize j = 0; j < MatrixImpl.getNumOfRows(); ++j )
    {
        const tElement *p = MatrixImpl.getRowPtr(j);
        const tElement *pMax = p + MatrixImpl.getNumOfCols();
        while( p < pMax )
        {
            if( *p > MaxVal )
            {
                MaxVal = *p;
            }
            ++p;
        }
    }
    return R(MaxVal);
}


template <class M>
std::ostream& operator <<(std::ostream& Stream, const CMatrixBase<M>& Matrix)
{
    const M& MatrixImpl = Matrix.getImpl();

//  Stream.setf(std::ios_base::fixed);
    for( tSize i = 0; i < MatrixImpl.getNumOfRows(); ++i )
    {
        for( tSize j = 0; j < MatrixImpl.getNumOfCols(); ++j )
        {
            Stream << MatrixImpl(i,j) << "  ";
        }
        if( i < MatrixImpl.getNumOfRows() - 1 )
        {
            Stream << std::endl;
        }
    }
    
    return Stream;
}

