//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    MatrixFunctions/mdsEig.hxx          \n
 * Section: libMath                             \n
 * Date:    2006/10/12                          \n
 *
 * $Id: mdsEig.hxx 331 2007-04-17 14:43:06Z spanel $
 *
 * Description:
 * - Computes eigenvalues and eigenvectors of a square symmetric matrix.
 */


//==============================================================================
/*
 * Functions templates.
 */

template <class M, class V>
inline void eig(CMatrixBase<M>& Matrix, CVectorBase<V>& Values)
{
    typedef typename M::tElement tMatrixElement;
    M& MatrixImpl = Matrix.getImpl();

    typedef typename V::tElement tVectorElement;
    V& ValuesImpl = Values.getImpl();

    // Check matrix and vector elements types
    enum { Helper = mds::SCheckIdentity<tMatrixElement, tVectorElement>::RESULT };
    MDS_STATIC_ASSERT(Helper, Matrix_element_type_differs_from_vector_type);

    // Check that the matrix is square
    if( !MatrixImpl.isSquare() || MatrixImpl.getColOffset() != 1 )
    {
        MDS_THROW_WARNING("Function eig() failed");
    }

    // Size of the matrix
    tSize Size = MatrixImpl.getNumOfRows();

    // Check size of the vector
    if( ValuesImpl.getSize() != Size )
    {
        MDS_THROW_WARNING("Function eig() failed");
    }

    // Estimate size of workspace
    CVector<tVectorElement> Work(1);
    MDS_LAPACK.syev('V', 'U',
                    Size,
                    MatrixImpl.getPtr(),
                    MatrixImpl.getRowOffset(),
                    ValuesImpl.getPtr(),
                    Work.getPtr(),
                    -1
                    );

    // Resize the workspace
    Work.create(tSize(Work(0)));

    // LU factorization of the matrix
    MDS_LAPACK.syev('V', 'U',
                    Size,
                    MatrixImpl.getPtr(),
                    MatrixImpl.getRowOffset(),
                    ValuesImpl.getPtr(),
                    Work.getPtr(),
                    Work.getSize()
                    );
}

