//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    MatrixFunctions/mdsDeterminant.hxx  \n
 * Section: libMath                             \n
 * Date:    2006/02/16                          \n
 *
 * $Id: mdsDeterminant.hxx 449 2007-09-05 10:50:05Z siler $
 * 
 * Description:
 * - Computes determinant of a square matrix.
 */


//==============================================================================
/*
 * Functions templates.
 */

template <typename R, class M>
inline R getDeterminant(const CMatrixBase<M>& Matrix)
{
    typedef typename M::tElement tElement;
    const M& MatrixImpl = Matrix.getImpl();

    // Check that the matrix is square
    if( !MatrixImpl.isSquare() )
    {
        MDS_THROW_WARNING("Function getDeterminant() failed");
    }

    // Size of the matrix
    tSize Size = MatrixImpl.getNumOfRows();

    R Result;
    switch( Size )
    {
        // The matrix 1x1
        case 1:
            Result = R(MatrixImpl.get(0,0));
            break;

        // The matrix 2x2
        case 2:
            Result  = R(MatrixImpl.get(0,0) * MatrixImpl.get(1,1));
            Result -= R(MatrixImpl.get(0,1) * MatrixImpl.get(1,0));
            break;

        // The matrix 3x3
        case 3:
            Result  = R(MatrixImpl.get(0,0) * MatrixImpl.get(1,1) * MatrixImpl.get(2,2));
            Result += R(MatrixImpl.get(0,1) * MatrixImpl.get(1,2) * MatrixImpl.get(2,0));
            Result += R(MatrixImpl.get(0,2) * MatrixImpl.get(1,0) * MatrixImpl.get(2,1));
            Result -= R(MatrixImpl.get(0,2) * MatrixImpl.get(1,1) * MatrixImpl.get(2,0));
            Result -= R(MatrixImpl.get(0,0) * MatrixImpl.get(1,2) * MatrixImpl.get(2,1));
            Result -= R(MatrixImpl.get(0,1) * MatrixImpl.get(1,0) * MatrixImpl.get(2,2));
            break;

        // The matrix nxn
        default: {
            // Helper vector
            CVector<int> Pivots(Size);
    
            // Helper matrix
            M InvMatrix(MatrixImpl);
    
            // LU factorization of the matrix
            MDS_LAPACK.getrf(CblasRowMajor,
                             Size, Size,
                             InvMatrix.getPtr(),
                             Size,
                             Pivots.getPtr()
                             );
    
            // Compute determinant
            Result = R(1);
            for( tSize i = 0; i < Size; ++i )
            {
                if( Pivots(i) != i )
                {
                    Result *= R(-1);
                }
                Result *= R(InvMatrix(i,i));
            }
            } break;
    } // end of switch

    // O.K.
    return Result;
}

