//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 *          Premysl Krsek, krsek@fit.vutbr.cz   \n
 * File:    mdsDicomSlice.h                     \n
 * Section: libImageIO                          \n
 * Date:    2003/12/15                          \n
 *
 * $Id: mdsDicomSlice.h 353 2007-06-12 14:07:22Z spanel $
 *
 * Description:
 * - Manipulation with DICOM files.
 */

#ifndef MDS_DICOMSLICE_H
#define MDS_DICOMSLICE_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Image/mdsSlice.h>
#include <MDSTk/Image/mdsPoint.h>
#include <MDSTk/Image/mdsVector.h>
#include <MDSTk/Module/mdsChannel.h>

// STL
#include <string>


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Class encapsulating DICOM file operations.
 */
class CDicomSlice : public CSlice
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CDicomSlice);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CDicomSlice);

public:
    //! Patient name.
    std::string m_sPatientName;

    //! Patient Id.
    std::string m_sPatientId;
    
    //! Patient birthday.
    std::string m_sPatientBirthday;

    //! Patient sex.
    std::string m_sPatientSex;

    //! Patient description.
    std::string m_sPatientDescription;

    //! Study uid.
    std::string m_sStudyUid;    
    
    //! Study id.
    std::string m_sStudyId;
    
    //! Study date.
    std::string m_sStudyDate;    

    //! Study description.
    std::string m_sStudyDescription; 
    
    //! Series Uid.
    std::string m_sSeriesUid;

    //! Series number.
    int m_iSeriesNumber;

    //! Modality.
    std::string m_sModality;

    //! Series date.
    std::string m_sSeriesDate;

    //! Series time.
    std::string m_sSeriesTime;

    //! Series description.
    std::string m_sSeriesDescription;

    //! Slice number.
    int m_iSliceNumber;

    //! Density window center position.
    int m_iWindowCenter;

    //! Density window width.
    int m_iWindowWidth;

    //! Patient position.
    std::string m_sPatientPosition;

    //! Explicit transfer syntax.
    bool m_bExplicitTransfer;

    //! Pixel value representation.
    unsigned short int m_usPixelRepresentation;

    //! Slice base point position.
    CPoint3D m_ImagePosition;

    //! Slice X axis (first row) direction.
    CVector3D m_ImageOrientationX;

    //! Slice Y axis (first column) direction.
    CVector3D m_ImageOrientationY;

public:
    //! Default constructor.
    CDicomSlice();

    //! Constructor.
    CDicomSlice(tSize XSize, tSize YSize);

    //! Constructor.
    //! - Plus a pixel initial value.
    CDicomSlice(tSize XSize,
                tSize YSize,
                const CDImage::tPixel& Value
                );

    //! Copy constructor.
    //! - Makes a new copy of the image data.
    CDicomSlice(const CDImage& Image);

    //! Copy constructor.
    //! - Makes a reference only.
    CDicomSlice(const CDImage& Image, EMakeRef);

    //! Copy constructor.
    //! - Makes a new copy of the image data.
    CDicomSlice(const CSlice& Slice);

    //! Copy constructor.
    //! - Makes a reference only.
    CDicomSlice(const CSlice& Slice, EMakeRef);

    //! Copy constructor.
    //! - Makes a new copy of the image data.
    CDicomSlice(const CDicomSlice& Slice);

    //! Copy constructor.
    //! - Makes a reference only.
    CDicomSlice(const CDicomSlice& Slice, EMakeRef);

    //! Destructor.
    virtual ~CDicomSlice();

    //! Image assignment operator.
    virtual CDicomSlice& operator =(const CDImage& Image);

    //! Slice assignment operator.
    virtual CDicomSlice& operator =(const CSlice& Slice);

    //! Slice assignment operator.
    virtual CDicomSlice& operator =(const CDicomSlice& Slice);


    //! Loads DICOM slice from a given channel.
    //! - Returns false on failure.
    bool loadDicom(mds::mod::CChannel *pChannel);

private:
    //! Reads and checks DICOM header.
    //! - Based on P. Krsek's code.
    bool readDicomHeader(mds::mod::CChannel *pChannel);

    //! Reads DICOM slice data.
    bool readDicom(mds::mod::CChannel *pChannel);

    //! Reads DICOM data element with VR = SQ, jump across, do not take data
    bool readSQDataElement(mds::mod::CChannel *pChannel);

    //! Gets element tag group and element.
    bool readElementTag(mds::mod::CChannel *pChannel,
                        unsigned short *pusGroup,
                        unsigned short *pusElem
                        );

    //! Reads the element tag value length.
    bool readValueLength(mds::mod::CChannel *pChannel,
                         char *pcType,
                         bool bExplicit,
                         unsigned long *pulLength,
                         unsigned long *pulOffset
                         );

    //! Reads the element data.
    char *readValueData(mds::mod::CChannel *pChannel, unsigned long ulLength);
};


//==============================================================================
/*
 * Smart pointer to DICOM slice.
 */
typedef CDicomSlice::tSmartPtr  CDicomSlicePtr;


} // namespace img
} // namespace mds

#endif // MDS_DICOMSLICE_H

