//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsVolumeIterator.h                 \n
 * Section: libVolume                           \n
 * Date:    2005/02/08                          \n
 *
 * $Id: mdsVolumeIterator.h 345 2007-06-11 13:23:09Z spanel $
 * 
 * Description:
 * - Volume iterator definition.
 */

#ifndef MDS_VOLUMEITERATOR_H
#define MDS_VOLUMEITERATOR_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsIterator.h>


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Iterator used to traverse a volume by lines.
 * - Template parameter V is the volume type.
 * - Parameter T is the voxel type.
 */
template <class V, typename T>
class CVolumeIterator : public mds::base::CComplexIteratorBase<V,T>
{
public:
    //! Check that type V is a volume. You will see name of this enum somewhere
    //! in compiler error message if the type V is not volume.
    enum { TEMPLATE_PARAMETER_IS_NOT_VOLUME = V::CLASS_VOLUME };

    //! Iterator base.
    typedef mds::base::CComplexIteratorBase<V,T> base;

    //! Volume type.
    typedef V tVolume;

    //! Voxel type.
    typedef T tVoxel;

public:
    //! Default constructor.
    CVolumeIterator() {}

    //! Constructor.
    CVolumeIterator(tVolume& Volume,
                    tSize x,
                    tSize y,
                    tSize z
                    )
        : mds::base::CComplexIteratorBase<V,T>(&Volume, Volume.getPtr(x, y, z))
        , m_x(x)
        , m_y(y)
        , m_z(z)
    {}

    //! Copy constructor.
    CVolumeIterator(const CVolumeIterator& It)
        : mds::base::CComplexIteratorBase<V,T>(It)
        , m_x(It.m_x)
        , m_y(It.m_y)
        , m_z(It.m_z)
    {}

    //! Destructor.
    ~CVolumeIterator() {}

    //! Assignment operator.
    CVolumeIterator& operator=(const CVolumeIterator& It)
    {
        base::m_pContainer = It.m_pContainer;
        base::m_pItem = It.m_pItem;
        m_x = It.m_x;
        m_y = It.m_y;
        m_z = It.m_z;
        return *this;
    }


    //! Operator moves the iterator on the next voxel.
    //! - Prefix notation.
    CVolumeIterator& operator++()
    {
        advance();
        return *this;
    }

    //! Operator moves the iterator to the next item.
    //! - Postfix notation.
    CVolumeIterator operator++(int)
    {
        CVolumeIterator Temp(*this);
        advance();
        return Temp;
    }

    //! Returns the number of increments between the positions addressed
    //! by two iterators.
    tSize getDistance(const CVolumeIterator& End)
    {
        tSize Count = (End.m_z * End.m_pContainer->getYSize() + End.m_y) * End.m_pContainer->getXSize() + End.m_x;
        Count -= (m_z * base::m_pContainer->getYSize() + m_y) * base::m_pContainer->getXSize() + m_x;
        return (Count > 0) ? Count : 0;
    }

    //! Returns current iterator position.
    tSize getX() const { return m_x; }
    tSize getY() const { return m_y; }
    tSize getZ() const { return m_z; }

    //! Sets the iterator position.
    void setPosition(tVolume& Volume, tSize x, tSize y, tSize z)
    {
        base::m_pContainer = &Volume;
        base::m_pItem = Volume.getPtr(x, y, z);
        m_x = x;
        m_y = y;
        m_z = z;
    }

protected:
    //! Iterator position.
    tSize m_x, m_y, m_z;

protected:
    //! Moves iterator to the next voxel.
    void advance()
    {
        base::m_pItem += base::m_pContainer->getXOffset();
        if( ++m_x >= base::m_pContainer->getXSize() )
        {
            m_x = 0;
            if( ++m_y >= base::m_pContainer->getYSize() )
            {
                m_y = 0;
                ++m_z;
            }
            base::m_pItem = base::m_pContainer->getPtr(m_x, m_y, m_z);
            return;
        }
    }
};


} // namespace img
} // namespace mds

#endif // MDS_VOLUMEITERATOR_H

