//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsVolumeFilterPolicy.h             \n
 * Section: libImage                            \n
 * Date:    2005/10/07                          \n
 *
 * $Id: mdsVolumeFilterPolicy.h 331 2007-04-17 14:43:06Z spanel $
 * 
 * Description:
 * - Policy for volume filter response normalization.
 */

#ifndef MDS_VOLUMEFILTERPOLICY_H
#define MDS_VOLUMEFILTERPOLICY_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsObject.h>
#include <MDSTk/Math/mdsBase.h>

#include "mdsImageTypes.h"
#include "mdsPixelTraits.h"
#include "mdsVolume.h"


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Policy for volume filter response normalization.
 * - Simple conversion to a pixel value.
 * - Parameter V is a volume type.
 */
template <class V>
class VFN_CONV
{
public:
    //! Check that type V is a volume. You will see name of this enum somewhere
    //! in compiler error message if the type V is not volume.
    enum { TEMPLATE_PARAMETER_IS_NOT_VOLUME = V::CLASS_VOLUME };

    //! Declare name of the class.
    MDS_STATIC_NAME(VFN_CONV);

    //! Volume voxel type.
    typedef typename V::tVoxel tVoxel;

    //! Volume filter response type.
    typedef typename CPixelTraits<tVoxel>::tFilterResponse tFilterResponse;

public:
    //! Static point response normalization function.
    static tVoxel normalize(tFilterResponse Value)
    {
        return tVoxel(Value);
    }
};


//==============================================================================
/*!
 * Policy for volume filter response normalization.
 * - Absolute value of the filter response.
 * - Parameter V is a volume type.
 */
template <class V>
class VFN_ABS
{
public:
    //! Check that type V is a volume. You will see name of this enum somewhere
    //! in compiler error message if the type V is not volume.
    enum { TEMPLATE_PARAMETER_IS_NOT_VOLUME = V::CLASS_VOLUME };

    //! Declare name of the class.
    MDS_STATIC_NAME(VFN_ABS);

    //! Volume voxel type.
    typedef typename V::tVoxel tVoxel;

    //! Volume filter response type.
    typedef typename CPixelTraits<tVoxel>::tFilterResponse tFilterResponse;

public:
    //! Static point response normalization function.
    static tVoxel normalize(tFilterResponse Value)
    {
        return tVoxel(mds::math::getAbs(Value));
    }
};


//==============================================================================
/*!
 * Policy for volume filter response normalization.
 * - Makes zero response equal to gray (mean pixel value).
 * - Parameter V is a volume type.
 */
template <class V>
class VFN_MEAN
{
public:
    //! Check that type V is a volume. You will see name of this enum somewhere
    //! in compiler error message if the type V is not volume.
    enum { TEMPLATE_PARAMETER_IS_NOT_VOLUME = V::CLASS_VOLUME };

    //! Declare name of the class.
    MDS_STATIC_NAME(VFN_MEAN);

    //! Volume voxel type.
    typedef typename V::tVoxel tVoxel;

    //! Volume filter response type.
    typedef typename CPixelTraits<tVoxel>::tFilterResponse tFilterResponse;

public:
    //! Static point response normalization function.
    static tVoxel normalize(tFilterResponse Value)
    {
        return tVoxel(Value + CPixelTraits<tVoxel>::getGray());
    }
};


//==============================================================================
/*!
 * Policy for volume filter response normalization.
 * - Only the positive filter response is accepted.
 * - Parameter V is a volume type.
 */
template <class V>
class VFN_POS
{
public:
    //! Check that type V is a volume. You will see name of this enum somewhere
    //! in compiler error message if the type V is not volume.
    enum { TEMPLATE_PARAMETER_IS_NOT_VOLUME = V::CLASS_VOLUME };

    //! Declare name of the class.
    MDS_STATIC_NAME(VFN_POS);

    //! Volume voxel type.
    typedef typename V::tVoxel tVoxel;

    //! Volume filter response type.
    typedef typename CPixelTraits<tVoxel>::tFilterResponse tFilterResponse;

public:
    //! Static point response normalization function.
    static tVoxel normalize(tFilterResponse Value)
    {
        static const tVoxel Zero = CPixelTraits<tVoxel>::getZero();

        return (tVoxel(Value) > Zero) ? tVoxel(Value) : Zero;
    }
};


//==============================================================================
/*!
 * Policy for volume filter response normalization.
 * - Specific normalization function can be set.
 * - Parameter V is a volume type.
 */
template <class V>
class VFN_FUNC
{
public:
    //! Check that type V is a volume. You will see name of this enum somewhere
    //! in compiler error message if the type V is not volume.
    enum { TEMPLATE_PARAMETER_IS_NOT_VOLUME = V::CLASS_VOLUME };

    //! Declare name of the class.
    MDS_STATIC_NAME(VFN_FUNC);

    //! Volume voxel type.
    typedef typename V::tVoxel tVoxel;

    //! Volume filter response type.
    typedef typename CPixelTraits<tVoxel>::tFilterResponse tFilterResponse;

    //! Type of the normalization function.
    typedef tVoxel (*tNormFunc)(tFilterResponse);

public:
    //! Default constructor.
    VFN_FUNC() : m_NormFunc(VFN_CONV<V>::normalize) {}

    //! Point response normalization function.
    tVoxel normalize(tFilterResponse Value)
    {
        return m_NormFunc(Value);
    }

    //! Sets the normalization function.
    void setNormFunc(tNormFunc Func)
    {
        MDS_ASSERT(Func);

        m_NormFunc = Func;
    }

protected:
    //! Pointer to the concrete normalization function.
    tNormFunc m_NormFunc;
};


} // namespace img
} // namespace mds

#endif // MDS_VOLUMEFILTERPOLICY_H

