//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsVolumeFilter.h                   \n
 * Section: libImage                            \n
 * Date:    2004/05/04                          \n
 *
 * $Id: mdsVolumeFilter.h 50 2006-08-08 13:05:53Z spanel $
 * 
 * Description:
 * - Volume filtering functions.
 */

#ifndef MDS_VOLUMEFILTER_H
#define MDS_VOLUMEFILTER_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsIdentifier.h>

#include "mdsImageTypes.h"
#include "mdsVolume.h"
#include "mdsVolumeFilterPolicy.h"


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Basic class for all volume filters.
 * - Parameter V is a volume type.
 * - Policy 'CNormPolicy' is used for filter response normalization.
 */
template <class V, template <typename> class CNormPolicy>
struct CVolumeFilterBase : public CNormPolicy<V>
{
    //! Check that type V is a volume. You will see name of this enum somewhere
    //! in compiler error message if the type V is not volume.
    enum { TEMPLATE_PARAMETER_IS_NOT_VOLUME = V::CLASS_VOLUME };

    //! Filter response normalization policy.
    typedef CNormPolicy<V> norm;

    //! Volume type.
    typedef typename V::tVolume tVolume;

    //! Volume voxel type.
    typedef typename V::tVoxel tVoxel;

    //! Volume filter response type.
    typedef typename CPixelTraits<tVoxel>::tFilterResponse tResult;
};


//==============================================================================
/*!
 * Basic template providing volume filtering functions. This template
 * is further specialized for concrete filter types.
 * - Parameter V is a volume type.
 * - 'Id' is a filter type.
 * - Policy N is used for filter response normalization.
 */
template <class V, typename Id, template <typename> class N = VFN_CONV>
class CVolumeFilter : public CVolumeFilterBase<V,N>
{
public:
    //! Volume filter base.
    typedef CVolumeFilterBase<V,N> base;
    typedef typename base::norm norm;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;

public:
    //! Filtering of input/source volume.
    //! - Returns false on failure.
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns filter response at specified volume position.
    //! - Value is not normalized!
    tResult getResponse(const V& SrcVolume, tSize x, tSize y, tSize z);
};


} // namespace img
} // namespace mds

#endif // MDS_VOLUMEFILTER_H

