//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsVolume.hxx                       \n
 * Section: libImage                            \n
 * Date:    2004/07/11                          \n
 *
 * $Id: mdsVolume.hxx 414 2007-07-03 14:06:55Z spanel $
 * 
 * Description:
 * - Volume template implementation.
 */


//=============================================================================
/*
 * Implementation of the mds::img::CVolume class.
 */

// Constructor
template <typename T>
CVolume<T>::CVolume(tSize XSize,
                    tSize YSize,
                    tSize ZSize,
                    tSize Margin
                    )
    : m_XSize(XSize)
    , m_YSize(YSize)
    , m_ZSize(ZSize)
    , m_Margin(Margin)
{
    // Helper value
    tSize Margins = 2 * m_Margin;

    // Data allocation
    m_YOffset = m_XSize + Margins;
    m_ZOffset = m_YOffset * (m_YSize + Margins);
    m_DataStorage.init(m_ZOffset * (m_ZSize + Margins));
    m_pData = m_DataStorage.getPtr((m_ZOffset + m_YOffset + 1) * m_Margin);
}


// Constructor
template <typename T>
CVolume<T>::CVolume(tSize XSize,
                    tSize YSize,
                    tSize ZSize,
                    const T& Value,
                    tSize Margin
                    )
    : m_XSize(XSize)
    , m_YSize(YSize)
    , m_ZSize(ZSize)
    , m_Margin(Margin)
{
    // Helper value
    tSize Margins = 2 * m_Margin;

    // Data allocation
    m_YOffset = m_XSize + Margins;
    m_ZOffset = m_YOffset * (m_YSize + Margins);
    m_DataStorage.init(m_ZOffset * (m_ZSize + Margins));
    m_pData = m_DataStorage.getPtr((m_ZOffset + m_YOffset + 1) * m_Margin);

    // Set voxel values
    fillEntire(Value);
}


// Constructor
template <typename T>
CVolume<T>::CVolume(const CVolume<T>& Volume,
                    tSize x,
                    tSize y,
                    tSize z,
                    tSize XSize,
                    tSize YSize,
                    tSize ZSize
                    )
{
    MDS_ASSERT(x >= 0 && XSize >= 0 && x < Volume.m_XSize
               && y >= 0 && YSize >= 0 && y < Volume.m_YSize
               && z >= 0 && ZSize >= 0 && z < Volume.m_ZSize);

    // Helper value
    tSize Margins = 2 * Volume.m_Margin;

    // Size of the volume data to copy
    m_XSize = MIN(x + XSize, Volume.m_XSize) - x;
    m_YSize = MIN(y + YSize, Volume.m_YSize) - y;
    m_ZSize = MIN(z + ZSize, Volume.m_ZSize) - z;

    // Data allocation
    m_Margin = Volume.m_Margin;
    m_YOffset = m_XSize + Margins;
    m_ZOffset = m_YOffset * (m_YSize + Margins);
    m_DataStorage.init(m_ZOffset * (m_ZSize + Margins));
    m_pData = m_DataStorage.getPtr((m_ZOffset + m_YOffset + 1) * m_Margin);

    // Size of the volume data to copy
    tSize XCount = m_XSize + Margins;
    tSize YCount = m_YSize + m_Margin;
    tSize ZCount = m_ZSize + m_Margin;

    // Copy the data
    for( tSize k = -m_Margin; k < ZCount; ++k )
    {
        for( tSize j = -m_Margin; j < YCount; ++j )
        {
            memCopy(getPtr(-m_Margin, j, k),
                    Volume.getPtr(x - m_Margin, y + j, z + k),
                    XCount
                   );
        }
    }
}


// Constructor
template <typename T>
CVolume<T>::CVolume(const CVolume<T>& Volume,
                    tSize x,
                    tSize y,
                    tSize z,
                    tSize XSize,
                    tSize YSize,
                    tSize ZSize,
                    EMakeRef MakeRef
                    )
    : m_Margin(Volume.m_Margin)
    , m_YOffset(Volume.m_YOffset)
    , m_ZOffset(Volume.m_ZOffset)
    , m_DataStorage(Volume.m_DataStorage, REFERENCE)
    , m_pData(Volume.m_pData)
{
    MDS_ASSERT(x >= 0 && XSize >= 0 && x < Volume.m_XSize
               && y >= 0 && YSize >= 0 && y < Volume.m_YSize
               && z >= 0 && ZSize >= 0 && z < Volume.m_ZSize);

    // Image size
    m_XSize = MIN(x + XSize, Volume.m_XSize) - x;
    m_YSize = MIN(y + YSize, Volume.m_YSize) - y;
    m_ZSize = MIN(z + ZSize, Volume.m_ZSize) - z;

    // Image data
    m_pData += z * m_ZOffset + y * m_YOffset + x;
}


// Copy constructor
template <typename T>
CVolume<T>::CVolume(const CVolume<T>& Volume)
    : m_XSize(Volume.m_XSize)
    , m_YSize(Volume.m_YSize)
    , m_ZSize(Volume.m_ZSize)
    , m_Margin(Volume.m_Margin)
{
    // Helper value
    tSize Margins = 2 * Volume.m_Margin;

    // Data allocation
    m_YOffset = m_XSize + Margins;
    m_ZOffset = m_YOffset * (m_YSize + Margins);
    m_DataStorage.init(m_ZOffset * (m_ZSize + Margins));
    m_pData = m_DataStorage.getPtr((m_ZOffset + m_YOffset + 1) * m_Margin);

    // Size of the volume data to copy
    tSize XCount = m_XSize + Margins;
    tSize YCount = m_YSize + m_Margin;
    tSize ZCount = m_ZSize + m_Margin;

    // Copy the data
    for( tSize k = -m_Margin; k < ZCount; ++k )
    {
        for( tSize j = -m_Margin; j < YCount; ++j )
        {
            memCopy(getPtr(-m_Margin, j, k),
                    Volume.getPtr(-m_Margin, j, k),
                    XCount
                   );
        }
    }
}


// Copy constructor
template <typename T>
CVolume<T>::CVolume(const CVolume<T>& Volume, EMakeRef MakeRef)
    : m_XSize(Volume.m_XSize)
    , m_YSize(Volume.m_YSize)
    , m_ZSize(Volume.m_ZSize)
    , m_Margin(Volume.m_Margin)
    , m_YOffset(Volume.m_YOffset)
    , m_ZOffset(Volume.m_ZOffset)
    , m_DataStorage(Volume.m_DataStorage, REFERENCE)
    , m_pData(Volume.m_pData)
{
}


// Destructor
template <typename T>
CVolume<T>::~CVolume()
{
}


// Creates a new volume
template <typename T>
bool CVolume<T>::create(tSize XSize,
                        tSize YSize,
                        tSize ZSize,
                        tSize Margin
                        )
{
    // Helper value
    tSize Margins = 2 * Margin;

    // Reallocate the data
    m_XSize = XSize;
    m_YSize = YSize;
    m_ZSize = ZSize;
    m_Margin = Margin;
    m_YOffset = m_XSize + Margins;
    m_ZOffset = m_YOffset * (m_YSize + Margins);
    m_DataStorage.resize(m_ZOffset * (m_ZSize + Margins));
    m_pData = m_DataStorage.getPtr((m_ZOffset + m_YOffset + 1) * m_Margin);

    // O.K.
    return true;
}


// Creates a new volume
template <typename T>
bool CVolume<T>::create(tSize XSize,
                        tSize YSize,
                        tSize ZSize,
                        const T& Value,
                        tSize Margin
                        )
{
    // Helper value
    tSize Margins = 2 * Margin;

    // Reallocate the data
    m_XSize = XSize;
    m_YSize = YSize;
    m_ZSize = ZSize;
    m_Margin = Margin;
    m_YOffset = m_XSize + Margins;
    m_ZOffset = m_YOffset * (m_YSize + Margins);
    m_DataStorage.resize(m_ZOffset * (m_ZSize + Margins));
    m_pData = m_DataStorage.getPtr((m_ZOffset + m_YOffset + 1) * m_Margin);

    // Fill the image
    fillEntire(Value);

    // O.K.
    return true;
}


// Creates a new volume
template <typename T>
bool CVolume<T>::create(const CVolume<T>& Volume,
                        tSize x,
                        tSize y,
                        tSize z,
                        tSize XSize,
                        tSize YSize,
                        tSize ZSize
                        )
{
    // Check the position
    MDS_CHECK(x >= 0 && XSize >= 0 && x < Volume.m_XSize
              && y >= 0 && YSize >= 0 && y < Volume.m_YSize
              && z >= 0 && ZSize >= 0 && z < Volume.m_ZSize, return false);

    // Helper value
    tSize Margins = 2 * Volume.m_Margin;

    // Size of the image
    m_XSize = MIN(x + XSize, Volume.m_XSize) - x;
    m_YSize = MIN(y + YSize, Volume.m_YSize) - y;
    m_ZSize = MIN(z + ZSize, Volume.m_ZSize) - z;

    // Reallocate the data
    m_Margin = Volume.Margin;
    m_YOffset = m_XSize + Margins;
    m_ZOffset = m_YOffset * (m_YSize + Margins);
    m_DataStorage.resize(m_ZOffset * (m_ZSize + Margins));
    m_pData = m_DataStorage.getPtr((m_ZOffset + m_YOffset + 1) * m_Margin);

    // Image size
    tSize XCount = m_XSize + Margins;
    tSize YCount = m_YSize + m_Margin;
    tSize ZCount = m_ZSize + m_Margin;

    // Copy the data
    for( tSize k = -m_Margin; k < ZCount; ++k )
    {
        for( tSize j = -m_Margin; j < YCount; ++j )
        {
            memCopy(getPtr(-m_Margin, j, k),
                    Volume.getPtr(x - m_Margin, y + j, z + k),
                    XCount
                   );
        }
    }

    // O.K.
    return true;
}


// Creates a new volume
template <typename T>
bool CVolume<T>::create(const CVolume<T>& Volume,
                        tSize x,
                        tSize y,
                        tSize z,
                        tSize XSize,
                        tSize YSize,
                        tSize ZSize,
                        EMakeRef MakeRef
                        )
{
    // Check the position
    MDS_CHECK(x >= 0 && XSize >= 0 && x < Volume.m_XSize
              && y >= 0 && YSize >= 0 && y < Volume.m_YSize
              && z >= 0 && ZSize >= 0 && z < Volume.m_ZSize, return false);

    // Image size
    m_XSize = MIN(x + XSize, Volume.m_XSize) - x;
    m_YSize = MIN(y + YSize, Volume.m_YSize) - y;
    m_ZSize = MIN(z + ZSize, Volume.m_ZSize) - z;

    // Create a reference to the image data
    m_Margin = Volume.m_Margin;
    m_YOffset = Volume.m_YOffset;
    m_ZOffset = Volume.m_ZOffset;
    m_DataStorage.create(Volume.m_DataStorage, REFERENCE);
    m_pData = Volume.m_pData + z * m_ZOffset + y * m_YOffset + x;

    // O.K.
    return true;
}


// Creates a new volume
template <typename T>
bool CVolume<T>::create(const CVolume<T>& Volume)
{
    // Helper value
    tSize Margins = 2 * Volume.m_Margin;

    // Reallocate the data
    m_XSize = Volume.m_XSize;
    m_YSize = Volume.m_YSize;
    m_ZSize = Volume.m_ZSize;
    m_Margin = Volume.m_Margin;
    m_YOffset = m_XSize + Margins;
    m_ZOffset = m_YOffset * (m_YSize + Margins);
    m_DataStorage.resize(m_ZOffset * (m_ZSize + Margins));
    m_pData = m_DataStorage.getPtr((m_ZOffset + m_YOffset + 1) * m_Margin);

    // Image size
    tSize XCount = Volume.m_XSize + Margins;
    tSize YCount = Volume.m_YSize + m_Margin;
    tSize ZCount = Volume.m_ZSize + m_Margin;

    // Copy the data
    for( tSize k = -m_Margin; k < ZCount; ++k )
    {
        for( tSize j = -m_Margin; j < YCount; ++j )
        {
            memCopy(getPtr(-m_Margin, j, k),
                    Volume.getPtr(-m_Margin, j, k),
                    XCount
                   );
        }
    }

    // O.K.
    return true;
}


// Creates a new volume
template <typename T>
bool CVolume<T>::create(const CVolume<T>& Volume, EMakeRef MakeRef)
{
    // Make a reference to the data
    m_XSize = Volume.m_XSize;
    m_YSize = Volume.m_YSize;
    m_ZSize = Volume.m_ZSize;
    m_Margin = Volume.m_Margin;
    m_YOffset = Volume.m_YOffset;
    m_ZOffset = Volume.m_ZOffset;
    m_DataStorage.create(Volume.m_DataStorage, REFERENCE);
    m_pData = Volume.m_pData;

    return true;
}


// Image conversion
template <typename T>
template <typename U>
bool CVolume<T>::convert(const CVolume<U>& Volume)
{
    // Helper value
    tSize Margins = 2 * Volume.getMargin();

    // Reallocate the data
    m_XSize = Volume.getXSize();
    m_YSize = Volume.getYSize();
    m_ZSize = Volume.getZSize();
    m_Margin = Volume.getMargin();
    m_YOffset = m_XSize + Margins;
    m_ZOffset = m_YOffset * (m_YSize + Margins);
    m_DataStorage.resize(m_ZOffset * (m_ZSize + Margins));
    m_pData = m_DataStorage.getPtr((m_ZOffset + m_YOffset + 1) * m_Margin);

    // Image size
    tSize XCount = m_XSize + m_Margin;
    tSize YCount = m_YSize + m_Margin;
    tSize ZCount = m_ZSize + m_Margin;

    // Convert voxels
    for( tSize k = -m_Margin; k < ZCount; ++k )
    {
        for( tSize j = -m_Margin; j < YCount; ++j )
        {
            for( tSize i = -m_Margin; i < XCount; ++i )
            {
                get(i, j, k) = CPixelConversion<U,T>::convert(Volume.get(i, j, k));
            }
        }
    }

    // O.K.
    return true;
}


// Assignment operator
template <typename T>
CVolume<T>& CVolume<T>::operator =(const CVolume<T>& Volume)
{
    if( &Volume == this )
    {
        return *this;
    }

    tSize XCount = MIN(m_XSize, Volume.m_XSize);
    tSize YCount = MIN(m_YSize, Volume.m_YSize);
    tSize ZCount = MIN(m_ZSize, Volume.m_ZSize);

    for( tSize k = 0; k < ZCount; ++k )
    {
        for( tSize j = 0; j < YCount; ++j )
        {
            memCopy(m_pData + k * m_ZOffset + j * m_YOffset,
                    Volume.m_pData + k * Volume.m_ZOffset + j * Volume.m_YOffset,
                    XCount
                   );
        }
    }

    return *this;
}


template <typename T>
inline CVolume<T>& CVolume<T>::operator +=(const CVolume<T>& Volume)
{
    tSize XCount = MIN(m_XSize, Volume.m_XSize);
    tSize YCount = MIN(m_YSize, Volume.m_YSize);
    tSize ZCount = MIN(m_ZSize, Volume.m_ZSize);

    for( tSize k = 0; k < ZCount; ++k )
    {
        for( tSize j = 0; j < YCount; ++j )
        {
            memVectAdd(m_pData + j * m_YOffset + k * m_ZOffset,
                       Volume.m_pData + j * Volume.m_YOffset + k * Volume.m_ZOffset,
                       XCount
                       );
        }
    }

    return *this;
}


template <typename T>
inline CVolume<T>& CVolume<T>::operator -=(const CVolume<T>& Volume)
{
    tSize XCount = MIN(m_XSize, Volume.m_XSize);
    tSize YCount = MIN(m_YSize, Volume.m_YSize);
    tSize ZCount = MIN(m_ZSize, Volume.m_ZSize);

    for( tSize k = 0; k < ZCount; ++k )
    {
        for( tSize j = 0; j < YCount; ++j )
        {
            memVectSub(m_pData + j * m_YOffset + k * m_ZOffset,
                       Volume.m_pData + j * Volume.m_YOffset + k * Volume.m_ZOffset,
                       XCount
                       );
        }
    }

    return *this;
}


template <typename T>
inline CVolume<T>& CVolume<T>::operator *=(const CVolume<T>& Volume)
{
    tSize XCount = MIN(m_XSize, Volume.m_XSize);
    tSize YCount = MIN(m_YSize, Volume.m_YSize);
    tSize ZCount = MIN(m_ZSize, Volume.m_ZSize);

    for( tSize k = 0; k < ZCount; ++k )
    {
        for( tSize j = 0; j < YCount; ++j )
        {
            memVectMult(m_pData + j * m_YOffset + k * m_ZOffset,
                        Volume.m_pData + j * Volume.m_YOffset + k * Volume.m_ZOffset,
                        XCount
                        );
        }
    }

    return *this;
}


template <typename T>
inline CVolume<T>& CVolume<T>::operator /=(const CVolume<T>& Volume)
{
    tSize XCount = MIN(m_XSize, Volume.m_XSize);
    tSize YCount = MIN(m_YSize, Volume.m_YSize);
    tSize ZCount = MIN(m_ZSize, Volume.m_ZSize);

    for( tSize k = 0; k < ZCount; ++k )
    {
        for( tSize j = 0; j < YCount; ++j )
        {
            memVectDiv(m_pData + j * m_YOffset + k * m_ZOffset,
                       Volume.m_pData + j * Volume.m_YOffset + k * Volume.m_ZOffset,
                       XCount
                       );
        }
    }

    return *this;
}


template <typename T>
template <typename U>
inline CVolume<T>& CVolume<T>::operator +=(const U& c)
{
    for( tSize k = 0; k < m_ZSize; ++k )
    {
        for( tSize j = 0; j < m_YSize; ++j )
        {
            memAdd(m_pData + j * m_YOffset + k * m_ZOffset, c, m_XSize);
        }
    }

    return *this;
}


template <typename T>
template <typename U>
inline CVolume<T>& CVolume<T>::operator -=(const U& c)
{
    for( tSize k = 0; k < m_ZSize; ++k )
    {
        for( tSize j = 0; j < m_YSize; ++j )
        {
            memSub(m_pData + j * m_YOffset + k * m_ZOffset, c, m_XSize);
        }
    }

    return *this;
}


template <typename T>
template <typename U>
inline CVolume<T>& CVolume<T>::operator *=(const U& c)
{
    for( tSize k = 0; k < m_ZSize; ++k )
    {
        for( tSize j = 0; j < m_YSize; ++j )
        {
            memMult(m_pData + j * m_YOffset + k * m_ZOffset, c, m_XSize);
        }
    }

    return *this;
}


template <typename T>
template <typename U>
inline CVolume<T>& CVolume<T>::operator /=(const U& c)
{
    for( tSize k = 0; k < m_ZSize; ++k )
    {
        for( tSize j = 0; j < m_YSize; ++j )
        {
            memDiv(m_pData + j * m_YOffset + k * m_ZOffset, c, m_XSize);
        }
    }

    return *this;
}


template <typename T>
inline void CVolume<T>::setRow(tSize y, tSize z, const T *pSrc)
{
    memCopy(m_pData + y * m_YOffset + z * m_ZOffset, pSrc, m_XSize);
}


template <typename T>
inline void CVolume<T>::copyRow(tSize y, tSize z, T *pDst) const
{
    memCopy(pDst, m_pData + y * m_YOffset + z * m_ZOffset, m_XSize);
}


template <typename T>
inline void CVolume<T>::copyWindow(tSize x,
                                   tSize y,
                                   tSize z,
                                   tSize Size,
                                   T *pDst
                                   ) const
{
    const tSize Half = Size / 2;
    const tSize YOffset = m_YOffset - Size;
    const tSize ZOffset = m_ZOffset - Size * m_YOffset;

    T *p = m_pData + (z - Half) * m_ZOffset + (y - Half) * m_YOffset + x - Half;
    for( tSize k = 0; k < Size; ++k )
    {
        for( tSize j = 0; j < Size; ++j )
        {
            T *pMax = p + Size;
            while( p < pMax )
            {
                *(pDst++) = *(p++);
            }
            p += YOffset;
        }
        p += ZOffset;
    }
}


template <typename T>
inline void CVolume<T>::fill(const T& c)
{
    for( tSize k = 0; k < m_ZSize; k++ )
    {
        for( tSize j = 0; j < m_YSize; j++ )
        {
            memSet(m_pData + j * m_YOffset + k * m_ZOffset, c, m_XSize);
        }
    }
}


template <typename T>
inline void CVolume<T>::fillEntire(const T& c)
{
    memSet(m_DataStorage.getPtr(), c, m_DataStorage.getSize());
}


template <typename T>
inline void CVolume<T>::fillMargin(const T& c)
{
    // Fill front and back margin
    tSize Count = m_Margin * m_ZOffset;
    memSet(getPtr(-m_Margin, -m_Margin, -m_Margin), c, Count);
    memSet(getPtr(-m_Margin, -m_Margin, m_ZSize), c, Count);

    // Fill top and bottom margin
    Count = m_Margin * m_YOffset;
    for( tSize k = 0; k < m_ZSize; ++k )
    {
        memSet(getPtr(-m_Margin, -m_Margin, k), c, Count);
        memSet(getPtr(-m_Margin, m_YSize, k), c, Count);
    }

    // Fill left and right margin
    for( tSize k = 0; k < m_ZSize; ++k )
    {
        for( tSize j = 0; j < m_YSize; ++j )
        {
            memSet(getPtr(-m_Margin, j, k), c, m_Margin);
            memSet(getPtr(m_XSize, j, k), c, m_Margin);
        }
    }
}


template <typename T>
inline void CVolume<T>::mirrorMargin()
{
    for( tSize k = -m_Margin; k < m_ZSize + m_Margin; ++k )
    {
        tSize kk = k;
        if( k < 0 )
        {
            kk = -k;
        }
        else if ( k >= m_ZSize )
        {
            kk = 2 * m_ZSize - 2 - k;
        }
        for( tSize j = -m_Margin; j < m_YSize + m_Margin; ++j )
        {
            tSize jj = j;
            if( j < 0 )
            {
                jj = -j;
            }
            else if ( j >= m_YSize )
            {
                jj = 2 * m_YSize - 2 - j;
            }
            tSize i;
            for( i = 1; i <= m_Margin; ++i )
            {
                get(-i, j, k) = get(i, jj, kk);
                get(m_XSize - 1 + i, j, k) = get(m_XSize - 1 - i, jj, kk);
            }
            if( j < 0 || j >= m_YSize || k < 0 || k >= m_ZSize )
            {
                for( i = 0; i < m_XSize; ++i )
                {
                    get(i, j, k) = get(i, jj, kk);
                }
            }
        }
    }
}


template <typename T>
inline void CVolume<T>::replace(const T& Value, const T& NewValue)
{
    for( tSize k = 0; k < m_ZSize; ++k )
    {
        for( tSize j = 0; j < m_YSize; ++j )
        {
            T *p = m_pData + j * m_YOffset + k * m_ZOffset;
            T *pMax = p + m_XSize;
            while( p < pMax )
            {
                if( *p == Value )
                {
                    *p = NewValue;
                }
                ++p;
            }
        }
    }
}


template <typename T>
inline void CVolume<T>::abs()
{
    for( tSize k = 0; k < m_ZSize; ++k )
    {
        for( tSize j = 0; j < m_YSize; ++j )
        {
            T *p = m_pData + j * m_YOffset + k * m_ZOffset;
            T *pMax = p + m_XSize;
            while( p < pMax )
            {
                if( !(*p > T(0)) )
                {
                    *p = -(*p);
                }
                ++p;
            }
        }
    }
}


template <typename T>
inline void CVolume<T>::clip(const T& Lower, const T& Upper)
{
    for( tSize k = 0; k < m_ZSize; ++k )
    {
        for( tSize j = 0; j < m_YSize; ++j )
        {
            T *p = m_pData + j * m_YOffset + k * m_ZOffset;
            T *pMax = p + m_XSize;
            while( p < pMax )
            {
                if( *p < Lower )
                {
                    *p = Lower;
                }
                else if( *p > Upper )
                {
                    *p = Upper;
                }
                ++p;
            }
        }
    }
}


template <typename T>
inline void CVolume<T>::cut(const T& Lower, const T& Upper)
{
    for( tSize k = 0; k < m_ZSize; ++k )
    {
        for( tSize j = 0; j < m_YSize; ++j )
        {
            T *p = m_pData + j * m_YOffset + k * m_ZOffset;
            T *pMax = p + m_XSize;
            while( p < pMax )
            {
                if( *p < Lower )
                {
                    *p = T(0);
                }
                else if( *p > Upper )
                {
                    *p = T(0);
                }
                ++p;
            }
        }
    }
}


template <typename T>
inline void CVolume<T>::subSample(const CVolume<T>& Image,
                                  tSize l,
                                  tSize m,
                                  tSize n
                                  )
{
    MDS_CHECK(l > 0 && m > 0 && n > 0, return);

    tSize XCount = MIN(Image.m_XSize / l, m_XSize);
    tSize YCount = MIN(Image.m_YSize / m, m_YSize);
    tSize ZCount = MIN(Image.m_ZSize / n, m_ZSize);

    for( tSize k = 0; k < ZCount; ++k )
    {
        for( tSize j = 0; j < YCount; ++j )
        {
            T *pI = Image.m_pData + m * j * Image.m_YOffset + n * k * Image.m_ZOffset;
            T *p = m_pData + j * m_YOffset + k * m_ZOffset;
            T *pMax = p + XCount;
            while( p < pMax )
            {
                *(p++) = *pI;
                pI += l;
            }
        }
    }
}


template <typename T>
inline T CVolume<T>::interpolate(const CPoint3D& Point) const
{
    int iX = (int)Point.x;
    int iY = (int)Point.y;
    int iZ = (int)Point.z;

    double dX = Point.x - (double)iX;
    double dY = Point.y - (double)iY;
    double dZ = Point.z - (double)iZ;
    double dInvX = 1.0 - dX;
    double dInvY = 1.0 - dY;
    double dInvZ = 1.0 - dZ;

    T *p = m_pData + iZ * m_ZOffset + iY * m_YOffset + iX;

    // Interpolate voxel value
    double dValue = dInvX * dInvY * dInvZ * (double)*(p);
    dValue += dX * dInvY * dInvZ * (double)*(p + 1);
    dValue += dInvX * dY * dInvZ * (double)*(p + m_YOffset);
    dValue += dX * dY * dInvZ * (double)*(p + m_YOffset + 1);
    dValue += dInvX * dInvY * dZ * (double)*(p + m_ZOffset);
    dValue += dX * dInvY * dZ * (double)*(p + m_ZOffset + 1);
    dValue += dInvX * dY * dZ * (double)*(p + m_ZOffset + m_YOffset);
    dValue += dX * dY * dZ * (double)*(p + m_ZOffset + m_YOffset + 1);

    // Return the interpolated value
    return (tVoxel)dValue;
}


template <typename T>
inline T CVolume<T>::color2Voxel(CColor Color) const
{
    // Direct conversion of RGB color to the RGB pixel
    tRGBPixel Pixel = tRGBPixel::tRGB(Color);

    // Return the pixel value
    return CPixelConversion<tRGBPixel,T>::convert(Pixel);
}


template <typename T>
inline bool CVolume<T>::checkPosition(tSize x, tSize y, tSize z) const
{
    return (x >= 0 && x < m_XSize
            && y >= 0 && y < m_YSize
            && z >= 0 && z < m_ZSize
           );
}


template <typename T>
inline bool CVolume<T>::getPlaneXY(tSize z, CImage<T>& Plane)
{
    MDS_CHECK(z >= 0 && z < m_ZSize, return false);

    tSize XCount = MIN(m_XSize, Plane.getXSize());
    tSize YCount = MIN(m_YSize, Plane.getYSize());

    tSize Offset = m_YOffset - XCount;
    tSize OffsetP = Plane.getYOffset() - XCount;

    T *p = m_pData + z * m_ZOffset;
    T *pP = Plane.getPtr();
    for( tSize j = 0; j < YCount; ++j )
    {
        T *pMax = p + XCount;
        while( p < pMax )
        {
            *(pP++) = *(p++);
        }
        p += Offset;
        pP += OffsetP;
    }

    return true;
}


template <typename T>
inline bool CVolume<T>::getPlaneXZ(tSize y, CImage<T>& Plane)
{
    MDS_CHECK(y >= 0 && y < m_YSize, return false);

    tSize XCount = MIN(m_XSize, Plane.getXSize());
    tSize ZCount = MIN(m_ZSize, Plane.getYSize());

    tSize Offset = m_ZOffset - XCount;
    tSize OffsetP = Plane.getYOffset() - XCount;

    T *p = m_pData + y * m_YOffset;
    T *pP = Plane.getPtr();
    for( tSize j = 0; j < ZCount; ++j )
    {
        T *pMax = p + XCount;
        while( p < pMax )
        {
            *(pP++) = *(p++);
        }
        p += Offset;
        pP += OffsetP;
    }

    return true;
}


template <typename T>
inline bool CVolume<T>::getPlaneYZ(tSize x, CImage<T>& Plane)
{
    MDS_CHECK(x >= 0 && x < m_XSize, return false);

    tSize YCount = MIN(m_YSize, Plane.getXSize());
    tSize ZCount = MIN(m_ZSize, Plane.getYSize());

    tSize Offset = m_ZOffset - YCount * m_YOffset;
    tSize OffsetP = Plane.getYOffset() - YCount;

    T *p = m_pData + x;
    T *pP = Plane.getPtr();
    for( tSize j = 0; j < ZCount; ++j )
    {
        T *pMax = p + YCount * m_YOffset;
        while( p < pMax )
        {
            *(pP++) = *(p);
            p += m_YOffset;
        }
        p += Offset;
        pP += OffsetP;
    }

    return true;
}


template <typename T>
inline bool CVolume<T>::setPlaneXY(tSize z, const CImage<T>& Plane)
{
    MDS_CHECK(z >= 0 && z < m_ZSize, return false);

    tSize XCount = MIN(m_XSize, Plane.getXSize());
    tSize YCount = MIN(m_YSize, Plane.getYSize());

    tSize Offset = m_YOffset - XCount;
    tSize OffsetP = Plane.getYOffset() - XCount;

    T *p = m_pData + z * m_ZOffset;
    const T *pP = Plane.getPtr();
    for( tSize j = 0; j < YCount; ++j )
    {
        T *pMax = p + XCount;
        while( p < pMax )
        {
            *(p++) = *(pP++);
        }
        p += Offset;
        pP += OffsetP;
    }

    return true;
}


template <typename T>
inline bool CVolume<T>::setPlaneXZ(tSize y, const CImage<T>& Plane)
{
    MDS_CHECK(y >= 0 && y < m_YSize, return false);

    tSize XCount = MIN(m_XSize, Plane.getXSize());
    tSize ZCount = MIN(m_ZSize, Plane.getYSize());

    tSize Offset = m_ZOffset - XCount;
    tSize OffsetP = Plane.getYOffset() - XCount;

    T *p = m_pData + y * m_YOffset;
    const T *pP = Plane.getPtr();
    for( tSize j = 0; j < ZCount; ++j )
    {
        T *pMax = p + XCount;
        while( p < pMax )
        {
            *(p++) = *(pP++);
        }
        p += Offset;
        pP += OffsetP;
    }

    return true;
}


template <typename T>
inline bool CVolume<T>::setPlaneYZ(tSize x, const CImage<T>& Plane)
{
    MDS_CHECK(x >= 0 && x < m_XSize, return false);

    tSize YCount = MIN(m_YSize, Plane.getXSize());
    tSize ZCount = MIN(m_ZSize, Plane.getYSize());

    tSize Offset = m_ZOffset - YCount * m_YOffset;
    tSize OffsetP = Plane.getYOffset() - YCount;

    T *p = m_pData + x;
    const T *pP = Plane.getPtr();
    for( tSize j = 0; j < ZCount; ++j )
    {
        T *pMax = p + YCount * m_YOffset;
        while( p < pMax )
        {
            *(p) = *(pP++);
            p += m_YOffset;
        }
        p += Offset;
        pP += OffsetP;
    }

    return true;
}


//==============================================================================
/*
 * - Volume data serialization functions.
 */

template <typename T>
void CVolume<T>::serialize(mds::mod::CChannelSerializer& Writer)
{
    // Begin of data serialization block
    MDS_DE_SERIALIZE_BEGIN;

        // Check if serializer supports shared memory
        if( Writer.testFlag(mds::mod::CSHMSerializer::SHM_FLAG) )
        {
            // Use SHM serializer
            mds::mod::CSHMSerializer& SHMWriter = static_cast<mds::mod::CSHMSerializer&>(Writer);
            
            // Volume info
            SHMWriter.writeSize(m_XSize);
            SHMWriter.writeSize(m_YSize);
            SHMWriter.writeSize(m_ZSize);
            SHMWriter.writeSize(m_Margin);
            SHMWriter.writeSize(m_YOffset);
            SHMWriter.writeSize(m_ZOffset);
            SHMWriter.writeData(m_DataStorage);            
        }
        else
        {
            // Volume info
            Writer.writeSize(m_XSize);
            Writer.writeSize(m_YSize);
            Writer.writeSize(m_ZSize);
           
            // Volume data
            for( tSize k = 0; k < m_ZSize; ++k )
            {
                for( tSize j = 0; j < m_YSize; ++j )
                {
                    Writer.writeBytes(m_pData + k * m_ZOffset + j * m_YOffset,
                                      m_XSize * CTypeTraits<tVoxel>::getSize()
                                      );
                }
            }
        }
    
    // End of the block
    MDS_DE_SERIALIZE_END;
}


template <typename T>
void CVolume<T>::deserialize(mds::mod::CChannelSerializer& Reader)
{
    // Begin of data deserialization block
    MDS_DE_DESERIALIZE_BEGIN;

        // Check if serializer supports shared memory
        if( Reader.testFlag(mds::mod::CSHMSerializer::SHM_FLAG) )
        {
            // Use SHM serializer
            mds::mod::CSHMSerializer& SHMReader = static_cast<mds::mod::CSHMSerializer&>(Reader);
            
            // Volume info
            SHMReader.readSize(m_XSize);
            SHMReader.readSize(m_YSize);
            SHMReader.readSize(m_ZSize);
            SHMReader.readSize(m_Margin);
            SHMReader.readSize(m_YOffset);
            SHMReader.readSize(m_ZOffset);
            SHMReader.readData(m_DataStorage);
            if( SHMReader.isError() )
            {
                return;
            }
            
            // Check volume size
            tSize Size = m_ZOffset * (m_ZSize + 2 * m_Margin);
            if( Size < 0 || Size > m_DataStorage.getSize() )
            {
                SHMReader.setError();
                return;
            }
                       
            // Initialize the internal pointer            
            m_pData = m_DataStorage.getPtr((m_ZOffset + m_YOffset + 1) * m_Margin);
        }
        else
        {
            // Read the volume info struct
            Reader.readSize(m_XSize);
            Reader.readSize(m_YSize);
            Reader.readSize(m_ZSize);
        
            // Create a new volume
            if( !create(m_XSize, m_YSize, m_ZSize, m_Margin) )
            {
                Reader.setError();
                return;
            }
        
            // Volume data
            for( tSize k = 0; k < m_ZSize; ++k )
            {
                for( tSize j = 0; j < m_YSize; ++j )
                {
                    Reader.readBytes(m_pData + k * m_ZOffset + j * m_YOffset,
                                     m_XSize * CTypeTraits<tVoxel>::getSize()
                                     );
                }
            }
        }
    
    // End of the block
    MDS_DE_DESERIALIZE_END;
}

