//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsVolume.h                         \n
 * Section: libImage                            \n
 * Date:    2004/07/11                          \n
 *
 * $Id: mdsVolume.h 387 2007-06-22 09:33:41Z spanel $
 * 
 * Description:
 * - Template providing basic volume operations.
 */

#ifndef MDS_VOLUME_H
#define MDS_VOLUME_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsMemory.h>
//#include <MDSTk/Base/mdsRefData.h>
#include <MDSTk/Base/mdsSHMData.h>
#include <MDSTk/Base/mdsSharedPtr.h>
#include <MDSTk/Math/mdsBase.h>
//#include <MDSTk/Module/mdsSerializer.h>
#include <MDSTk/Module/mdsSHMSerializer.h>

#include "mdsVolumeBase.h"
#include "mdsVolumeIterator.h"
#include "mdsPoint.h"
#include "mdsPixelTraits.h"
#include "mdsPixelConversions.h"
#include "mdsColor.h"
#include "mdsImage.h"

#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Template volume class providing basic volume operations.
 * - Parameter T is a voxel type.
 */
template <typename T>
class CVolume
    : public mds::CObject
    , public CVolumeBase<CVolume<T> >
    , public mds::mod::CSerializable
{
public:
    //! Templates that require members of the CVolume class can use
    //! this enum to check the existence.
    enum { CLASS_VOLUME };

    //! Standard method getClassName().
    MDS_CLASS_NAME(CVolume);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CVolume);

    //! Voxel type.
    typedef T tVoxel;

    //! Volume data storage type.
//    typedef mds::base::CRefData<T> tDataStorage;
    typedef mds::base::CSHMData<T> tDataStorage;

    //! Iterator type.
    //! - Declares type tIterator.
    MDS_COMPLEX_ITERATOR(CVolume, tVoxel, CVolumeIterator);

    //! Standard method getEntityName().
    MDS_ENTITY_TEMPLATE_NAME(Volume, CPixelTraits<T>::getPixelName());

    //! Standard method getEntityCompression().
    MDS_ENTITY_COMPRESSION(mds::mod::CC_RLE16);

public:
    //! Constructor.
    //! - 'Min' and 'Max' parameters specify the range of allowed
    //!   voxel values.
    CVolume(tSize XSize = 0,
            tSize YSize = 0,
            tSize ZSize = 0,
            tSize Margin = 0
            );

    //! Constructor.
    //! - A given parameter value is the voxel initial value.
    //! - 'Min' and 'Max' parameters specify the range of allowed
    //!   voxel values.
    CVolume(tSize XSize,
            tSize YSize,
            tSize ZSize,
            const T& Value,
            tSize Margin = 0
            );

    //! Constructor.
    //! - Creates a subvolume of a given volume.
    CVolume(const CVolume& Volume,
            tSize x,
            tSize y,
            tSize z,
            tSize XSize,
            tSize YSize,
            tSize ZSize
            );

    //! Constructor.
    //! - Creates a subvolume of a given volume.
    //! - Makes only a reference to the data.
    CVolume(const CVolume& Volume,
            tSize x,
            tSize y,
            tSize z,
            tSize XSize,
            tSize YSize,
            tSize ZSize,
            EMakeRef
            );

    //! Copy constructor.
    //! - Makes a new copy of the data.
    CVolume(const CVolume& Volume);

    //! Copy constructor.
    //! - Only reference to the data of the given volume.
    CVolume(const CVolume& Volume, EMakeRef);

    //! Destructor.
    virtual ~CVolume();

    //! Creates a new volume.
    bool create(tSize XSize,
                tSize YSize,
                tSize ZSize,
                tSize Margin = 0
                );

    //! Creates a new volume.
    bool create(tSize XSize,
                tSize YSize,
                tSize ZSize,
                const T& Value,
                tSize Margin = 0
                );

    //! Creates a subvolume of a given volume.
    bool create(const CVolume& Volume,
                tSize x,
                tSize y,
                tSize z,
                tSize XSize,
                tSize YSize,
                tSize ZSize
                );

    //! Creates a subvolume of a given volume.
    //! - Makes only a reference to the data.
    bool create(const CVolume& Volume,
                tSize x,
                tSize y,
                tSize z,
                tSize XSize,
                tSize YSize,
                tSize ZSize,
                EMakeRef
                );

    //! Creates a new volume.
    //! - Makes a new copy of the data.
    bool create(const CVolume& Volume);

    //! Creates a new volume.
    //! - Makes only reference to the data of the given volume.
    bool create(const CVolume& Volume, EMakeRef);

    //! Assignment operator.
    virtual CVolume& operator =(const CVolume& Volume);

    //! Conversion of volumes of different types.
    template <typename U>
    bool convert(const CVolume<U>& Volume);


    //! Returns the volume size (dimensions).
    tSize getXSize() const { return m_XSize; }
    tSize getYSize() const { return m_YSize; }
    tSize getZSize() const { return m_ZSize; }

    //! Returns offset between two neigbouring voxels in x-axis.
    tSize getXOffset() const { return 1; }

    //! Returns offset between two neigbouring voxels in y-axis.
    tSize getYOffset() const { return m_YOffset; }

    //! Returns offset between two neigbouring voxels in z-axis.
    tSize getZOffset() const { return m_ZOffset; }

    //! Returns the volume Margin size in voxels.
    tSize getMargin() const { return m_Margin; }

    //! Returns the subscripted voxel [x][y][z].
    T& operator ()(tSize x, tSize y, tSize z)
    {
        return *(m_pData + z * m_ZOffset + y * m_YOffset + x);
    }
    const T& operator ()(tSize x, tSize y, tSize z) const
    {
        return *(m_pData + z * m_ZOffset + y * m_YOffset + x);
    }

    //! Gets the voxel [x][y][z].
    T& get(tSize x, tSize y, tSize z)
    {
        return *(m_pData + z * m_ZOffset + y * m_YOffset + x);
    }
    const T& get(tSize x, tSize y, tSize z) const
    {
        return *(m_pData + z * m_ZOffset + y * m_YOffset + x);
    }

    //! Returns pointer to the given voxel.
    T *getPtr(tSize x, tSize y, tSize z)
    {
        return (m_pData + z * m_ZOffset + y * m_YOffset + x);
    }
    const T *getPtr(tSize x, tSize y, tSize z) const
    {
        return (m_pData + z * m_ZOffset + y * m_YOffset + x);
    }

    //! Returns pointer to the first element of the volume row.
    //! - z and x coordinate are taken to be zero.
    T *getRowPtr(tSize y, tSize z)
    {
        return (m_pData + y * m_YOffset + z * m_ZOffset);
    }
    const T *getRowPtr(tSize y, tSize z) const
    {
        return (m_pData + y * m_YOffset + z * m_ZOffset);
    }

    //! Returns pointer to the volume data.
    T *getPtr() { return m_pData; }
    const T *getPtr() const { return m_pData; }

    //! Sets the pixel at the position [x][y].
    void set(tSize x, tSize y, tSize z, const T& Value)
    {
        *(m_pData + z * m_ZOffset + y * m_YOffset + x) = Value;
    }


    //! Returns iterator that points at the first volume voxel.
    tIterator getBegin()
    {
        return tIterator(*this, 0, 0, 0);
    }
    tConstIterator getBegin() const
    {
        return tConstIterator(*this, 0, 0, 0);
    }

    //! Returns iterator that points beyond the last volume voxel.
    tIterator getEnd()
    {
        return tIterator(*this, 0, 0, m_ZSize);
    }
    tConstIterator getEnd() const
    {
        return tConstIterator(*this, 0, 0, m_ZSize);
    }

    //! Returns iterator which points at a specified voxel.
    tIterator getIterator(tSize x, tSize y, tSize z)
    {
        return tIterator(*this, x, y, z);
    }
    tConstIterator getIterator(tSize x, tSize y, tSize z) const
    {
        return tConstIterator(*this, x, y, z);
    }


    //! Miscellaneous combined assignment operators.
    inline CVolume& operator +=(const CVolume& Volume);
    inline CVolume& operator -=(const CVolume& Volume);
    inline CVolume& operator *=(const CVolume& Volume);
    inline CVolume& operator /=(const CVolume& Volume);

    //! Combined assignment operator.
    //! - Operator T& operator +=(const U&) must be defined for the pixel type T!
    template <typename U>
    inline CVolume& operator +=(const U& c);

    //! Combined assignment operator.
    //! - Operator T& operator -=(const U&) must be defined for the pixel type T!
    template <typename U>
    inline CVolume& operator -=(const U& c);

    //! Combined assignment operator.
    //! - Operator T& operator *=(const U&) must be defined for the pixel type T!
    template <typename U>
    inline CVolume& operator *=(const U& c);

    //! Combined assignment operator.
    //! - Operator T& operator /=(const U&) must be defined for the pixel type T!
    template <typename U>
    inline CVolume& operator /=(const U& c);


    //! Fills the y,z-th volume row using a given input buffer.
    inline void setRow(tSize y, tSize z, const T *pSrc);

    //! Copies they y,z-th volume row to the output buffer.
    inline void copyRow(tSize y, tSize z, T *pDst) const;

    //! Copies the sub-volume of side 'Size' to the output buffer.
    inline void copyWindow(tSize x,
                           tSize y,
                           tSize z,
                           tSize Size,
                           T *pDst
                           ) const;

    //! Fills the volume using a given voxel value.
    inline void fill(const T& c);

    //! Fills the volume including its margin using a given voxel value.
    inline void fillEntire(const T& c);

    //! Pads the volume margin with constant value.
    inline void fillMargin(const T& c);

    //! Pads the volume margin using a simple mirroring.
    inline void mirrorMargin();

    //! Replaces all voxels of a given value by a specified value.
    inline void replace(const T& Value, const T& NewValue);

    //! Absolute value of all voxels.
    inline void abs();

    //! Clip range of pixel values.
    inline void clip(const T& Lower, const T& Upper);

    //! Cuts range of pixel values.
    inline void cut(const T& Lower, const T& Upper);

    //! Subsample the volume.
    inline void subSample(const CVolume& Volume,
                          tSize l = 2,
                          tSize m = 2,
                          tSize n = 2
                          );

    //! Bilinear subvoxel value interpolation.
    inline T interpolate(const CPoint3D& Point) const;

    //! Converts a specified color to the volume voxel format and range.
    inline T color2Voxel(CColor Color) const;

    //! Checks the voxel position.
    //! - Returns true if a specified position is satisfactory.
    inline bool checkPosition(tSize x, tSize y, tSize z) const;


    //! Cuts the volume using plane XY and saves obtained data to a given
    //! image. Position of the plane is specified by the z coordinate.
    inline bool getPlaneXY(tSize z, CImage<T>& Plane);

    //! Cuts the volume using plane XZ and saves obtained data to a given
    //! image. Position of the plane is specified by the y coordinate.
    inline bool getPlaneXZ(tSize y, CImage<T>& Plane);

    //! Cuts the volume using plane YZ and saves obtained data to a given
    //! image. Position of the plane is specified by the x coordinate.
    inline bool getPlaneYZ(tSize x, CImage<T>& Plane);

    //! Projects a given image data to the volume using a specified plane.
    inline bool setPlaneXY(tSize z, const CImage<T>& Plane);

    //! Projects a given image data to the volume using a specified plane.
    inline bool setPlaneXZ(tSize y, const CImage<T>& Plane);

    //! Projects a given image data to the volume using a specified plane.
    inline bool setPlaneYZ(tSize x, const CImage<T>& Plane);


    //! Serializes all class members.
    void serialize(mds::mod::CChannelSerializer& Writer);

    //! Deserializes all class members.
    void deserialize(mds::mod::CChannelSerializer& Reader);

protected:
    //! Volume dimensions.
    tSize m_XSize, m_YSize, m_ZSize;

    //! Volume margin size.
    tSize m_Margin;

    //! Offsets used by subscripting functions.
    tSize m_YOffset, m_ZOffset;

    //! Volume data storage.
    tDataStorage m_DataStorage;

    //! Pointer to the volume data.
    T *m_pData;
};


//=============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsVolume.hxx"


//=============================================================================
/*
 * Basic template instances and type definitions.
 */

//! Grayscale volume, 8-bits per pixel
typedef CVolume<tPixel8>        CVolume8;

//! Grayscale volume, 16-bits per pixel
typedef CVolume<tPixel16>       CVolume16;

//! Float volume
typedef CVolume<tFloatPixel>    CFVolume;

//! Density volume
typedef CVolume<tDensityPixel>  CDVolume;

//! Color RGB volume
typedef CVolume<tRGBPixel>      CRGBVolume;


//=============================================================================
/*
 * Basic template instances and type definitions.
 * - Using smart pointers.
 */

//! Grayscale volume, 8-bit per pixel
typedef CVolume8::tSmartPtr     CVolume8Ptr;

//! Grayscale volume, 16-bit per pixel
typedef CVolume16::tSmartPtr    CVolume16Ptr;

//! Float volume
typedef CFVolume::tSmartPtr     CFVolumePtr;

//! Density volume
typedef CDVolume::tSmartPtr     CDVolumePtr;

//! Color RGB volume
typedef CRGBVolume::tSmartPtr   CRGBVolumePtr;


} // namespace img
} // namespace mds

#endif // MDS_VOLUME_H

