//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsVector.h                         \n
 * Section: libImage                            \n
 * Date:    2005/06/29                          \n
 *
 * $Id: mdsVector.h 277 2007-03-08 10:12:10Z spanel $
 *
 * Description:
 * - Vector in 3D space.
 */

#ifndef MDS_VECTOR3D_H
#define MDS_VECTOR3D_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsSmallObject.h>
#include <MDSTk/Math/mdsBase.h>

#include "mdsImageTypes.h"
#include "mdsPoint.h"

#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Class representing vector in 3D space.
 */
class CVector3D : public mds::base::CSmallValueObject<>
{
public:
    //! Vector coordinates.
    tCoordinate x, y, z;

public:
    //! Default constructor.
//    CVector3D() : x(0), y(0), z(0) {}
    CVector3D() {}

    //! Constructor initializes vector.
    CVector3D(tCoordinate px, tCoordinate py, tCoordinate pz = 0)
        : x(px)
        , y(py)
        , z(pz)
    {}

    //! Constructor makes vector from two given points.
    CVector3D(const CPoint3D& p0, const CPoint3D& p1)
        : x(p1.x - p0.x)
        , y(p1.y - p0.y)
        , z(p1.z - p0.z)
    {}

    //! Copy constructor.
    CVector3D(const CVector3D& p) : x(p.x), y(p.y), z(p.z) {}

    //! Destructor.
    ~CVector3D() {}

    //! Assignment operator.
    CVector3D& operator =(const CVector3D& p)
    {
        x = p.x;
        y = p.y;
        z = p.z;
        return *this;
    }

    //! Creates the vector from two given points.
    void create(const CPoint3D& p0, const CPoint3D& p1)
    {
        x = p1.x - p0.x;
        y = p1.y - p0.y;
        z = p1.z - p0.z;
    }


    //! Returns vector coordinate.
    tCoordinate getX() const { return x; }
    tCoordinate getY() const { return y; }
    tCoordinate getZ() const { return z; }

    //! Returns XY vector coordinates.
    void getXY(tCoordinate& px, tCoordinate& py) const
    {
        px = x;
        py = y;
    }

    //! Returns all vector coordinates.
    void getXYZ(tCoordinate& px, tCoordinate& py, tCoordinate& pz) const
    {
        px = x;
        py = y;
        pz = z;
    }

    //! Sets vector coordinate.
    void setX(tCoordinate px) { x = px; }
    void setY(tCoordinate py) { y = py; }
    void setZ(tCoordinate pz) { z = pz; }

    //! Sets XY vector coordinates.
    void setXY(tCoordinate px, tCoordinate py)
    {
        x = px;
        y = py;
    }

    //! Sets all vector coordinates.
    void setXYZ(tCoordinate px, tCoordinate py, tCoordinate pz)
    {
        x = px;
        y = py;
        z = pz;
    }


    //! Arithmetic operations.
    friend CVector3D operator +(const CVector3D& l, const CVector3D& r)
    {
        return CVector3D(l.x + r.x, l.y + r.y, l.z + r.z);
    }

    friend CVector3D operator -(const CVector3D& l, const CVector3D& r)
    {
        return CVector3D(l.x - r.x, l.y - r.y, l.z - r.z);
    }

    friend CVector3D operator *(const CVector3D& l, const CVector3D& r)
    {
        return CVector3D(l.x * r.x, l.y * r.y, l.z * r.z);
    }

    friend CVector3D operator /(const CVector3D& l, const CVector3D& r)
    {
        return CVector3D(l.x / r.x, l.y / r.y, l.z / r.z);
    }

    //! Miscellaneous combined assignment operators.
    CVector3D& operator +=(const CVector3D& p)
    {
        x += p.x;
        y += p.y;
        z += p.z;
        return *this;
    }

    CVector3D& operator -=(const CVector3D& p)
    {
        x -= p.x;
        y -= p.y;
        z -= p.z;
        return *this;
    }

    CVector3D& operator *=(const CVector3D& p)
    {
        x *= p.x;
        y *= p.y;
        z *= p.z;
        return *this;
    }

    CVector3D& operator /=(const CVector3D& p)
    {
        x /= p.x;
        y /= p.y;
        z /= p.z;
        return *this;
    }

    CVector3D& operator +=(tCoordinate c)
    {
        x += c;
        y += c;
        z += c;
        return *this;
    }

    CVector3D& operator -=(tCoordinate c)
    {
        x -= c;
        y -= c;
        z -= c;
        return *this;
    }

    CVector3D& operator *=(tCoordinate c)
    {
        x *= c;
        y *= c;
        z *= c;
        return *this;
    }

    CVector3D& operator /=(tCoordinate c)
    {
        x /= c;
        y /= c;
        z /= c;
        return *this;
    }

    //! Comparison operator.
    friend bool operator ==(const CVector3D& l, const CVector3D& r)
    {
        return (mds::math::getAbs(l.x - r.x) < CPoint3D::RESOLUTION
                && mds::math::getAbs(l.y - r.y) < CPoint3D::RESOLUTION
                && mds::math::getAbs(l.z - r.z) < CPoint3D::RESOLUTION
                );
    }

    //! Vector product.
/*  friend CVector3D operator %(const CVector3D& l, const CVector3D& r)
    {
        return CVector3D(l.y * r.z - l.z * r.y, l.z * r.x - l.x * r.z, l.x * r.y - l.y * r.x);
    }*/

    CVector3D& getVectorProduct(const CVector3D& u, const CVector3D& v)
    {
        x = u.y * v.z - u.z * v.y;
        y = u.z * v.x - u.x * v.z;
        z = u.x * v.y - u.y * v.x;
        return *this;
    }

    //! Scalar product.
/*  friend tCoordinate operator *(const CVector3D& l, const CVector3D& r)
    {
        return (l.x * r.x + l.y * r.y + l.z * r.z);
    }*/

    static tCoordinate getScalarProduct(const CVector3D& u, const CVector3D& v)
    {
        return (u.x * v.x + u.y * v.y + u.z * v.z);
    }


    //! Returns the vector length.
    tCoordinate getLength() const
    {
        return tCoordinate(sqrt(x * x + y * y + z * z));
    }

    //! Returns sum of all vector coordinates.
    tCoordinate getSum() const
    {
        return (x + y + z);
    }

    //! Returns multiplication of all vector coordinates.
    tCoordinate getMult() const
    {
        return (x * y * z);
    }

    //! Normalizes the vector coordinates.
    CVector3D& normalize()
    {
        tCoordinate Temp = getLength();
        if( Temp == 0 )
        {
            x = 0;
            y = 0;
            z = 0;
        }
        else
        {
            tCoordinate InvTemp = 1 / Temp;
            x *= InvTemp;
            y *= InvTemp;
            z *= InvTemp;
        }
        return *this;
    };

    //! Computes normalized normal vector of given points.
    CVector3D& getNormal(const CPoint3D& p0,
                         const CPoint3D& p1,
                         const CPoint3D& p2
                        )
    {
        CVector3D v01(p0, p1), v02(p0, p2);
        x = v01.x * v02.z - v01.z * v02.y;
        y = v01.z * v02.x - v01.x * v02.z;
        z = v01.x * v02.y - v01.y * v02.x;
        normalize();
        return *this;
    };
};


} // namespace img
} // namespace mds

#endif // MDS_VECTOR3D_H

