//==============================================================================
/*! \filter
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSlice.h                          \n
 * Section: libImage                            \n
 * Date:    2003/12/13                          \n
 *
 * $Id: mdsSlice.h 383 2007-06-21 12:17:06Z spanel $
 * 
 * Description:
 * - Density slice.
 */

#ifndef MDS_SLICE_H
#define MDS_SLICE_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsSharedPtr.h>
#include <MDSTk/Math/mdsBase.h>

#include "mdsImage.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Global definitions.
 */

//! Slice orientation.
enum EPlane
{
    PLANE_NON    = -1,
    PLANE_XY    = 0,
    PLANE_XZ    = 1,
    PLANE_YZ    = 2,
};


//==============================================================================
/*!
 * Class ecapsulating a density slice.
 */
class CSlice : public CDImage
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CSlice);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CSlice);

    //! Standard method getEntityName().
    MDS_ENTITY_NAME(Slice);

    //! Standard method getEntityCompression().
    MDS_ENTITY_COMPRESSION(mds::mod::CC_RAW);

public:
    //! Default constructor.
    CSlice();

    //! Constructor.
    CSlice(tSize XSize, tSize YSize);

    //! Constructor.
    CSlice(tSize XSize, tSize YSize, const CDImage::tPixel& Value);

    //! Copy constructor.
    //! - Makes a new copy of the slice data.
    CSlice(const CSlice& Slice);

    //! Copy constructor.
    //! - Makes only reference to the exisiting data.
    CSlice(const CSlice& Slice, EMakeRef);

    //! Copy constructor.
    //! - Makes a new copy of the slice data.
    CSlice(const CDImage& Image);

    //! Copy constructor.
    //! - Makes only reference to the exisiting data.
    CSlice(const CDImage& Image, EMakeRef);

    //! Destructor.
    virtual ~CSlice();

    //! Creates a new slice.
    bool create(tSize XSize, tSize YSize);

    //! Creates a new slice.
    bool create(tSize XSize, tSize YSize, const CDImage::tPixel& Value);

    //! Creates a new slice.
    bool create(tSize XSize, tSize YSize, double dDX, double dDY);

    //! Creates a new slice.
    bool create(tSize XSize, tSize YSize, const CDImage::tPixel& Value, double dDX, double dDY);

    //! Creates a new slice.
    //! - Makes a new copy of the slice data.
    bool create(const CSlice& Slice);

    //! Creates a new slice.
    //! - Makes only reference to the exisiting data.
    bool create(const CSlice& Slice, EMakeRef);

    //! Image assignment operator.
    CSlice& operator =(const CSlice& Slice);


    //! Returns the real pixel size in x-axis.
    double getDX() const { return m_dDX; }

    //! Returns the real pixel size in y-axis.
    double getDY() const { return m_dDY; }

    //! Sets the pixel size.
    void setPixel(double dDX, double dDY) { m_dDX = dDX; m_dDY = dDY; }

    //! Returns the slice thickness.
    double getThickness() const { return m_dThickness; }

    //! Sets the slice thickness.
    void setThickness(double dThickness) { m_dThickness = dThickness; }

    //! Gets the slice position index.
    int getIndex() const { return m_iIndex; }

    //! Sets the position index.
    void setIndex(int iIndex) { m_iIndex = iIndex; }

    //! Returns the slice position.
    double getPosition() const { return m_dPosition; }

    //! Sets the slice position.
    void setPosition(double dPosition) { m_dPosition = dPosition; }

    //! Returns orientation of the slice plane.
    EPlane getOrientation() const { return m_eOrientation; }

    //! Sets the slice plane orientation.
    void setOrientation(EPlane ePlane) { m_eOrientation = ePlane; }

    //! Operator smaller for comparision of positions of two slices.
    bool operator <(const CSlice& Slice) { return (m_dPosition < Slice.m_dPosition); }


    //! Serializes all class members.
    void serialize(mds::mod::CChannelSerializer& Writer);

    //! Deserializes all class members.
    void deserialize(mds::mod::CChannelSerializer& Reader);

protected:
    //! Real pixel size
    double m_dDX, m_dDY;

    //! Slice thickness
    double m_dThickness;

    //! Slice position
    double m_dPosition;

    //! Position index
    int m_iIndex;

    //! Slice plane orientation
    EPlane m_eOrientation;
};


//=============================================================================
/*
 * Smart pointer to slice.
 */
typedef CSlice::tSmartPtr   CSlicePtr;


} // namespace img
} // namespace mds

#endif // MDS_SLICE_H

