//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsPoint.h                          \n
 * Section: libImage                            \n
 * Date:    2004/04/21                          \n
 *
 * $Id: mdsPoint.h 457 2007-09-06 13:37:39Z spanel $
 *
 * Description:
 * - Point in 3D representation.
 */

#ifndef MDS_POINT3D_H
#define MDS_POINT3D_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsSmallObject.h>
#include <MDSTk/Math/mdsBase.h>

#include "mdsImageTypes.h"

#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Class representing a point in 3D space.
 */
class CPoint3D : public mds::base::CSmallValueObject<>
{
public:
    //! Point in 3D and pixel/voxel coordinates.
    tCoordinate x, y, z;

    //! Minimal distance to recognize two different points.
    static const tCoordinate RESOLUTION;

public:
    //! Default constructor.
//    CPoint3D() : x(0), y(0), z(0) {}
    CPoint3D() {}

    //! Constructor initializes point position.
    CPoint3D(tCoordinate px, tCoordinate py, tCoordinate pz = 0)
        : x(px)
        , y(py)
        , z(pz)
    {}

    //! Copy constructor.
    CPoint3D(const CPoint3D& p) : x(p.x), y(p.y), z(p.z) {}

    //! Destructor.
    ~CPoint3D() {}

    //! Assignment operator.
    CPoint3D& operator =(const CPoint3D& p)
    {
        x = p.y;
        y = p.y;
        z = p.z;
        return *this;
    }


    //! Returns point coordinate.
    tCoordinate getX() const { return x; }
    tCoordinate getY() const { return y; }
    tCoordinate getZ() const { return z; }

    //! Returns XY point coordinates.
    void getXY(tCoordinate& px, tCoordinate& py) const
    {
        px = x;
        py = y;
    }

    //! Returns all point coordinates.
    void getXYZ(tCoordinate& px, tCoordinate& py, tCoordinate& pz) const
    {
        px = x;
        py = y;
        pz = z;
    }

    //! Returns point coordinate.
    void setX(tCoordinate px) { x = px; }
    void setY(tCoordinate py) { y = py; }
    void setZ(tCoordinate pz) { z = pz; }

    //! Sets XY point coordinates.
    void setXY(tCoordinate px, tCoordinate py)
    {
        x = px;
        y = py;
    }

    //! Sets all point coordinates.
    void setXYZ(tCoordinate px, tCoordinate py, tCoordinate pz)
    {
        x = px;
        y = py;
        z = pz;
    }


    //! Arithmetic operations.
    friend CPoint3D operator +(const CPoint3D& l, const CPoint3D& r)
    {
        return CPoint3D(l.x + r.x, l.y + r.y, l.z + r.z);
    }

    friend CPoint3D operator -(const CPoint3D& l, const CPoint3D& r)
    {
        return CPoint3D(l.x - r.x, l.y - r.y, l.z - r.z);
    }

    friend CPoint3D operator *(const CPoint3D& l, const CPoint3D& r)
    {
        return CPoint3D(l.x * r.x, l.y * r.y, l.z * r.z);
    }

    friend CPoint3D operator /(const CPoint3D& l, const CPoint3D& r)
    {
        return CPoint3D(l.x / r.x, l.y / r.y, l.z / r.z);
    }

    //! Miscellaneous combined assignment operators.
    CPoint3D& operator +=(const CPoint3D& p)
    {
        x += p.x;
        y += p.y;
        z += p.z;
        return *this;
    }

    CPoint3D& operator -=(const CPoint3D& p)
    {
        x -= p.x;
        y -= p.y;
        z -= p.z;
        return *this;
    }

    CPoint3D& operator *=(const CPoint3D& p)
    {
        x *= p.x;
        y *= p.y;
        z *= p.z;
        return *this;
    }

    CPoint3D& operator /=(const CPoint3D& p)
    {
        x /= p.x;
        y /= p.y;
        z /= p.z;
        return *this;
    }

    CPoint3D& operator +=(tCoordinate c)
    {
        x += c;
        y += c;
        z += c;
        return *this;
    }

    CPoint3D& operator -=(tCoordinate c)
    {
        x -= c;
        y -= c;
        z -= c;
        return *this;
    }

    CPoint3D& operator *=(tCoordinate c)
    {
        x *= c;
        y *= c;
        z *= c;
        return *this;
    }

    CPoint3D& operator /=(tCoordinate c)
    {
        x /= c;
        y /= c;
        z /= c;
        return *this;
    }

    //! Comparison operator.
    friend bool operator ==(const CPoint3D& l, const CPoint3D& r)
    {
        return (mds::math::getAbs(l.x - r.x) < RESOLUTION
                && mds::math::getAbs(l.y - r.y) < RESOLUTION
                && mds::math::getAbs(l.z - r.z) < RESOLUTION
                );
    }


    //! Returns sum of all point coordinates.
    double getSum() const
    {
        return (x + y + z);
    }

    //! Returns multiplication of all point coordinates.
    double getMult() const
    {
        return (x * y * z);
    }

    //! Euclidean distance of two points.
    double getDistance(const CPoint3D& p) const
    {
        tCoordinate dx = p.x - x;
        tCoordinate dy = p.y - y;
        tCoordinate dz = p.z - z;
        return sqrt(dx * dx + dy * dy + dz * dz);
    }

    //! Square of the Euclidean distance of two points.
    double getSquareDistance(const CPoint3D& p) const
    {
        tCoordinate dx = p.x - x;
        tCoordinate dy = p.y - y;
        tCoordinate dz = p.z - z;
        return (dx * dx + dy * dy + dz * dz);
    }
};


} // namespace img
} // namespace mds

#endif // MDS_POINT3D_H

