//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsPixelTypes.h                     \n
 * Section: libImage                            \n
 * Date:    2004/05/18                          \n
 *
 * $Id: mdsPixelTypes.h 358 2007-06-13 11:08:43Z spanel $
 * 
 * Description:
 * - Basic pixel types definition.
 */

#ifndef MDS_PIXELTYPES_H
#define MDS_PIXELTYPES_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Math/mdsComplex.h>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Basic pixel types definition.
 */

//! Pixel used for grayscale 8-bit images.
typedef mds::sys::tUInt8            tPixel8;

//! Pixel used for grayscale 16-bit images.
typedef mds::sys::tUInt16           tPixel16;

//! Pixel used for float images.
typedef float                       tFloatPixel;

//! Density pixel used by medical images (slices).
typedef mds::sys::tInt16            tDensityPixel;

//! Complex pixel definition.
typedef mds::math::CComplex<float>  tComplexPixel;


//==============================================================================
/*!
 * RGB pixel definition.
 */

class CRGB
{
public:
    //! RGB pixel component type.
    typedef mds::sys::tUInt8 tComponent;

    //! RGB pixel represented as unsigned long.
    typedef mds::sys::tUInt32 tRGB;

public:
    //! RGB pixel components.
    union
    {
        //! Separated RGB pixel components.
        struct
        {
            tComponent r, g, b, a;
        };

        //! RGB pixel represented by a one tRGB value.
        tRGB rgb;
    };

public:
    //! Default constructor.
    CRGB() : rgb(0xff000000) {}

    //! Constructor.
    //! - Separated RGB color components
    CRGB(tComponent Red, tComponent Green, tComponent Blue)
        : r(Red), g(Green), b(Blue), a(0xff)
    {}

    //! Constructor
    //! - RGB pixel.
    CRGB(tRGB Pixel) : rgb(Pixel) {}

    //! Copy constructor.
    CRGB(const CRGB& Pixel) : rgb(Pixel.rgb) {}

    //! Assignment operators.
    CRGB& operator =(CRGB Pixel)
    {
        rgb = Pixel.rgb;
        return *this;
    }

    CRGB& operator =(tRGB Pixel)
    {
        rgb = Pixel;
        return *this;
    }

    //! Conversion operators.
    operator tRGB() { return (rgb | 0xff000000); }


    //! Returns individual component value.
    tComponent getRed() const { return r; }
    tComponent getGreen() const { return g; }
    tComponent getBlue() const { return b; }
    tComponent getAlpha() const { return a; }

    //! Changes component value.
    void setRed(tComponent Red) { r = Red; }
    void setGreen(tComponent Green) { g = Green; }
    void setBlue(tComponent Blue) { b = Blue; }
    void setAlpha(tComponent Alpha) { a = Alpha; }


    //! Arithmetic operations
    friend CRGB operator +(CRGB l, CRGB r)
    {
        return CRGB(l.r + r.r, l.g + r.g, l.b + r.b);
    }

    friend CRGB operator -(CRGB l, CRGB r)
    {
        return CRGB(l.r - r.r, l.g - r.g, l.b - r.b);
    }

    //! Miscellaneous combined assignment operators
    CRGB& operator +=(CRGB p)
    {
        r += p.r;
        g += p.g;
        b += p.b;
        return *this;
    }

    CRGB& operator -=(CRGB p)
    {
        r -= p.r;
        g -= p.g;
        b -= p.b;
        return *this;
    }

    CRGB& operator +=(tComponent c)
    {
        r += c;
        g += c;
        b += c;
        return *this;
    }

    CRGB& operator -=(tComponent c)
    {
        r -= c;
        g -= c;
        b -= c;
        return *this;
    }

    //! Comparison operator
    friend bool operator ==(const CRGB& l, const CRGB& r)
    {
        return (l.r == r.r && l.g == r.g && l.b == r.b);
    }
};

//! RGB pixel of a color image.
typedef CRGB tRGBPixel;


} // namespace img
} // namespace mds

#endif // MDS_PIXELTYPES_H

