//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsKernel.h                         \n
 * Section: libImage                            \n
 * Date:    2004/04/19                          \n
 *
 * $Id: mdsKernel.h 64 2006-08-11 08:45:24Z spanel $
 * 
 * Description:
 * - Convolution kernel definition.
 */

#ifndef MDS_KERNEL_H
#define MDS_KERNEL_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsData.h>

#include "mdsImageTypes.h"


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Class containing a 1D convolution kernel.
 * - Kernel width must be an odd number.
 */
class CKernel1D : private mds::base::CData<tConvKernelData>
{
public:
    //! Kernel data.
    typedef mds::base::CData<tConvKernelData> data;
    typedef data::tData tData;
    typedef data::tDataPtr tDataPtr;
    typedef data::tConstDataPtr tConstDataPtr;

public:
    //! Constructor that creates a new 1D convolution kernel.
    CKernel1D(tSize Size);

    //! Constructor.
    CKernel1D(tSize Size, const tData& Value);

    //! Constructor.
    CKernel1D(tConstDataPtr pData, tSize Size);

    //! Copy constructor.
    CKernel1D(const CKernel1D& k);

    //! Virtual destructor.
    virtual ~CKernel1D() {}

    //! Creates a new 1D convolution kernel.
    void resize(tSize Size);

    //! Assignment operator.
    CKernel1D& operator =(const CKernel1D& k);


    //! Returns the kernel size.
    tSize getSize() const { return m_Size; }

    //! Returns subscripted kernel data [i].
    //! - Kernel data are viewed as one dimensional.
    tData& operator ()(tSize i) { return *(m_pData + i); }
    const tData& operator ()(tSize i) const { return *(m_pData + i); }

    //! Gets the kernel data [i].
    //! - Kernel data are viewed as one dimensional.
    tData& get(tSize i) { return *(m_pData + i); }
    const tData& get(tSize i) const { return *(m_pData + i); }

    //! Sets the kernel data.
    void set(tSize i, const tData& Value) { *(m_pData + i) = Value; }

    //! Returns pointer to the kernel data.
    tDataPtr getPtr(tSize i) { return (m_pData + i); }
    const tDataPtr getPtr(tSize i) const { return (m_pData + i); }

    //! Returns pointer to the kernel data.
    tDataPtr getPtr() { return m_pData; }
    const tDataPtr getPtr() const { return m_pData; }

protected:
    //! Default constructor.
    //! - It does no allocation of the kernel data.
    //! - Be sure that the method init(...) is used somewhere.
    //! - Avoid of using it!
    CKernel1D() {}

    //! Allocates data of a given size.
    //! - This method is primarily designed for kernel data allocation
    //!   when the default constructor was used.
    //! - Provides no checking of the existing data and its (de,re)allocation.
    //! - Avoid of using it!
    void init(tSize Size)
    {
        MDS_ASSERT((Size % 2) == 1);

        data::init(Size);
    }
};


//==============================================================================
/*!
 * Class encapsulating a 2D convolution kernel.
 */
class CKernel2D : private mds::base::CData<tConvKernelData>
{
public:
    //! Data base.
    typedef mds::base::CData<tConvKernelData> data;
    typedef data::tData tData;
    typedef data::tDataPtr tDataPtr;
    typedef data::tConstDataPtr tConstDataPtr;

public:
    //! Constructor that creates a new 2D convolution kernel.
    CKernel2D(tSize Size);

    //! Constructor.
    CKernel2D(tSize Size, const tData& Value);

    //! Constructor.
    CKernel2D(tConstDataPtr pData, tSize Size);

    //! Copy constructor.
    CKernel2D(const CKernel2D& k);

    //! Virtual destructor.
    virtual ~CKernel2D() {}

    //! Creates a new 2D convolution kernel.
    void resize(tSize Size);

    //! Assignment operator.
    CKernel2D& operator =(const CKernel2D& k);


    //! Returns the kernel size.
    tSize getSize() const { return m_YOffset; }

    //! Returns subscripted kernel data [x][y].
    //! - Kernel data are viewed as two dimensional.
    tData& operator ()(tSize x, tSize y)
    {
        return *(m_pData + y * m_YOffset + x);
    }
    const tData& operator ()(tSize x, tSize y) const
    {
        return *(m_pData + y * m_YOffset + x);
    }

    //! Gets the kernel data [x][y].
    //! - Kernel data viewed as two dimensional array.
    tData& get(tSize x, tSize y)
    {
        return *(m_pData + y * m_YOffset + x);
    }
    const tData& get(tSize x, tSize y) const
        {
            return *(m_pData + y * m_YOffset + x);
        }

    //! Sets the kernel data.
    void set(tSize x, tSize y, const tData& Value)
    {
        *(m_pData + y * m_YOffset + x) = Value;
    }

    //! Returns pointer to a given kernel data [x][y].
    tDataPtr getPtr(tSize x, tSize y)
    {
        return (m_pData + y * m_YOffset + x);
    }
    const tDataPtr getPtr(tSize x, tSize y) const
    {
        return (m_pData + y * m_YOffset + x);
    }

    //! Returns pointer to the kernel data.
    tDataPtr getPtr() { return m_pData; }
    const tDataPtr getPtr() const { return m_pData; }

protected:
    //! Kernel size in y-dimension.
    tSize m_YOffset;

protected:
    //! Default constructor.
    //! - It does no allocation of the kernel data.
    //! - Be sure that the method init(...) is used somewhere.
    //! - Avoid of using it!
    CKernel2D() {}

    //! Allocates data of a given size.
    //! - This method is primarily designed for kernel data allocation
    //!   when the default constructor was used.
    //! - Provides no checking of the existing data and its (de,re)allocation.
    //! - Avoid of using it!
    void init(tSize Size)
    {
        MDS_ASSERT((Size % 2) == 1);

        data::init(Size * Size);
        m_YOffset = Size;
    }
};


//==============================================================================
/*!
 * Class encapsulating a 3D convolution kernel.
 */
class CKernel3D : private mds::base::CData<tConvKernelData>
{
public:
    //! Data base.
    typedef mds::base::CData<tConvKernelData> data;
    typedef data::tData tData;
    typedef data::tDataPtr tDataPtr;
    typedef data::tConstDataPtr tConstDataPtr;

public:
    //! Constructor that creates a new 3D convolution kernel.
    CKernel3D(tSize Size);

    //! Constructor.
    CKernel3D(tSize Size, const tData& Value);

    //! Constructor.
    CKernel3D(tConstDataPtr pData, tSize Size);

    //! Copy constructor.
    CKernel3D(const CKernel3D& k);

    //! Virtual destructor.
    virtual ~CKernel3D() {}

    //! Creates a new 2D convolution kernel.
    void resize(tSize Size);

    //! Assignment operator.
    CKernel3D& operator =(const CKernel3D& k);


    //! Returns the kernel size.
    tSize getSize() const { return m_YOffset; }

    //! Returns subscripted kernel data [x][y][z].
    //! - Kernel data are viewed as three dimensional.
    tData& operator ()(tSize x, tSize y, tSize z)
    {
        return *(m_pData + z * m_ZOffset + y * m_YOffset + x);
    }
    const tData& operator ()(tSize x, tSize y, tSize z) const
    {
        return *(m_pData + z * m_ZOffset + y * m_YOffset + x);
    }

    //! Gets the kernel data [x][y][z].
    //! - Kernel data are viewed as three dimensional.
    tData& get(tSize x, tSize y, tSize z)
    {
        return *(m_pData + z * m_ZOffset + y * m_YOffset + x);
    }
    const tData& get(tSize x, tSize y, tSize z) const
        {
            return *(m_pData + z * m_ZOffset + y * m_YOffset + x);
        }

    //! Sets the kernel data.
    //! - 3D version.
    void set(tSize x, tSize y, tSize z, const tData& Value)
    {
        *(m_pData + z * m_ZOffset + y * m_YOffset + x) = Value;
    }

    //! Returns pointer to a given kernel data [x][y][z].
    //! - 3D version.
    tDataPtr getPtr(tSize x, tSize y, tSize z)
    {
        return (m_pData + z * m_ZOffset + y * m_YOffset + x);
    }
    const tDataPtr getPtr(tSize x, tSize y, tSize z) const
    {
        return (m_pData + z * m_ZOffset + y * m_YOffset + x);
    }

    //! Returns pointer to the kernel data.
    tDataPtr getPtr() { return m_pData; }
    const tDataPtr getPtr() const { return m_pData; }

protected:
    //! Kernel size.
    tSize m_YOffset, m_ZOffset;

protected:
    //! Default constructor.
    //! - It does no allocation of the kernel data.
    //! - Be sure that the method init(...) is used somewhere.
    //! - Avoid of using it!
    CKernel3D() {}

    //! Allocates data of a given size.
    //! - This method is primarily designed for kernel data allocation
    //!   when the default constructor was used.
    //! - Provides no checking of the existing data and its (de,re)allocation.
    //! - Avoid of using it!
    void init(tSize Size)
    {
        MDS_ASSERT((Size % 2) == 1);

        data::init(Size * Size * Size);
        m_YOffset = Size;
        m_ZOffset = Size * Size;
    }
};


} // namespace img
} // namespace mds

#endif // MDS_KERNEL_H

