//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsImageIterator.h                  \n
 * Section: libImage                            \n
 * Date:    2005/02/05                          \n
 *
 * $Id: mdsImageIterator.h 350 2007-06-12 07:42:17Z spanel $
 * 
 * Description:
 * - Image iterator definition.
 */

#ifndef MDS_IMAGEITERATOR_H
#define MDS_IMAGEITERATOR_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsIterator.h>


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Iterator used to traverse image pixels by lines.
 * - Template parameter I is the image type.
 * - Parameter T is the image pixel type.
 */
template <class I, typename T>
class CImageIterator : public mds::base::CComplexIteratorBase<I,T>
{
public:
    //! Check that I is an image. You will see name of this enum somewhere
    //! in compiler error message if the type I is not image.
    enum { TEMPLATE_PARAMETER_IS_NOT_IMAGE = I::CLASS_IMAGE };

    //! Iterator base.
    typedef mds::base::CComplexIteratorBase<I,T> base;

    //! Image type.
    typedef I tImage;

    //! Pixel type.
    typedef T tPixel;

public:
    //! Default constructor.
    CImageIterator() {}

    //! Constructor.
    CImageIterator(tImage& Image, tSize x, tSize y)
        : mds::base::CComplexIteratorBase<I,T>(&Image, Image.getPtr(x, y))
        , m_x(x)
        , m_y(y)
    {}

    //! Copy constructor.
    CImageIterator(const CImageIterator& It)
        : mds::base::CComplexIteratorBase<I,T>(It)
        , m_x(It.m_x)
        , m_y(It.m_y)
    {}

    //! Destructor.
    ~CImageIterator() {}

    //! Assignment operator.
    CImageIterator& operator=(const CImageIterator& It)
    {
        base::m_pContainer = It.m_pContainer;
        base::m_pItem = It.m_pItem;
        m_x = It.m_x;
        m_y = It.m_y;
        return *this;
    }


    //! Operator moves the iterator to the next pixel.
    //! - Prefix notation.
    CImageIterator& operator++()
    {
        advance();
        return *this;
    }

    //! Operator moves the iterator to the next pixel.
    //! - Postfix notation.
    CImageIterator operator++(int)
    {
        CImageIterator Temp(*this);
        advance();
        return Temp;
    }

    //! Returns the number of increments between the positions addressed
    //! by two iterators.
    tSize getDistance(const CImageIterator& End)
    {
        tSize Count = End.m_y * End.m_pContainer->getXSize() + End.m_x;
        Count -= m_y * base::m_pContainer->getXSize() + m_x;
        return (Count > 0) ? Count : 0;
    }

    //! Returns current iterator position.
    tSize getX() const { return m_x; }
    tSize getY() const { return m_y; }

    //! Sets the iterator position.
    void setPosition(tImage& Image, tSize x, tSize y)
    {
        base::m_pContainer = &Image;
        base::m_pItem = Image.getPtr(x, y);
        m_x = x;
        m_y = y;
    }

protected:
    //! Iterator position.
    tSize m_x, m_y;
    
public:
    //! Moves iterator to the next pixel.
    void advance()
    {      
        base::m_pItem += base::m_pContainer->getXOffset();
        if( ++m_x >= base::m_pContainer->getXSize() )
        {
            m_x = 0;
            ++m_y;
            base::m_pItem = base::m_pContainer->getPtr(m_x, m_y);
        }
    }
};


} // namespace img
} // namespace mds

#endif // MDS_IMAGEITERATOR_H

