//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsImageFilterPolicy.h              \n
 * Section: libImage                            \n
 * Date:    2005/10/04                          \n
 *
 * $Id: mdsImageFilterPolicy.h 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Policy for image filter response normalization.
 */

#ifndef MDS_IMAGEFILTERPOLICY_H
#define MDS_IMAGEFILTERPOLICY_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsObject.h>
#include <MDSTk/Math/mdsBase.h>

#include "mdsImageTypes.h"
#include "mdsPixelTraits.h"
#include "mdsImage.h"


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Policy for image filter response normalization.
 * - Simple conversion to a pixel value.
 * - Parameter I is an image type.
 */
template <class I>
class IFN_CONV
{
public:
    //! Check that I is an image. You will see name of this enum somewhere
    //! in compiler error message if the type I is not image.
    enum { TEMPLATE_PARAMETER_IS_NOT_IMAGE = I::CLASS_IMAGE };

    //! Declare name of the class.
    MDS_STATIC_NAME(IFN_CONV);

    //! Image pixel type.
    typedef typename I::tPixel tPixel;

    //! Image filter response type.
    typedef typename CPixelTraits<tPixel>::tFilterResponse tFilterResponse;

public:
    //! Static filter response normalization function.
    static tPixel normalize(tFilterResponse Value)
    {
        return tPixel(Value);
    }
};


//==============================================================================
/*!
 * Policy for image filter response normalization.
 * - Absolute value of the filter response.
 * - Parameter I is an image type.
 */
template <class I>
class IFN_ABS
{
public:
    //! Check that I is an image. You will see name of this enum somewhere
    //! in compiler error message if the type I is not image.
    enum { TEMPLATE_PARAMETER_IS_NOT_IMAGE = I::CLASS_IMAGE };

    //! Declare name of the class.
    MDS_STATIC_NAME(IFN_ABS);

    //! Image pixel type.
    typedef typename I::tPixel tPixel;

    //! Image filter response type.
    typedef typename CPixelTraits<tPixel>::tFilterResponse tFilterResponse;

public:
    //! Static point response normalization function.
    static tPixel normalize(tFilterResponse Value)
    {
        return tPixel(mds::math::getAbs(Value));
    }
};


//==============================================================================
/*!
 * Policy for image filter response normalization.
 * - Makes zero response equal to gray (mean pixel value).
 * - Parameter I is an image type.
 */
template <class I>
class IFN_MEAN
{
public:
    //! Check that I is an image. You will see name of this enum somewhere
    //! in compiler error message if the type I is not image.
    enum { TEMPLATE_PARAMETER_IS_NOT_IMAGE = I::CLASS_IMAGE };

    //! Declare name of the class.
    MDS_STATIC_NAME(IFN_MEAN);

    //! Image pixel type.
    typedef typename I::tPixel tPixel;

    //! Image filter response type.
    typedef typename CPixelTraits<tPixel>::tFilterResponse tFilterResponse;

public:
    //! Static point response normalization function.
    static tPixel normalize(tFilterResponse Value)
    {
        return tPixel(Value + CPixelTraits<tPixel>::getGray());
    }
};


//==============================================================================
/*!
 * Policy for image filter response normalization.
 * - Only the positive filter response is accepted.
 * - Parameter I is an image type.
 */
template <class I>
class IFN_POS
{
public:
    //! Check that I is an image. You will see name of this enum somewhere
    //! in compiler error message if the type I is not image.
    enum { TEMPLATE_PARAMETER_IS_NOT_IMAGE = I::CLASS_IMAGE };

    //! Declare name of the class.
    MDS_STATIC_NAME(IFN_POS);

    //! Image pixel type.
    typedef typename I::tPixel tPixel;

    //! Image filter response type.
    typedef typename CPixelTraits<tPixel>::tFilterResponse tFilterResponse;

public:
    //! Static point response normalization function.
    static tPixel normalize(tFilterResponse Value)
    {
        static const tPixel Zero = CPixelTraits<tPixel>::getZero();

        return (tPixel(Value) > Zero) ? tPixel(Value) : Zero;
    }
};


//==============================================================================
/*!
 * Policy for image filter response normalization.
 * - Specific normalization function can be set.
 * - Parameter I is an image type.
 */
template <class I>
class IFN_FUNC
{
public:
    //! Check that I is an image. You will see name of this enum somewhere
    //! in compiler error message if the type I is not image.
    enum { TEMPLATE_PARAMETER_IS_NOT_IMAGE = I::CLASS_IMAGE };

    //! Declare name of the class.
    MDS_STATIC_NAME(IFN_FUNC);

    //! Image pixel type.
    typedef typename I::tPixel tPixel;

    //! Image filter response type.
    typedef typename CPixelTraits<tPixel>::tFilterResponse tFilterResponse;

    //! Type of the normalization function.
    typedef tPixel (*tNormFunc)(tFilterResponse);

public:
    //! Default constructor.
    IFN_FUNC() : m_NormFunc(IFN_CONV<I>::normalize) {}

    //! Point response normalization function.
    tPixel normalize(tFilterResponse Value)
    {
        return m_NormFunc(Value);
    }

    //! Sets the normalization function.
    void setNormFunc(tNormFunc Func)
    {
        MDS_ASSERT(Func);

        m_NormFunc = Func;
    }

protected:
    //! Pointer to the concrete normalization function.
    tNormFunc m_NormFunc;
};


} // namespace img
} // namespace mds

#endif // MDS_IMAGEFILTERPOLICY_H

