//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsImageFilter.h                    \n
 * Section: libImage                            \n
 * Date:    2004/05/04                          \n
 *
 * $Id: mdsImageFilter.h 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Image filtering functions.
 */

#ifndef MDS_IMAGEFILTER_H
#define MDS_IMAGEFILTER_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsIdentifier.h>

#include "mdsImageTypes.h"
#include "mdsImage.h"
#include "mdsImageFilterPolicy.h"


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Base class for all image filters.
 * - Parameter I is an image type.
 * - Policy 'CNormPolicy' is used for filter response normalization.
 */
template <class I, template <typename> class CNormPolicy>
struct CImageFilterBase : public CNormPolicy<I>
{
    //! Check that I is an image. You will see name of this enum somewhere
    //! in compiler error message if the type I is not image.
    enum { TEMPLATE_PARAMETER_IS_NOT_IMAGE = I::CLASS_IMAGE };

    //! Filter response normalization policy.
    typedef CNormPolicy<I> norm;

    //! Image type.
    typedef typename I::tImage tImage;

    //! Image pixel type.
    typedef typename I::tPixel tPixel;

    //! Image filter response type.
    typedef typename CPixelTraits<tPixel>::tFilterResponse tResult;
};


//==============================================================================
/*!
 * Basic class providing image filtering functions. This template
 * is further specialized for concrete filter types.
 * - Parameter I is an image type.
 * - Id is a filter type.
 * - Policy N is used for filter response normalization.
 */
template <class I, typename Id, template <typename> class N = IFN_CONV>
class CImageFilter : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

public:
    //! Filtering of input/source image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);
};


} // namespace img
} // namespace mds

#endif // MDS_IMAGEFILTER_H

