//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsImage.hxx                        \n
 * Section: libImage                            \n
 * Date:    2003/12/08                          \n
 *
 * $Id: mdsImage.hxx 387 2007-06-22 09:33:41Z spanel $
 * 
 * Description:
 * - Template providing basic image operations.
 */


//==============================================================================
/*
 * Methods templates.
 */

// Constructor
template <typename T>
CImage<T>::CImage(tSize XSize, tSize YSize, tSize Margin)
    : m_XSize(XSize)
    , m_YSize(YSize)
    , m_Margin(Margin)
{
    // Helper value
    tSize Margins = 2 * m_Margin;

    // Allocate the data
    m_YOffset = m_XSize + Margins;
    m_DataStorage.init(m_YOffset * (YSize + Margins));
    m_pData = m_DataStorage.getPtr((m_YOffset + 1) * m_Margin);
}


// Constructor
template <typename T>
CImage<T>::CImage(tSize XSize,
                  tSize YSize,
                  const T& Value,
                  tSize Margin
                 )
    : m_XSize(XSize)
    , m_YSize(YSize)
    , m_Margin(Margin)
{
    // Helper value
    tSize Margins = 2 * m_Margin;

    // Allocate the data
    m_YOffset = m_XSize + Margins;
    m_DataStorage.init(m_YOffset * (YSize + Margins));
    m_pData = m_DataStorage.getPtr((m_YOffset + 1) * m_Margin);

    // Fill the image
    fillEntire(Value);
}


// Constructor
template <typename T>
CImage<T>::CImage(const CImage<T>& Image,
                  tSize x,
                  tSize y,
                  tSize XSize,
                  tSize YSize
                  )
{
    MDS_ASSERT(x >= 0 && XSize >= 0 && x < Image.m_XSize
               && y >= 0 && YSize >= 0 && y < Image.m_YSize);

    // Helper value
    tSize Margins = 2 * Image.m_Margin;

    // Size of the image
    m_XSize = MIN(x + XSize, Image.m_XSize) - x;
    m_YSize = MIN(y + YSize, Image.m_YSize) - y;

    // Allocate the data
    m_Margin = Image.m_Margin;
    m_YOffset = m_XSize + Margins;
    m_DataStorage.init(m_YOffset * (YSize + Margins));
    m_pData = m_DataStorage.getPtr((m_YOffset + 1) * m_Margin);

    // Size of the data to copy
    tSize XCount = m_XSize + Margins;
    tSize YCount = m_YSize + Margins;

    // Copy the data
    T *p = m_DataStorage.getPtr();
    T *pMax = p + YCount * m_YOffset;
    const T *pSrc = Image.getPtr(x - m_Margin, y - m_Margin);
    while( p < pMax )
    {
        memCopy(p, pSrc, XCount);
        p += m_YOffset;
        pSrc += Image.m_YOffset;
    }
}


// Constructor
template <typename T>
CImage<T>::CImage(const CImage<T>& Image,
                  tSize x,
                  tSize y,
                  tSize XSize,
                  tSize YSize,
                  EMakeRef MakeRef
                  )
    : m_Margin(Image.m_Margin)
    , m_YOffset(Image.m_YOffset)
    , m_DataStorage(Image.m_DataStorage, REFERENCE)
    , m_pData(Image.m_pData)
{
    MDS_ASSERT(x >= 0 && XSize >= 0 && x < Image.m_XSize
               && y >= 0 && YSize >= 0 && y < Image.m_YSize);

    // Image size
    m_XSize = MIN(x + XSize, Image.m_XSize) - x;
    m_YSize = MIN(y + YSize, Image.m_YSize) - y;

    // Image data
    m_pData += y * m_YOffset + x;
}


// Copy constructor
template <typename T>
CImage<T>::CImage(const CImage<T>& Image)
    : m_XSize(Image.m_XSize)
    , m_YSize(Image.m_YSize)
    , m_Margin(Image.m_Margin)
{
    // Helper value
    tSize Margins = 2 * Image.m_Margin;

    // Allocate the data
    m_YOffset = m_XSize + Margins;
    m_DataStorage.init(m_YOffset * (m_YSize + Margins));
    m_pData = m_DataStorage.getPtr((m_YOffset + 1) * m_Margin);

    // Size of the image data to copy
    tSize XCount = m_XSize + Margins;
    tSize YCount = m_YSize + Margins;

    // Copy the data
    T *p = m_DataStorage.getPtr();
    T *pMax = p + YCount * m_YOffset;
    const T *pSrc = Image.getPtr(-m_Margin, -m_Margin);
    while( p < pMax )
    {
        memCopy(p, pSrc, XCount);
        p += m_YOffset;
        pSrc += Image.m_YOffset;
    }
}


// Copy constructor
template <typename T>
CImage<T>::CImage(const CImage<T>& Image, EMakeRef MakeRef)
    : m_XSize(Image.m_XSize)
    , m_YSize(Image.m_YSize)
    , m_Margin(Image.m_Margin)
    , m_YOffset(Image.m_YOffset)
    , m_DataStorage(Image.m_DataStorage, REFERENCE)
    , m_pData(Image.m_pData)
{
}


// Destructor
template <typename T>
CImage<T>::~CImage()
{
}


// Creates a new true image
template <typename T>
bool CImage<T>::create(tSize XSize,
                       tSize YSize,
                       tSize Margin
                       )
{
    // Helper value
    tSize Margins = 2 * Margin;

    // Reallocate the data
    m_XSize = XSize;
    m_YSize = YSize;
    m_Margin = Margin;
    m_YOffset = m_XSize + Margins;
    m_DataStorage.resize(m_YOffset * (m_YSize + Margins));
    m_pData = m_DataStorage.getPtr((m_YOffset + 1) * m_Margin);

    // O.K.
    return true;
}


// Creates a new true image
template <typename T>
bool CImage<T>::create(tSize XSize,
                       tSize YSize,
                       const T& Value,
                       tSize Margin
                       )
{
    // Helper value
    tSize Margins = 2 * Margin;

    // Reallocate the data
    m_XSize = XSize;
    m_YSize = YSize;
    m_Margin = Margin;
    m_YOffset = m_XSize + Margins;
    m_DataStorage.resize(m_YOffset * (m_YSize + Margins));
    m_pData = m_DataStorage.getPtr((m_YOffset + 1) * m_Margin);

    // Fill the image
    fillEntire(Value);

    // O.K.
    return true;
}


// Creates a new image
template <typename T>
bool CImage<T>::create(const CImage<T>& Image,
                       tSize x,
                       tSize y,
                       tSize XSize,
                       tSize YSize
                       )
{
    // Check the position
    MDS_CHECK(x >= 0 && XSize >= 0 && x < Image.m_XSize
              && y >= 0 && YSize >= 0 && y < Image.m_YSize, return false);

    // Helper value
    tSize Margins = 2 * Image.m_Margin;

    // Image size
    m_XSize = MIN(x + XSize, Image.m_XSize) - x;
    m_YSize = MIN(y + YSize, Image.m_YSize) - y;

    // Reallocate the data
    m_XSize = XSize;
    m_YSize = YSize;
    m_Margin = Image.m_Margin;
    m_YOffset = m_XSize + Margins;
    m_DataStorage.resize(m_YOffset * (m_YSize + Margins));
    m_pData = m_DataStorage.getPtr((m_YOffset + 1) * m_Margin);

    // Image size
    tSize XCount = m_XSize + Margins;
    tSize YCount = m_YSize + Margins;

    // Copy the data
    T *p = m_DataStorage.getPtr();
    T *pMax = p + YCount * m_YOffset;
    const T *pSrc = Image.getPtr(x - m_Margin, y - m_Margin);
    while( p < pMax )
    {
        memCopy(p, pSrc, XCount);
        p += m_YOffset;
        pSrc += Image.m_YOffset;
    }

    // O.K.
    return true;
}

// Creates a new image
template <typename T>
bool CImage<T>::create(const CImage<T>& Image,
                       tSize x,
                       tSize y,
                       tSize XSize,
                       tSize YSize,
                       EMakeRef MakeRef
                       )
{
    // Check the position
    MDS_CHECK(x >= 0 && XSize >= 0 && x < Image.m_XSize
              && y >= 0 && YSize >= 0 && y < Image.m_YSize, return false);

    // Image size
    m_XSize = MIN(x + XSize, Image.m_XSize) - x;
    m_YSize = MIN(y + YSize, Image.m_YSize) - y;

    // Create a reference to the image data
    m_Margin = Image.m_Margin;
    m_YOffset = Image.m_YOffset;
    m_DataStorage.create(Image.m_DataStorage, REFERENCE);
    m_pData = Image.m_pData + y * m_YOffset + x;

    // O.K.
    return true;
}


// Creates a new image
template <typename T>
bool CImage<T>::create(const CImage<T>& Image)
{
    // Helper value
    tSize Margins = 2 * Image.m_Margin;

    // Reallocate the data
    m_XSize = Image.m_XSize;
    m_YSize = Image.m_YSize;
    m_Margin = Image.m_Margin;
    m_YOffset = m_XSize + Margins;
    m_DataStorage.resize(m_YOffset * (m_YSize + Margins));
    m_pData = m_DataStorage.getPtr((m_YOffset + 1) * m_Margin);

    // Size of the image data to copy
    tSize XCount = m_XSize + Margins;
    tSize YCount = m_YSize + Margins;

    // Copy the data
    T *p = m_DataStorage.getPtr();
    T *pMax = p + YCount * m_YOffset;
    const T *pSrc = Image.getPtr(-m_Margin, -m_Margin);
    while( p < pMax )
    {
        memCopy(p, pSrc, XCount);
        p += m_YOffset;
        pSrc += Image.m_YOffset;
    }

    // O.K.
    return true;
}


// Creates a new image
template <typename T>
bool CImage<T>::create(const CImage<T>& Image, EMakeRef MakeRef)
{
    // Make a reference to the data
    m_XSize = Image.m_XSize;
    m_YSize = Image.m_YSize;
    m_Margin = Image.m_Margin;
    m_YOffset = Image.m_YOffset;
    m_DataStorage.create(Image.m_DataStorage, REFERENCE);
    m_pData = Image.m_pData;

    // O.K.
    return true;
}


// Image conversion
template <typename T>
template <typename U>
bool CImage<T>::convert(const CImage<U>& Image)
{
    // Helper value
    tSize Margins = 2 * Image.getMargin();

    // Reallocate the data
    m_XSize = Image.getXSize();
    m_YSize = Image.getYSize();
    m_Margin = Image.getMargin();
    m_YOffset = m_XSize + Margins;
    m_DataStorage.resize(m_YOffset * (m_YSize + Margins));
    m_pData = m_DataStorage.getPtr((m_YOffset + 1) * m_Margin);

    // Size of the image data to copy
    tSize XCount = m_XSize + m_Margin;
    tSize YCount = m_YSize + m_Margin;

    // Convert image pixels
    for( tSize j = -m_Margin; j < YCount; ++j )
    {
        for( tSize i = -m_Margin; i < XCount; ++i )
        {
            get(i, j) = CPixelConversion<U,T>::convert(Image.get(i, j));
        }
    }

    // O.K.
    return true;
}


// Assignment operator
template <typename T>
CImage<T>& CImage<T>::operator =(const CImage<T>& Image)
{
    if( &Image == this )
    {
        return *this;
    }

    tSize XCount = MIN(m_XSize, Image.m_XSize);
    tSize YCount = MIN(m_YSize, Image.m_YSize);

    for( tSize j = 0; j < YCount; ++j )
    {
        memCopy(m_pData + j * m_YOffset,
                Image.m_pData + j * Image.m_YOffset,
                XCount
               );
    }

    return *this;
}


template <typename T>
inline CImage<T>& CImage<T>::operator +=(const CImage<T>& Image)
{
    tSize XCount = MIN(m_XSize, Image.m_XSize);
    tSize YCount = MIN(m_YSize, Image.m_YSize);

    for( tSize j = 0; j < YCount; ++j )
    {
        memVectAdd(m_pData + j * m_YOffset,
                   Image.m_pData + j * Image.m_YOffset,
                   XCount
                   );
    }

    return *this;
}


template <typename T>
inline CImage<T>& CImage<T>::operator -=(const CImage<T>& Image)
{
    tSize XCount = MIN(m_XSize, Image.m_XSize);
    tSize YCount = MIN(m_YSize, Image.m_YSize);

    for( tSize j = 0; j < YCount; ++j )
    {
        memVectSub(m_pData + j * m_YOffset,
                   Image.m_pData + j * Image.m_YOffset,
                   XCount
                   );
    }

    return *this;
}


template <typename T>
inline CImage<T>& CImage<T>::operator *=(const CImage<T>& Image)
{
    tSize XCount = MIN(m_XSize, Image.m_XSize);
    tSize YCount = MIN(m_YSize, Image.m_YSize);

    for( tSize j = 0; j < YCount; ++j )
    {
        memVectMult(m_pData + j * m_YOffset,
                    Image.m_pData + j * Image.m_YOffset,
                    XCount
                    );
    }

    return *this;
}


template <typename T>
inline CImage<T>& CImage<T>::operator /=(const CImage<T>& Image)
{
    tSize XCount = MIN(m_XSize, Image.m_XSize);
    tSize YCount = MIN(m_YSize, Image.m_YSize);

    for( tSize j = 0; j < YCount; ++j )
    {
        memVectDiv(m_pData + j * m_YOffset,
                   Image.m_pData + j * Image.m_YOffset,
                   XCount
                   );
    }

    return *this;
}


template <typename T>
template <typename U>
inline CImage<T>& CImage<T>::operator +=(const U& c)
{
    for( tSize j = 0; j < m_YSize; ++j )
    {
        memAdd(m_pData + j * m_YOffset, c, m_XSize);
    }
    return *this;
}


template <typename T>
template <typename U>
inline CImage<T>& CImage<T>::operator -=(const U& c)
{
    for( tSize j = 0; j < m_YSize; ++j )
    {
        memSub(m_pData + j * m_YOffset, c, m_XSize);
    }
    return *this;
}


template <typename T>
template <typename U>
inline CImage<T>& CImage<T>::operator *=(const U& c)
{
    for( tSize j = 0; j < m_YSize; ++j )
    {
        memMult(m_pData + j * m_YOffset, c, m_XSize);
    }
    return *this;
}


template <typename T>
template <typename U>
inline CImage<T>& CImage<T>::operator /=(const U& c)
{
    for( tSize j = 0; j < m_YSize; ++j )
    {
        memDiv(m_pData + j * m_YOffset, c, m_XSize);
    }
    return *this;
}


template <typename T>
inline void CImage<T>::setRow(tSize y, const T *pSrc)
{
    memCopy(m_pData + y * m_YOffset, pSrc, m_XSize);
}


template <typename T>
inline void CImage<T>::copyRow(tSize y, T *pDst) const
{
    memCopy(pDst, m_pData + y * m_YOffset, m_XSize);
}


template <typename T>
inline void CImage<T>::copyWindow(tSize x,
                                  tSize y,
                                  tSize Size,
                                  T *pDst
                                  ) const
{
    tSize Half = Size / 2;
    tSize Offset = m_YOffset - Size;

    T *p = m_pData + (y - Half) * m_YOffset + x - Half;
    for( tSize j = 0; j < Size; ++j )
    {
        T *pMax = p + Size;
        while( p < pMax )
        {
            *(pDst++) = *(p++);
        }
        p += Offset;
    }
}


template <typename T>
inline void CImage<T>::fill(const T& c)
{
    for( tSize j = 0; j < m_YSize; ++j )
    {
        memSet(m_pData + j * m_YOffset, c, m_XSize);
    }
}


template <typename T>
inline void CImage<T>::fillEntire(const T& c)
{
    memSet(m_DataStorage.getPtr(), c, m_DataStorage.getSize());
}


template <typename T>
inline void CImage<T>::fillMargin(const T& c)
{
    // Fill margin on the top and bottom
    tSize Count = m_Margin * m_YOffset;
    memSet(getPtr(-m_Margin, -m_Margin), c, Count);
    memSet(getPtr(-m_Margin, m_YSize), c, Count);

    // Fill left and right margin
    for( tSize j = 0; j < m_YSize; ++j )
    {
        memSet(getPtr(-m_Margin, j), c, m_Margin);
        memSet(getPtr(m_XSize, j), c, m_Margin);
    }
}


template <typename T>
inline void CImage<T>::mirrorMargin()
{
    for( tSize j = -m_Margin; j < m_YSize + m_Margin; ++j )
    {
        tSize jj = j;
        if( j < 0 )
        {
            jj = -j;
        }
        else if ( j >= m_YSize )
        {
            jj = 2 * m_YSize - 2 - j;
        }
        tSize i;
        for( i = 1; i <= m_Margin; ++i )
        {
            get(-i, j) = get(i, jj);
            get(m_XSize - 1 + i, j) = get(m_XSize - 1 - i, jj);
        }
        if( j < 0 || j >= m_YSize )
        {
            for( i = 0; i < m_XSize; ++i )
            {
                get(i, j) = get(i, jj);
            }
        }
    }
}


template <typename T>
inline void CImage<T>::replace(const T& Value, const T& NewValue)
{
    for( tSize j = 0; j < m_YSize; ++j )
    {
        T *p = m_pData + j * m_YOffset;
        T *pMax = p + m_XSize;
        while( p < pMax )
        {
            if( *p == Value )
            {
                *p = NewValue;
            }
            ++p;
        }
    }
}


template <typename T>
inline void CImage<T>::abs()
{
    for( tSize j = 0; j < m_YSize; ++j )
    {
        T *p = m_pData + j * m_YOffset;
        T *pMax = p + m_XSize;
        while( p < pMax )
        {
            if( !(*p > T(0)) )
            {
                *p = -(*p);
            }
            ++p;
        }
    }
}


template <typename T>
inline void CImage<T>::clip(const T& Lower, const T& Upper)
{
    for( tSize j = 0; j < m_YSize; ++j )
    {
        T *p = m_pData + j * m_YOffset;
        T *pMax = p + m_XSize;
        while( p < pMax )
        {
            if( *p < Lower )
            {
                *p = Lower;
            }
            else if( *p > Upper )
            {
                *p = Upper;
            }
            ++p;
        }
    }
}


template <typename T>
inline void CImage<T>::cut(const T& Lower, const T& Upper)
{
    for( tSize j = 0; j < m_YSize; ++j )
    {
        T *p = m_pData + j * m_YOffset;
        T *pMax = p + m_XSize;
        while( p < pMax )
        {
            if( *p < Lower )
            {
                *p = T(0);
            }
            else if( *p > Upper )
            {
                *p = T(0);
            }
            ++p;
        }
    }
}


template <typename T>
inline void CImage<T>::subSample(const CImage<T>& Image, tSize k, tSize l)
{
    MDS_CHECK(k > 0 && l > 0, return);

    tSize XCount = MIN(Image.m_XSize / k, m_XSize);
    tSize YCount = MIN(Image.m_YSize / l, m_YSize);

    for( tSize j = 0; j < YCount; ++j )
    {
        T *p = m_pData + j * m_YOffset;
        T *pMax = p + XCount;
        const T *pI = Image.m_pData + l * j * Image.m_YOffset;
        while( p < pMax )
        {
            *(p++) = *pI;
            pI += k;
        }
    }
}


template <typename T>
inline T CImage<T>::interpolate(const CPoint3D& Point) const
{
    int iX = (int)Point.x;
    int iY = (int)Point.y;

    double dX = Point.x - (double)iX;
    double dY = Point.y - (double)iY;
    double dTempX = 1.0 - dX;
    double dTempY = 1.0 - dY;

    T *p = m_pData + iY * m_YOffset + iX;

    // Interpolate subpixel value
    double dValue = dTempX * dTempY * (double)*(p);
    dValue += dX * dTempY * (double)*(p + 1);
    dValue += dTempX * dY * (double)*(p + m_YOffset);
    dValue += dX * dY * (double)*(p + m_YOffset + 1);

    // Return the interpolated pixel value
    return (tPixel)dValue;
}


template <typename T>
inline T CImage<T>::color2Pixel(CColor color) const
{
    // Direct conversion of RGB color to the RGB pixel
    tRGBPixel Pixel = tRGBPixel::tRGB(color);

    // Return the pixel value
    return CPixelConversion<tRGBPixel,T>::convert(Pixel);
}


template <typename T>
inline bool CImage<T>::checkPosition(tSize x, tSize y) const
{
    return (x >= 0 && x < m_XSize && y >= 0 && y < m_YSize);
}


//==============================================================================
/*
 * Image data serialization functions.
 */

template <typename T>
void CImage<T>::serialize(mds::mod::CChannelSerializer& Writer)
{
    // Begin of data serialization block
    MDS_DE_SERIALIZE_BEGIN;

        // Check if serializer supports shared memory
        if( Writer.testFlag(mds::mod::CSHMSerializer::SHM_FLAG) )
        {
            // Use SHM serializer
            mds::mod::CSHMSerializer& SHMWriter = static_cast<mds::mod::CSHMSerializer&>(Writer);
            
            // Image info
            SHMWriter.writeSize(m_XSize);
            SHMWriter.writeSize(m_YSize);
            SHMWriter.writeSize(m_Margin);
            SHMWriter.writeSize(m_YOffset);
            SHMWriter.writeData(m_DataStorage);            
        }
        else
        {
            // Image info
            Writer.writeSize(m_XSize);
            Writer.writeSize(m_YSize);
            
            // Image data
            for( tSize j = 0; j < m_YSize; ++j )
            {
                Writer.writeBytes(m_pData + j * m_YOffset, m_XSize * CTypeTraits<tPixel>::getSize());
            }
        }
    
    // End of the block
    MDS_DE_SERIALIZE_END;
}


template <typename T>
void CImage<T>::deserialize(mds::mod::CChannelSerializer& Reader)
{
    // Begin of data deserialization block
    MDS_DE_DESERIALIZE_BEGIN;

        // Check if serializer supports shared memory
        if( Reader.testFlag(mds::mod::CSHMSerializer::SHM_FLAG) )
        {
            // Use SHM serializer
            mds::mod::CSHMSerializer& SHMReader = static_cast<mds::mod::CSHMSerializer&>(Reader);
            
            // Image info
            SHMReader.readSize(m_XSize);
            SHMReader.readSize(m_YSize);
            SHMReader.readSize(m_Margin);
            SHMReader.readSize(m_YOffset);
            SHMReader.readData(m_DataStorage);
            if( SHMReader.isError() )
            {
                return;
            }
            
            // Check image size
            tSize Size = m_YOffset * (m_YSize + 2 * m_Margin); 
            if( Size < 0 || Size > m_DataStorage.getSize() )
            {
                SHMReader.setError();
                return;
            }
                       
            // Initialize the internal pointer            
            m_pData = m_DataStorage.getPtr((m_YOffset + 1) * m_Margin);
        }
        else
        {
            // Read the image info struct
            Reader.readSize(m_XSize);
            Reader.readSize(m_YSize);
    
            // Create a new slice
            if( !create(m_XSize, m_YSize, m_Margin) )
            {
                Reader.setError();
                return;
            }
    
            // Image data
            for( tSize j = 0; j < m_YSize; ++j )
            {
                Reader.readBytes(m_pData + j * m_YOffset, m_XSize * CTypeTraits<tPixel>::getSize());
            }
        }

    // End of the block
    MDS_DE_DESERIALIZE_END;
}

