//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsImage.h                          \n
 * Section: libImage                            \n
 * Date:    2003/12/08                          \n
 *
 * $Id: mdsImage.h 387 2007-06-22 09:33:41Z spanel $
 *
 * Description:
 * - Template providing basic image operations.
 */

#ifndef MDS_IMAGE_H
#define MDS_IMAGE_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsMemory.h>
//#include <MDSTk/Base/mdsRefData.h>
#include <MDSTk/Base/mdsSHMData.h>
#include <MDSTk/Base/mdsSharedPtr.h>
#include <MDSTk/Math/mdsBase.h>
//#include <MDSTk/Module/mdsSerializer.h>
#include <MDSTk/Module/mdsSHMSerializer.h>

#include "mdsImageBase.h"
#include "mdsImageIterator.h"
#include "mdsPoint.h"
#include "mdsPixelTraits.h"
#include "mdsPixelConversions.h"
#include "mdsColor.h"

#include <cmath>
#include <string>


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Image class template providing basic image operations.
 * - Parameter T is image pixel type.
 * - Image data are stored in row-major format. Index of the row represents
 *   y-axis and is generally given as a second parameter of pixel subscripting
 *   functions.
 */
template <typename T>
class CImage
    : public mds::CObject
    , public CImageBase<CImage<T> >
    , public mds::mod::CSerializable
{
public:
    //! Templates that require members of the CImage class can use.
    //! this enum to check the existence.
    enum { CLASS_IMAGE };

    //! Standard method getClassName().
    MDS_CLASS_NAME(CImage);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CImage);

    //! Image pixel type.
    typedef T tPixel;

    //! Image data storage.
//    typedef mds::base::CRefData<T> tDataStorage;
    typedef mds::base::CSHMData<T> tDataStorage;

    //! Iterator type.
    //! - Declares type tIterator.
    MDS_COMPLEX_ITERATOR(CImage, tPixel, CImageIterator);

    //! Standard method getEntityName().
    MDS_ENTITY_TEMPLATE_NAME(Image, CPixelTraits<T>::getPixelName());

    //! Standard method getEntityCompression().
    MDS_ENTITY_COMPRESSION(mds::mod::CC_RLE16);

public:
    //! Constructor.
    //! - 'Min' and 'Max' parameters specify the range of allowed
    //!   image pixel values.
    CImage(tSize XSize = 0, tSize YSize = 0, tSize Margin = 0);

    //! Constructor.
    //! - Parameter 'Value' is the pixel initial value.
    //! - 'Min' and 'Max' parameters specify the range of allowed
    //!   image pixel values.
    CImage(tSize XSize,
           tSize YSize,
           const T& Value,
           tSize Margin = 0
           );

    //! Constructor.
    //! - Creates a subimage of a given image.
    CImage(const CImage& Image,
           tSize x,
           tSize y,
           tSize XSize,
           tSize YSize
           );

    //! Constructor.
    //! - Creates a subimage of a given image.
    //! - Makes only reference to the data.
    CImage(const CImage& Image,
           tSize x,
           tSize y,
           tSize XSize,
           tSize YSize,
           EMakeRef
           );

    //! Copy constructor.
    //! - Makes a new copy of the data.
    CImage(const CImage& Image);

    //! Copy constructor.
    //! - Only reference to the data of the given image.
    CImage(const CImage& Image, EMakeRef);

    //! Destructor.
    virtual ~CImage();

    //! Creates a new image.
    bool create(tSize XSize,
                tSize YSize,
                tSize Margin = 0
                );

    //! Creates and initializes a new image.
    bool create(tSize XSize,
                tSize YSize,
                const T& Value,
                tSize Margin = 0
                );

    //! Creates a new image.
    bool create(const CImage& Image,
                tSize x,
                tSize y,
                tSize XSize,
                tSize YSize
                );

    //! Creates a new image.
    bool create(const CImage& Image,
                tSize x,
                tSize y,
                tSize XSize,
                tSize YSize,
                EMakeRef
                );

    //! Creates a new image.
    //! - Makes a new copy of the data.
    bool create(const CImage& Image);

    //! Creates a new image.
    //! - Makes reference to the data of a given image.
    bool create(const CImage& Image, EMakeRef);

    //! Assignment operator.
    CImage& operator =(const CImage& Image);

    //! Conversion of images of different types.
    template <typename U>
    bool convert(const CImage<U>& Image);


    //! Returns the image size (dimensions).
    tSize getXSize() const { return m_XSize; }
    tSize getYSize() const { return m_YSize; }

    //! Returns column offset in x-axis. In other words, offset between
    //! two neighbouring image pixels in a row.
    tSize getXOffset() const { return 1; }

    //! Returns row offset in y-axis. In other words, offset between two
    //! neighbouring image pixels in a column.
    tSize getYOffset() const { return m_YOffset; }

    //! Returns the image margin size in pixels.
    tSize getMargin() const { return m_Margin; }

    //! Returns the subscripted pixel [x][y].
    T& operator ()(tSize x, tSize y)
    {
        return *(m_pData + y * m_YOffset + x);
    }
    const T& operator ()(tSize x, tSize y) const
    {
        return *(m_pData + y * m_YOffset + x);
    }

    //! Gets the pixel [x][y].
    T& get(tSize x, tSize y)
    {
        return *(m_pData + y * m_YOffset + x);
    }
    const T& get(tSize x, tSize y) const
    {
        return *(m_pData + y * m_YOffset + x);
    }

    //! Returns pointer to the given pixel.
    T *getPtr(tSize x, tSize y)
    {
        return (m_pData + y * m_YOffset + x);
    }
    const T *getPtr(tSize x, tSize y) const
    {
        return (m_pData + y * m_YOffset + x);
    }

    //! Returns pointer to the first element of the image row.
    T *getRowPtr(tSize y)
    {
        return (m_pData + y * m_YOffset);
    }
    const T *getRowPtr(tSize y) const
    {
        return (m_pData + y * m_YOffset);
    }

    //! Returns pointer to the first element of the image column.
    T *getColPtr(tSize x)
    {
        return (m_pData + x);
    }
    const T *getColPtr(tSize x) const
    {
        return (m_pData + x);
    }

    //! Returns pointer to the image data.
    T *getPtr() { return m_pData; }
    const T *getPtr() const { return m_pData; }

    //! Sets the pixel at the position [x][y].
    void set(tSize x, tSize y, const T& Value)
    {
        *(m_pData + y * m_YOffset + x) = Value;
    }


    //! Returns iterator that points at the first image pixel.
    tIterator getBegin()
    {
        return tIterator(*this, 0, 0);
    }
    tConstIterator getBegin() const
    {
        return tConstIterator(*this, 0, 0);
    }

    //! Returns iterator that points beyond the last image pixel.
    tIterator getEnd()
    {
        return tIterator(*this, 0, m_YSize);
    }
    tConstIterator getEnd() const
    {
        return tConstIterator(*this, 0, m_YSize);
    }

    //! Returns iterator which points at a specified position.
    tIterator getIterator(tSize x, tSize y)
    {
        return tIterator(*this, x, y);
    }
    tConstIterator getIterator(tSize x, tSize y) const
    {
        return tConstIterator(*this, x, y);
    }


    //! Miscellaneous combined assignment operators.
    inline CImage& operator +=(const CImage& Image);
    inline CImage& operator -=(const CImage& Image);
    inline CImage& operator *=(const CImage& Image);
    inline CImage& operator /=(const CImage& Image);

    //! Combined assignment operator.
    //! - Operator T& operator +=(const U&) must be defined for the pixel type T!
    template <typename U>
    inline CImage& operator +=(const U& c);
    
    //! Combined assignment operator.
    //! - Operator T& operator -=(const U&) must be defined for the pixel type T!
    template <typename U>
    inline CImage& operator -=(const U& c);
    
    //! Combined assignment operator.
    //! - Operator T& operator *=(const U&) must be defined for the pixel type T!
    template <typename U>
    inline CImage& operator *=(const U& c);
    
    //! Combined assignment operator.
    //! - Operator T& operator /=(const U&) must be defined for the pixel type T!
    template <typename U>
    inline CImage& operator /=(const U& c);


    //! Fills the y-th image row using a given input buffer.
    inline void setRow(tSize y, const T *pSrc);

    //! Copies the y-th image row to the output buffer.
    inline void copyRow(tSize y, T *pDst) const;

    //! Copies the image window having side 'Size' to the output buffer.
    inline void copyWindow(tSize x,
                           tSize y,
                           tSize Size,
                           T *pDst
                           ) const;

    //! Fills the image using a given pixel value.
    inline void fill(const T& c);

    //! Fills the image including margin using a given pixel value.
    inline void fillEntire(const T& c);

    //! Pads the image margin with constant value.
    inline void fillMargin(const T& c);

    //! Pads the image margin using a simple mirroring.
    inline void mirrorMargin();

    //! Replaces all pixels of a given value by a specified value.
    inline void replace(const T& Value, const T& NewValue);

    //! Absolute value of all pixels.
    inline void abs();

    //! Clip range of pixel values.
    inline void clip(const T& Lower, const T& Upper);

    //! Cut range of pixel values.
    inline void cut(const T& Lower, const T& Upper);

    //! Subsample the image.
    inline void subSample(const CImage& Image, tSize k = 2, tSize l = 2);

    //! Bilinear subpixel value interpolation.
    inline T interpolate(const CPoint3D& Point) const;

    //! Converts a specified color to the image pixel format and range.
    inline T color2Pixel(CColor color) const;

    //! Checks the pixel position.
    //! - Returns 'true' if a specified position is satisfactory.
    inline bool checkPosition(tSize x, tSize y) const;


    //! Serializes all class members.
    void serialize(mds::mod::CChannelSerializer& Writer);

    //! Deserializes all class members.
    void deserialize(mds::mod::CChannelSerializer& Reader);

protected:
    //! Image dimensions.
    tSize m_XSize, m_YSize;

    //! Image margin size.
    tSize m_Margin;

    //! Offset used by subscripting functions.
    tSize m_YOffset;

    //! Image data.
    tDataStorage m_DataStorage;

    //! Pointer to the image data.
    T *m_pData;
};


//=============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates.
#include "mdsImage.hxx"


//=============================================================================
/*
 * Basic template instances and type definitions.
 */

//! Grayscale image, 8-bits per pixel.
typedef CImage<tPixel8>         CImage8;

//! Grayscale image, 16-bits per pixel.
typedef CImage<tPixel16>        CImage16;

//! Float image.
typedef CImage<tFloatPixel>     CFImage;

//! Density image.
typedef CImage<tDensityPixel>   CDImage;

//! Color RGB image.
typedef CImage<tRGBPixel>       CRGBImage;

//! Complex image.
typedef CImage<tComplexPixel>   CComplexImage;


//=============================================================================
/*
 * Basic template instances and type definitions.
 * - Using smart pointers.
 */

//! Grayscale image, 8-bit per pixel.
typedef CImage8::tSmartPtr          CImage8Ptr;

//! Grayscale image, 16-bit per pixel.
typedef CImage16::tSmartPtr         CImage16Ptr;

//! Float image.
typedef CFImage::tSmartPtr          CFImagePtr;

//! Density image.
typedef CDImage::tSmartPtr          CDImagePtr;

//! Color RGB image.
typedef CRGBImage::tSmartPtr        CRGBImagePtr;

//! Complex image.
typedef CComplexImage::tSmartPtr    CComplexImagePtr;


} // namespace img
} // namespace mds

#endif // MDS_IMAGE_H

