//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsHistogram.cpp                    \n
 * Section: libImage                            \n
 * Date:    2005/02/10                          \n
 *
 * $Id: mdsHistogram.hxx 330 2007-04-17 14:30:54Z spanel $
 * 
 * Description:
 * - Histogram computation.
 */


//==============================================================================
/*
 * Methods templates.
 */

// Estimates quantized histogram of a given container
template <typename C>
inline CQuantizedHistogram<C>::CQuantizedHistogram(tSize NumOfBins,
                                                   tItem Min,
                                                   tItem Max
                                                   )
    : m_Min(Min)
    , m_Max(Max)
    , m_Histogram(NumOfBins, tBin(0))
{
	// Check whether the container item is a fundamental type,
	// e.g. float, int and so on
	MDS_STATIC_ASSERT(CTypeTraits<tItem>::isFundamental,
	                  HISTOGRAM_ITEM_IS_NOT_A_FUNDAMENTAL_TYPE
	                  );

    // Check histogram size and allowed range
    MDS_ASSERT(NumOfBins > 0 && Max >= Min);

    // Compute the bin size
    tItem Span = m_Max - m_Min;
    m_BinSize =  Span / tItem(NumOfBins);
    if( (m_BinSize * tItem(NumOfBins)) < Span )
    {
        m_BinSize += tItem(1);
    }
}


// Estimates quantized histogram of a given container
template <typename C>
bool CQuantizedHistogram<C>::operator ()(const tContainer& Container)
{
    // Compute per item statistics
    tContainerIterator it = Container.getBegin();
    tContainerIterator itEnd = Container.getEnd();
    while( it != itEnd )
    {
        tItem Value = *it;
        if( Value >= m_Min && Value <= m_Max )
        {
            ++m_Histogram(getIndex(Value));
        }
        ++it;
    }

    // O.K.
    return true;
}


// Estimates cumulative histogram
template <typename C>
void CQuantizedHistogram<C>::cumulate()
{
    // Compute per item statistics
    for( tSize i = 1; i < m_Histogram.getSize(); ++i )
    {
        m_Histogram(i) += m_Histogram(i - 1);
    }
}


// Estimates full histogram of a given container
template <typename C>
inline CFullHistogram<C>::CFullHistogram(tItem Min, tItem Max)
    : m_Min(Min)
    , m_Max(Max)
    , m_Histogram(Max - Min + 1, tBin(0))
{
	// Check whether the container item is a fundamental integral type
	MDS_STATIC_ASSERT(CTypeTraits<tItem>::isFundamental && CTypeTraits<tItem>::isIntegral,
	                  HISTOGRAM_ITEM_IS_NOT_A_FUNDAMENTAL_INTEGRAL_TYPE
	                  );

    // Check allowed range
    MDS_ASSERT(Max >= Min);
}


// Estimates histogram of a given image
template <typename C>
bool CFullHistogram<C>::operator ()(const tContainer& Container)
{
    // Compute per item statistics
    tContainerIterator it = Container.getBegin();
    tContainerIterator itEnd = Container.getEnd();
    while( it != itEnd )
    {
        tItem Value = *it;
        if( Value >= m_Min && Value <= m_Max )
        {
            ++m_Histogram(getIndex(Value));
        }
        ++it;
    }

    // O.K.
    return true;
}


// Estimates cumulative histogram
template <typename C>
void CFullHistogram<C>::cumulate()
{
    // Compute per item statistics
    for( tSize i = 1; i < m_Histogram.getSize(); ++i )
    {
        m_Histogram(i) += m_Histogram(i - 1);
    }
}

