//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsDensityVolume.h                  \n
 * Section: libImage                            \n
 * Date:    2004/07/11                          \n
 *
 * $Id: mdsDensityVolume.h 383 2007-06-21 12:17:06Z spanel $
 *
 * Description:
 * - Manipulation with the real volume data.
 */

#ifndef MDS_DENSITYVOLUME_H
#define MDS_DENSITYVOLUME_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsSharedPtr.h>
#include <MDSTk/Math/mdsBase.h>

#include "mdsSlice.h"
#include "mdsVolume.h"


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * Class ecapsulating volume density data.
 */
class CDensityVolume : public CDVolume
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CDensityVolume);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CDensityVolume);

    //! Standard method getEntityName().
    MDS_ENTITY_NAME(DensityVolume);

    //! Standard method getEntityCompression().
    MDS_ENTITY_COMPRESSION(mds::mod::CC_RAW);

public:
    //! Default constructor.
    CDensityVolume();

    //! Constructor.
    CDensityVolume(tSize XSize, tSize YSize, tSize ZSize);

    //! Constructor.
    CDensityVolume(tSize XSize,
                   tSize YSize,
                   tSize ZSize,
                   const CDVolume::tVoxel& Value
                   );

    //! Copy constructor.
    //! - Makes a new copy of the volume data.
    CDensityVolume(const CDensityVolume& Volume);

    //! Copy constructor.
    //! - Makes only a reference to the exisiting data.
    CDensityVolume(const CDensityVolume& Volume, EMakeRef);

    //! Copy constructor.
    //! - Makes a new copy of the volume data.
    CDensityVolume(const CDVolume& Volume);

    //! Copy constructor.
    //! - Makes only a reference to the exisiting data.
    CDensityVolume(const CDVolume& Volume, EMakeRef);

    //! Destructor.
    virtual ~CDensityVolume();

    //! Creates a new volume.
    bool create(tSize XSize, tSize YSize, tSize ZSize);

    //! Creates a new volume.
    bool create(tSize XSize, tSize YSize, tSize ZSize, double dDX, double dDY, double dDZ);

    //! Creates a new volume.
    bool create(tSize XSize,
                tSize YSize,
                tSize ZSize,
                const CDVolume::tVoxel& Value
                );

    //! Creates a new volume.
    bool create(tSize XSize,
                tSize YSize,
                tSize ZSize,
                const CDVolume::tVoxel& Value,
                double dDX,
                double dDY,
                double dDZ
                );

    //! Creates a new volume.
    //! - Makes a new copy of the volume data.
    bool create(const CDensityVolume& Volume);

    //! Creates a new volume.
    //! - Makes only a reference to the exisiting data.
    bool create(const CDensityVolume& Volume, EMakeRef);

    //! Volume assignment operator.
    CDensityVolume& operator =(const CDensityVolume& Volume);


    //! Returns the real voxel size in x-axis.
    double getDX() const { return m_dDX; }

    //! Returns the real voxel size in y-axis.
    double getDY() const { return m_dDY; }

    //! Returns the real voxel size in z-axis.
    double getDZ() const { return m_dDZ; }

    //! Set the real voxel size in x-axis.
    void setDX(double dValue) { m_dDX = dValue; }

    //! Set the real voxel size in y-axis.
    void setDY(double dValue) { m_dDY = dValue; }

    //! Set the real voxel size in z-axis.
    void setDZ(double dValue) { m_dDZ = dValue; }

    //! Sets the voxel size.
    void setVoxel(double dDX, double dDY, double dDZ)
    {
        m_dDX = dDX; m_dDY = dDY; m_dDZ = dDZ;
    }


    //! Cuts the volume using plane XY and saves obtained data to a given
    //! slice. Position of the plane is specified by the Z index.
    bool getSliceXY(tSize z, CSlice& Plane);

    //! Cuts the volume using plane XZ and saves obtained data to a given
    //! slice. Position of the plane is specified by the Y index.
    bool getSliceXZ(tSize y, CSlice& Plane);

    //! Cuts the volume using plane YZ and saves obtained data to a given
    //! slice. Position of the plane is specified by the X index.
    bool getSliceYZ(tSize x, CSlice& Plane);

    //! Cuts the volume using plane XY and saves obtained data to a given
    //! slice. Position of the plane is specified by the Z real position.
    bool getSliceXY(double dZ, CSlice& Plane);

    //! Cuts the volume using plane XZ and saves obtained data to a given
    //! slice. Position of the plane is specified by the Y real position.
    bool getSliceXZ(double dY, CSlice& Plane);

    //! Cuts the volume using plane YZ and saves obtained data to a given
    //! slice. Position of the plane is specified by the X real position.
    bool getSliceYZ(double dX, CSlice& Plane);

    //! Cuts the volume using plane by given slice and saves obtained data
    //! to the slice. Position of the plane is specified by the real position.
    bool getSlice(double dPosition, CSlice& Plane);

    //! Initializes slice parameters (size and etc.) by volume parameters
    //! in a given orientation.
    void initSlice(CSlice &Slice, EPlane eOrientation);


    //! Serializes all class members.
    void serialize(mds::mod::CChannelSerializer& Writer);

    //! Deserializes all class members.
    void deserialize(mds::mod::CChannelSerializer& Reader);

protected:
    //! Real voxel size.
    double m_dDX, m_dDY, m_dDZ;
};


//=============================================================================
/*
 * Smart pointer to density volume.
 */
typedef CDensityVolume::tSmartPtr   CDensityVolumePtr;


} // namespace img
} // namespace mds

#endif // MDS_DENSITYVOLUME_H

