//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsColor.h                          \n
 * Section: libImage                            \n
 * Date:    2004/04/25                          \n
 *
 * $Id: mdsColor.h 64 2006-08-11 08:45:24Z spanel $
 * 
 * Description:
 * - RGBA color definition.
 */

#ifndef MDS_COLOR_H
#define MDS_COLOR_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsSmallObject.h>


namespace mds
{
namespace img
{

//==============================================================================
/*!
 * RGBA color definition.
 */
class CColor : public mds::base::CSmallValueObject<>
{
public:
    //! Color component type.
    typedef mds::sys::tUInt8 tComponent;

    //! RGBA color represented as unsigned long.
    typedef mds::sys::tUInt32 tRGBA;

public:
    //! RGBA color components.
    union
    {
        //! Separated RGBA color components.
        struct
        {
            tComponent r, g, b, a;
        };

        //! RGBA color represented by a one tRGBA value.
        tRGBA rgba;
    };

public:
    //! Default constructor.
    //! - Black color, alpha is set to 255.
    CColor() : rgba(0xFF000000UL) {}

    //! Constructor.
    //! - Separated RGBA color components.
    CColor(tComponent Red, tComponent Green, tComponent Blue, tComponent Alpha = 0xFF)
            : r(Red), g(Green), b(Blue), a(Alpha)
    {}

    //! Constructor.
    //! - RGBA color.
    CColor(tRGBA Color) : rgba(Color) {}

    //! Copy constructor.
    CColor(const CColor& Color) : rgba(Color.rgba) {}

    //! Assignment operators.
    CColor& operator =(CColor Color)
    {
        rgba = Color.rgba;
        return *this;
    }

    CColor& operator =(tRGBA Color)
    {
        rgba = Color;
        return *this;
    }

    //! Conversion operators.
    operator tRGBA() { return rgba; }

    //! Comparison operator.
    friend bool operator ==(CColor l, CColor r)
    {
        return (l.rgba == r.rgba);
    }
};


//==============================================================================
/*
 * Basic colors.
 */

//! Several basic colors.
const CColor C_BLACK    (0x00, 0x00, 0x00, 0xFF);
const CColor C_RED      (0xFF, 0x00, 0x00, 0xFF);
const CColor C_GREEN    (0x00, 0xFF, 0x00, 0xFF);
const CColor C_BLUE     (0x00, 0x00, 0xFF, 0xFF);
const CColor C_WHITE    (0xFF, 0xFF, 0xFF, 0xFF);
const CColor C_YELLOW   (0xFF, 0xFF, 0x00, 0xFF);
const CColor C_MAGENTA  (0xFF, 0x00, 0xFF, 0xFF);
const CColor C_CYAN     (0x00, 0xFF, 0xFF, 0xFF);
const CColor C_GRAY     (0x80, 0x80, 0x80, 0xFF);


} // namespace img
} // namespace mds

#endif // MDS_COLOR_H

