//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFunctions/mdsGeneral.hxx      \n
 * Section: libImage                            \n
 * Date:    2006/02/21                          \n
 *
 * $Id: mdsGeneral.hxx 64 2006-08-11 08:45:24Z spanel $
 *
 * Description:
 * - General volume functions.
 */


//==============================================================================
/*
 * Functions templates.
 */

template <typename R, class V1, class V2>
inline R getProduct(const CVolumeBase<V1>& Volume1, const CVolumeBase<V2>& Volume2)
{
    typedef typename V1::tVoxel tVoxel1;
    typedef typename V2::tVoxel tVoxel2;
    const V1& Volume1Impl = Volume1.getImpl();
    const V2& Volume2Impl = Volume2.getImpl();

    tSize XCount = mds::math::getMin(Volume1Impl.getXSize(), Volume2Impl.getXSize());
    tSize YCount = mds::math::getMin(Volume1Impl.getYSize(), Volume2Impl.getYSize());
    tSize ZCount = mds::math::getMin(Volume1Impl.getZSize(), Volume2Impl.getZSize());

    tSize Count = XCount * YCount * ZCount;
    MDS_CHECK(Count > 0, return R(0));

    R Total = R(0);
    for( tSize k = 0; k < ZCount; ++k )
    {
        for( tSize j = 0; j < YCount; ++j )
        {
            const tVoxel1 *p1 = Volume1Impl.getRowPtr(j, k);
            const tVoxel1 *p1Max = p1 + XCount;
            const tVoxel2 *p2 = Volume2Impl.getRowPtr(j, k);
            while( p1 < p1Max )
            {
                Total += R(*(p1++)) * R(*(p2++));
            }
        }
    }
    return Total;
}


template <typename R, class V1, class V2>
inline R getMeanSquareError(const CVolumeBase<V1>& Volume1, const CVolumeBase<V2>& Volume2)
{
    typedef typename V1::tVoxel tVoxel1;
    typedef typename V2::tVoxel tVoxel2;
    const V1& Volume1Impl = Volume1.getImpl();
    const V2& Volume2Impl = Volume2.getImpl();

    tSize XCount = mds::math::getMin(Volume1Impl.getXSize(), Volume2Impl.getXSize());
    tSize YCount = mds::math::getMin(Volume1Impl.getYSize(), Volume2Impl.getYSize());
    tSize ZCount = mds::math::getMin(Volume1Impl.getZSize(), Volume2Impl.getZSize());

    tSize Count = XCount * YCount * ZCount;
    MDS_CHECK(Count > 0, return R(0));

    R Total = R(0);
    for( tSize k = 0; k < ZCount; ++k )
    {
        for( tSize j = 0; j < YCount; ++j )
        {
            const tVoxel1 *p1 = Volume1Impl.getRowPtr(j, k);
            const tVoxel1 *p1Max = p1 + XCount;
            const tVoxel2 *p2 = Volume2Impl.getRowPtr(j, k);
            while( p1 < p1Max )
            {
                R Temp = R(*(p2++)) - R(*(p1++));
                Total += Temp * Temp;
            }
        }
    }
    return Total / R(Count);
}


template <typename R, class V>
inline R getSum(const CVolumeBase<V>& Volume)
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    R Total = R(0);
    for( tSize k = 0; k < VolumeImpl.getZSize(); ++k )
    {
        for( tSize j = 0; j < VolumeImpl.getYSize(); ++j )
        {
            const tVoxel *p = VolumeImpl.getRowPtr(j, k);
            const tVoxel *pMax = p + VolumeImpl.getXSize();
            while( p < pMax )
            {
                Total += R(*(p++));
            }
        }
    }
    return Total;
}


template <typename R, class V>
inline R getSumOfSquares(const CVolumeBase<V>& Volume)
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    R Total = R(0);
    for( tSize k = 0; k < VolumeImpl.getZSize(); ++k )
    {
        for( tSize j = 0; j < VolumeImpl.getYSize(); ++j )
        {
            const tVoxel *p = VolumeImpl.getRowPtr(j, k);
            const tVoxel *pMax = p + VolumeImpl.getXSize();
            while( p < pMax )
            {
                R Sample = R(*(p++));
                Total += Sample * Sample;
            }
        }
    }
    return Total;
}


template <typename R, class V>
inline R getMean(const CVolumeBase<V>& Volume)
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    tSize Count = VolumeImpl.getZSize() * VolumeImpl.getYSize() * VolumeImpl.getXSize();
    MDS_CHECK(Count > 0, return R(0));

    R Total = R(0);
    for( tSize k = 0; k < VolumeImpl.getZSize(); ++k )
    {
        for( tSize j = 0; j < VolumeImpl.getYSize(); j++ )
        {
            const tVoxel *p = VolumeImpl.getRowPtr(j, k);
            const tVoxel *pMax = p + VolumeImpl.getXSize();
            while( p < pMax )
            {
                Total += R(*(p++));
            }
        }
    }
    return Total / R(Count);
}


template <typename R, class V>
inline R getVariance(const CVolumeBase<V>& Volume)
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    tSize Count = VolumeImpl.getZSize() * VolumeImpl.getYSize() * VolumeImpl.getXSize();
    MDS_CHECK(Count > 0, return R(0));

    R Sum = R(0), SumSqr = R(0);
    for( tSize k = 0; k < VolumeImpl.getZSize(); ++k )
    {
        for( tSize j = 0; j < VolumeImpl.getYSize(); ++j )
        {
            const tVoxel *p = VolumeImpl.getRowPtr(j, k);
            const tVoxel *pMax = p + VolumeImpl.getXSize();
            while( p < pMax )
            {
                R Temp = R(*(p++));
                Sum += Temp;
                SumSqr += Temp * Temp;
            }
        }
    }
    R Temp = R(1) / R(Count);
    return SumSqr * Temp - (Sum * Temp * Sum * Temp);
}


template <typename R, class V>
inline R getMin(const CVolumeBase<V>& Volume)
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    tSize Count = VolumeImpl.getZSize() * VolumeImpl.getYSize() * VolumeImpl.getXSize();
    MDS_CHECK(Count > 0, return R(0));

    tVoxel MinVal = *VolumeImpl.getPtr();
    for( tSize k = 0; k < VolumeImpl.getZSize(); ++k )
    {
        for( tSize j = 0; j < VolumeImpl.getYSize(); ++j )
        {
            const tVoxel *p = VolumeImpl.getRowPtr(j, k);
            const tVoxel *pMax = p + VolumeImpl.getXSize();
            while( p < pMax )
            {
                if( *p < MinVal )
                {
                    MinVal = *p;
                }
                ++p;
            }
        }
    }
    return R(MinVal);
}


template <typename R, class V>
inline R getMax(const CVolumeBase<V>& Volume)
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    tSize Count = VolumeImpl.getZSize() * VolumeImpl.getYSize() * VolumeImpl.getXSize();
    MDS_CHECK(Count > 0, return R(0));

    tVoxel MaxVal = *VolumeImpl.getPtr();
    for( tSize k = 0; k < VolumeImpl.getZSize(); ++k )
    {
        for( tSize j = 0; j < VolumeImpl.getYSize(); ++j )
        {
            const tVoxel *p = VolumeImpl.getRowPtr(j, k);
            const tVoxel *pMax = p + VolumeImpl.getXSize();
            while( p < pMax )
            {
                if( *p > MaxVal )
                {
                    MaxVal = *p;
                }
                ++p;
            }
        }
    }
    return R(MaxVal);
}

