//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFunctions/mdsConvolution.hxx  \n
 * Section: libImage                            \n
 * Date:    2006/02/21                          \n
 *
 * $Id: mdsConvolution.hxx 64 2006-08-11 08:45:24Z spanel $
 *
 * Description:
 * - Volume convolution functions.
 */


//==============================================================================
/*
 * Functions templates.
 */
template <typename R, class V>
inline R convolve(const CVolumeBase<V>& Volume,
                  tSize x,
                  tSize y,
                  tSize z,
                  const CKernel3D& Kernel
                 )
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;
    tSize YOffset = VolumeImpl.getYOffset() - Size;
    tSize ZOffset = VolumeImpl.getZOffset() - Size * VolumeImpl.getYOffset();

    const tVoxel *p = VolumeImpl.getPtr(x - Half, y - Half, z - Half);
    const CKernel3D::tData *pK = Kernel.getPtr();

    R Sum = R(0);
    for( tSize k = 0; k < Size; ++k )
    {
        for( tSize j = 0; j < Size; ++j )
        {
            const tVoxel *pMax = p + Size;
            while( p < pMax )
            {
                Sum += R(*(p++)) * R(*(pK++));
            }
            p += YOffset;
        }
        p += ZOffset;
    }
    return Sum;
}


template <typename R, class V>
inline R convolveXY(const CVolumeBase<V>& Volume,
                    tSize x,
                    tSize y,
                    tSize z,
                    const CKernel2D& Kernel
                   )
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;
    tSize Offset = VolumeImpl.getYOffset() - Size;

    const tVoxel *p = VolumeImpl.getPtr(x - Half, y - Half, z);
    const CKernel2D::tData *pK = Kernel.getPtr();

    R Sum = R(0);
    for( tSize j = 0; j < Size; ++j )
    {
        const tVoxel *pMax = p + Size;
        while( p < pMax )
        {
            Sum += R(*(p++)) * R(*(pK++));
        }
        p += Offset;
    }
    return Sum;
}


template <typename R, class V>
inline R convolveXZ(const CVolumeBase<V>& Volume,
                    tSize x,
                    tSize y,
                    tSize z,
                    const CKernel2D& Kernel
                   )
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;
    tSize Offset = VolumeImpl.getZOffset() - Size;

    const tVoxel *p = VolumeImpl.getPtr(x - Half, y, z - Half);
    const CKernel2D::tData *pK = Kernel.getPtr();

    R Sum = R(0);
    for( tSize j = 0; j < Size; ++j )
    {
        const tVoxel *pMax = p + Size;
        while( p < pMax )
        {
            Sum += R(*(p++)) * R(*(pK++));
        }
        p += Offset;
    }
    return Sum;
}


template <typename R, class V>
inline R convolveYZ(const CVolumeBase<V>& Volume,
                    tSize x,
                    tSize y,
                    tSize z,
                    const CKernel2D& Kernel
                   )
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;
    tSize Offset = VolumeImpl.getZOffset() - Size * VolumeImpl.getYOffset();

    const tVoxel *p = VolumeImpl.getPtr(x, y - Half, z - Half);
    const CKernel2D::tData *pK = Kernel.getPtr();

    R Sum = R(0);
    for( tSize j = 0; j < Size; ++j )
    {
        const tVoxel *pMax = p + Size * VolumeImpl.getYOffset();
        while( p < pMax )
        {
            Sum += R(*(p)) * R(*(pK++));
            p += VolumeImpl.getYOffset();
        }
        p += Offset;
    }
    return Sum;
}


template <typename R, class V>
inline R convolveX(const CVolumeBase<V>& Volume,
                   tSize x,
                   tSize y,
                   tSize z,
                   const CKernel1D& Kernel
                  )
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;

    const tVoxel *p = VolumeImpl.getPtr(x - Half, y, z);
    const tVoxel *pMax = p + Size;
    const CKernel1D::tData *pK = Kernel.getPtr();

    R Sum = R(0);
    while( p < pMax)
    {
        Sum += R(*(p++)) * R(*(pK++));
    }
    return Sum;
}


template <typename R, class V>
inline R convolveY(const CVolumeBase<V>& Volume,
                   tSize x,
                   tSize y,
                   tSize z,
                   const CKernel1D& Kernel
                  )
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;

    const tVoxel *p = VolumeImpl.getPtr(x, y - Half, z);
    const tVoxel *pMax = p + Size * VolumeImpl.getYOffset();
    const CKernel1D::tData *pK = Kernel.getPtr();

    R Sum = R(0);
    while( p < pMax)
    {
        Sum += R(*(p)) * R(*(pK++));
        p += VolumeImpl.getYOffset();
    }
    return Sum;
}


template <typename R, class V>
inline R convolveZ(const CVolumeBase<V>& Volume,
                   tSize x,
                   tSize y,
                   tSize z,
                   const CKernel1D& Kernel
                  )
{
    typedef typename V::tVoxel tVoxel;
    const V& VolumeImpl = Volume.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;

    const tVoxel *p = VolumeImpl.getPtr(x, y, z - Half);
    const tVoxel *pMax = p + Size * VolumeImpl.getZOffset();
    const CKernel1D::tData *pK = Kernel.getPtr();

    R Sum = R(0);
    while( p < pMax)
    {
        Sum += R(*(p)) * R(*(pK++));
        p += VolumeImpl.getZOffset();
    }
    return Sum;
}

