//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsSobel.h            \n
 * Section: libVolume                           \n
 * Date:    2005/09/20                          \n
 *
 * $Id: mdsSobel.hxx 50 2006-08-08 13:05:53Z spanel $
 * 
 * Description:
 * - Sobel operator.
 */


//==============================================================================
/*
 * Methods templates.
 */

// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_SOBEL_X, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the image
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                tResult Value = getResponse(SrcVolume, x, y, z);
                DstVolume.set(x, y, z, norm::normalize(Value));
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
inline typename CVolumeFilter<V, VF_SOBEL_X, N>::tResult CVolumeFilter<V, VF_SOBEL_X, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Compute filter response
/*  tResult Value = Denom * (
        4 * tResult(SrcVolume.get(x + 1, y, z))
        + 2 * tResult(SrcVolume.get(x + 1, y - 1, z))
        + 2 * tResult(SrcVolume.get(x + 1, y + 1, z))
        + 2 * tResult(SrcVolume.get(x + 1, y, z - 1))
        + 2 * tResult(SrcVolume.get(x + 1, y, z + 1))
        + tResult(SrcVolume.get(x + 1, y - 1, z - 1))
        + tResult(SrcVolume.get(x + 1, y + 1, z - 1))
        + tResult(SrcVolume.get(x + 1, y - 1, z + 1))
        + tResult(SrcVolume.get(x + 1, y + 1, z + 1))
        - 4 * tResult(SrcVolume.get(x - 1, y, z))
        - 2 * tResult(SrcVolume.get(x - 1, y - 1, z))
        - 2 * tResult(SrcVolume.get(x - 1, y + 1, z))
        - 2 * tResult(SrcVolume.get(x - 1, y, z - 1))
        - 2 * tResult(SrcVolume.get(x - 1, y, z + 1))
        - tResult(SrcVolume.get(x - 1, y - 1, z - 1))
        - tResult(SrcVolume.get(x - 1, y + 1, z - 1))
        - tResult(SrcVolume.get(x - 1, y - 1, z + 1))
        - tResult(SrcVolume.get(x - 1, y + 1, z + 1))
        );
    return Value;*/

    // Volume properties
    tSize XOffset = SrcVolume.getXOffset();
    tSize YOffset = SrcVolume.getYOffset();
    tSize ZOffset = SrcVolume.getZOffset();

    // Pointer to the voxel
    const tVoxel *p = SrcVolume.getPtr(x, y, z);

    // Compute filter response
    tResult Value = Denom * (
                        4 * tResult(*(p + XOffset))
                        + 2 * tResult(*(p + XOffset - YOffset))
                        + 2 * tResult(*(p + XOffset + YOffset))
                        + 2 * tResult(*(p + XOffset - ZOffset))
                        + 2 * tResult(*(p + XOffset + ZOffset))
                        + tResult(*(p + XOffset - YOffset - ZOffset))
                        + tResult(*(p + XOffset + YOffset - ZOffset))
                        + tResult(*(p + XOffset - YOffset + ZOffset))
                        + tResult(*(p + XOffset + YOffset + ZOffset))
                        - 4 * tResult(*(p - XOffset))
                        - 2 * tResult(*(p - XOffset - YOffset))
                        - 2 * tResult(*(p - XOffset + YOffset))
                        - 2 * tResult(*(p - XOffset - ZOffset))
                        - 2 * tResult(*(p - XOffset + ZOffset))
                        - tResult(*(p - XOffset - YOffset - ZOffset))
                        - tResult(*(p - XOffset + YOffset - ZOffset))
                        - tResult(*(p - XOffset - YOffset + ZOffset))
                        - tResult(*(p - XOffset + YOffset + ZOffset))
                        );
    return Value;
}


// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_SOBEL_Y, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the volume
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                tResult Value = getResponse(SrcVolume, x, y, z);
                DstVolume.set(x, y, z, norm::normalize(Value));
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
inline typename CVolumeFilter<V, VF_SOBEL_Y, N>::tResult CVolumeFilter<V, VF_SOBEL_Y, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Compute filter response
/*  tResult Value = Denom * (
        4 * tResult(SrcVolume.get(x, y + 1, z))
        + 2 * tResult(SrcVolume.get(x + 1, y + 1, z))
        + 2 * tResult(SrcVolume.get(x - 1, y + 1, z))
        + 2 * tResult(SrcVolume.get(x, y + 1, z - 1))
        + 2 * tResult(SrcVolume.get(x, y + 1, z + 1))
        + tResult(SrcVolume.get(x + 1, y + 1, z - 1))
        + tResult(SrcVolume.get(x - 1, y + 1, z - 1))
        + tResult(SrcVolume.get(x + 1, y + 1, z + 1))
        + tResult(SrcVolume.get(x - 1, y + 1, z + 1))
        - 4 * tResult(SrcVolume.get(x, y - 1, z))
        - 2 * tResult(SrcVolume.get(x + 1, y - 1, z))
        - 2 * tResult(SrcVolume.get(x - 1, y - 1, z))
        - 2 * tResult(SrcVolume.get(x, y - 1, z - 1))
        - 2 * tResult(SrcVolume.get(x, y - 1, z + 1 ))
        - tResult(SrcVolume.get(x + 1, y - 1, z - 1))
        - tResult(SrcVolume.get(x - 1, y - 1, z - 1))
        - tResult(SrcVolume.get(x + 1, y - 1, z + 1))
        - tResult(SrcVolume.get(x - 1, y - 1, z + 1))
        );
    return Value;*/

    // Volume properties
    tSize XOffset = SrcVolume.getXOffset();
    tSize YOffset = SrcVolume.getYOffset();
    tSize ZOffset = SrcVolume.getZOffset();

    // Pointer to the voxel
    const tVoxel *p = SrcVolume.getPtr(x, y, z);

    // Compute filter response
    tResult Value = Denom * (
                        4 * tResult(*(p + YOffset))
                        + 2 * tResult(*(p + XOffset + YOffset))
                        + 2 * tResult(*(p - XOffset + YOffset))
                        + 2 * tResult(*(p + YOffset - ZOffset))
                        + 2 * tResult(*(p + YOffset + ZOffset))
                        + tResult(*(p + XOffset + YOffset - ZOffset))
                        + tResult(*(p - XOffset + YOffset - ZOffset))
                        + tResult(*(p + XOffset + YOffset + ZOffset))
                        + tResult(*(p - XOffset + YOffset + ZOffset))
                        - 4 * tResult(*(p - YOffset))
                        - 2 * tResult(*(p + XOffset - YOffset))
                        - 2 * tResult(*(p - XOffset - YOffset))
                        - 2 * tResult(*(p - YOffset - ZOffset))
                        - 2 * tResult(*(p - YOffset + ZOffset ))
                        - tResult(*(p + XOffset - YOffset - ZOffset))
                        - tResult(*(p - XOffset - YOffset - ZOffset))
                        - tResult(*(p + XOffset - YOffset + ZOffset))
                        - tResult(*(p - XOffset - YOffset + ZOffset))
                        );
    return Value;
}


// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_SOBEL_Z, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the volume
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                tResult Value = getResponse(SrcVolume, x, y, z);
                DstVolume.set(x, y, z, norm::normalize(Value));
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
inline typename CVolumeFilter<V, VF_SOBEL_Z, N>::tResult CVolumeFilter<V, VF_SOBEL_Z, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Compute filter response
/*  tResult Value = Denom * (
        4 * tResult(SrcVolume.get(x, y, z + 1))
        + 2 * tResult(SrcVolume.get(x + 1, y, z + 1))
        + 2 * tResult(SrcVolume.get(x - 1, y, z + 1))
        + 2 * tResult(SrcVolume.get(x, y - 1, z + 1))
        + 2 * tResult(SrcVolume.get(x, y + 1, z + 1))
        + tResult(SrcVolume.get(x + 1, y - 1, z + 1))
        + tResult(SrcVolume.get(x - 1, y - 1, z + 1))
        + tResult(SrcVolume.get(x + 1, y + 1, z + 1))
        + tResult(SrcVolume.get(x - 1, y + 1, z + 1))
        - 4 * tResult(SrcVolume.get(x, y, z - 1))
        - 2 * tResult(SrcVolume.get(x + 1, y, z - 1))
        - 2 * tResult(SrcVolume.get(x - 1, y, z - 1))
        - 2 * tResult(SrcVolume.get(x, y - 1, z - 1))
        - 2 * tResult(SrcVolume.get(x, y + 1, z - 1))
        - tResult(SrcVolume.get(x + 1, y - 1, z - 1))
        - tResult(SrcVolume.get(x - 1, y - 1, z - 1))
        - tResult(SrcVolume.get(x + 1, y + 1, z - 1))
        - tResult(SrcVolume.get(x - 1, y + 1, z - 1))
        );
    return Value;*/

    // Volume properties
    tSize XOffset = SrcVolume.getXOffset();
    tSize YOffset = SrcVolume.getYOffset();
    tSize ZOffset = SrcVolume.getZOffset();

    // Pointer to the voxel
    const tVoxel *p = SrcVolume.getPtr(x, y, z);

    // Compute filter response
    tResult Value = Denom * (
                        4 * tResult(*(p + ZOffset))
                        + 2 * tResult(*(p + XOffset + ZOffset))
                        + 2 * tResult(*(p - XOffset + ZOffset))
                        + 2 * tResult(*(p - YOffset + ZOffset))
                        + 2 * tResult(*(p + YOffset + ZOffset))
                        + tResult(*(p + XOffset - YOffset + ZOffset))
                        + tResult(*(p - XOffset - YOffset + ZOffset))
                        + tResult(*(p + XOffset + YOffset + ZOffset))
                        + tResult(*(p - XOffset + YOffset + ZOffset))
                        - 4 * tResult(*(p - ZOffset))
                        - 2 * tResult(*(p + XOffset - ZOffset))
                        - 2 * tResult(*(p - XOffset - ZOffset))
                        - 2 * tResult(*(p - YOffset - ZOffset))
                        - 2 * tResult(*(p + YOffset - ZOffset))
                        - tResult(*(p + XOffset - YOffset - ZOffset))
                        - tResult(*(p - XOffset - YOffset - ZOffset))
                        - tResult(*(p + XOffset + YOffset - ZOffset))
                        - tResult(*(p - XOffset + YOffset - ZOffset))
                        );
    return Value;
}

