//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsSobel.h            \n
 * Section: libVolume                           \n
 * Date:    2005/09/20                          \n
 *
 * $Id: mdsSobel.h 442 2007-08-10 15:22:27Z spanel $
 * 
 * Description:
 * - Sobel operator.
 */

#ifndef MDS_VF_SOBEL_H
#define MDS_VF_SOBEL_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsVolumeFilter.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Sobel operator
MDS_DECLARE_ID(VF_SOBEL_X);
MDS_DECLARE_ID(VF_SOBEL_Y);
MDS_DECLARE_ID(VF_SOBEL_Z);


//==============================================================================
/*!
 * Sobel operator (x-axis).
 * - Parameter V is a used image type.
 * - Kernel definition: \n
 *     [ -1  0  1 ]   \n
 *     [ -2  0  2 ]   \n
 *     [ -1  0  1 ]   \n
 *    1 [ -2  0  2 ]  \n
 *   -- [ -4  0  4 ]  \n
 *   16 [ -2  0  2 ]  \n
 *       [ -1  0  1 ] \n
 *       [ -2  0  2 ] \n
 *       [ -1  0  1 ] \n
 */
template <class V, template <typename> class N>
class CVolumeFilter<V, VF_SOBEL_X, N> : public CVolumeFilterBase<V,N>
{
public:
    //! Volume filter base.
    typedef CVolumeFilterBase<V,N> base;
    typedef typename base::norm norm;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 16;

public:
    //! Default constructor.
    CVolumeFilter() {}

    //! Destructor.
    ~CVolumeFilter() {}

    //! Filtering of input/source volume.
    //! - Returns false on failure
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns filter response at specified volume position.
    //! - Value is not normalized!
    inline tResult getResponse(const V& SrcVolume, tSize x, tSize y, tSize z);

    //! Returns kernel filter size.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Sobel operator (y-axis).
 * - Parameter V is a used image type.
 * - Kernel definition: \n
 *     [ -1 -2 -1 ]   \n
 *     [  0  0  0 ]   \n
 *     [  1  2  1 ]   \n
 *    1 [ -2 -4 -2 ]  \n
 *   -- [  0  0  0 ]  \n
 *   16 [  2  4  2 ]  \n
 *       [ -1 -2 -1 ] \n
 *       [  0  0  0 ] \n
 *       [  1  2  1 ] \n
 */
template <class V, template <typename> class N>
class CVolumeFilter<V, VF_SOBEL_Y, N> : public CVolumeFilterBase<V,N>
{
public:
    //! Volume filter base.
    typedef CVolumeFilterBase<V,N> base;
    typedef typename base::norm norm;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 16;

public:
    //! Default constructor.
    CVolumeFilter() {}

    //! Destructor.
    ~CVolumeFilter() {}

    //! Filtering of input/source volume.
    //! - Returns false on failure
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns filter response at specified volume position.
    //! - Value is not normalized!
    inline tResult getResponse(const V& SrcVolume, tSize x, tSize y, tSize z);

    //! Returns kernel filter size.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Sobel operator (z-axis).
 * - Parameter V is a used image type.
 * - Kernel definition: \n
 *     [ -1 -2 -1 ]   \n
 *     [ -2 -4 -2 ]   \n
 *     [ -1 -2 -1 ]   \n
 *    1 [  0  0  0 ]  \n
 *   -- [  0  0  0 ]  \n
 *   16 [  0  0  0 ]  \n
 *       [  1  2  1 ] \n
 *       [  2  4  2 ] \n
 *       [  1  2  1 ] \n
 */
template <class V, template <typename> class N>
class CVolumeFilter<V, VF_SOBEL_Z, N> : public CVolumeFilterBase<V,N>
{
public:
    //! Volume filter base.
    typedef CVolumeFilterBase<V,N> base;
    typedef typename base::norm norm;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 16;

public:
    //! Default constructor.
    CVolumeFilter() {}

    //! Destructor.
    ~CVolumeFilter() {}

    //! Filtering of input/source volume.
    //! - Returns false on failure
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns filter response at specified volume position.
    //! - Value is not normalized!
    inline tResult getResponse(const V& SrcVolume, tSize x, tSize y, tSize z);

    //! Returns kernel filter size.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsSobel.hxx"


} // namespace img
} // namespace mds

#endif // MDS_VF_SOBEL_H

