//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsPrewit.h           \n
 * Section: libVolume                           \n
 * Date:    2005/09/20                          \n
 *
 * $Id: mdsPrewit.hxx 50 2006-08-08 13:05:53Z spanel $
 * 
 * Description:
 * - Prewit gradient edge detectors.
 */


//==============================================================================
/*
 * Methods templates.
 */

// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_PREWIT_X, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the volume
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                tResult Value = getResponse(SrcVolume, x, y, z);
                DstVolume.set(x, y, z, norm::normalize(Value));
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
inline typename CVolumeFilter<V, VF_PREWIT_X, N>::tResult CVolumeFilter<V, VF_PREWIT_X, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Compute filter response
    tResult Value = Denom * (
                        tResult(SrcVolume.get(x + 1, y, z))
                        + tResult(SrcVolume.get(x + 1, y - 1, z))
                        + tResult(SrcVolume.get(x + 1, y + 1, z))
                        + tResult(SrcVolume.get(x + 1, y, z - 1))
                        + tResult(SrcVolume.get(x + 1, y - 1, z - 1))
                        + tResult(SrcVolume.get(x + 1, y + 1, z - 1))
                        + tResult(SrcVolume.get(x + 1, y, z + 1))
                        + tResult(SrcVolume.get(x + 1, y - 1, z + 1))
                        + tResult(SrcVolume.get(x + 1, y + 1, z + 1))
                        - tResult(SrcVolume.get(x - 1, y, z))
                        - tResult(SrcVolume.get(x - 1, y - 1, z))
                        - tResult(SrcVolume.get(x - 1, y + 1, z))
                        - tResult(SrcVolume.get(x - 1, y, z - 1))
                        - tResult(SrcVolume.get(x - 1, y - 1, z - 1))
                        - tResult(SrcVolume.get(x - 1, y + 1, z - 1))
                        - tResult(SrcVolume.get(x - 1, y, z + 1))
                        - tResult(SrcVolume.get(x - 1, y - 1, z + 1))
                        - tResult(SrcVolume.get(x - 1, y + 1, z + 1))
                        );
    return Value;
}


// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_PREWIT_Y, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the volume
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                tResult Value = getResponse(SrcVolume, x, y, z);
                DstVolume.set(x, y, z, norm::normalize(Value));
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
inline typename CVolumeFilter<V, VF_PREWIT_Y, N>::tResult CVolumeFilter<V, VF_PREWIT_Y, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Compute filter response
    tResult Value = Denom * (
                        tResult(SrcVolume.get(x, y + 1, z))
                        + tResult(SrcVolume.get(x + 1, y + 1, z))
                        + tResult(SrcVolume.get(x - 1, y + 1, z))
                        + tResult(SrcVolume.get(x, y + 1, z - 1))
                        + tResult(SrcVolume.get(x + 1, y + 1, z - 1))
                        + tResult(SrcVolume.get(x - 1, y + 1, z - 1))
                        + tResult(SrcVolume.get(x, y + 1, z + 1))
                        + tResult(SrcVolume.get(x + 1, y + 1, z + 1))
                        + tResult(SrcVolume.get(x - 1, y + 1, z + 1))
                        - tResult(SrcVolume.get(x, y - 1, z))
                        - tResult(SrcVolume.get(x + 1, y - 1, z))
                        - tResult(SrcVolume.get(x - 1, y - 1, z))
                        - tResult(SrcVolume.get(x, y - 1, z - 1))
                        - tResult(SrcVolume.get(x + 1, y - 1, z - 1))
                        - tResult(SrcVolume.get(x - 1, y - 1, z - 1))
                        - tResult(SrcVolume.get(x, y - 1, z + 1 ))
                        - tResult(SrcVolume.get(x + 1, y - 1, z + 1))
                        - tResult(SrcVolume.get(x - 1, y - 1, z + 1))
                        );
    return Value;
}


// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_PREWIT_Z, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the volume
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                tResult Value = getResponse(SrcVolume, x, y, z);
                DstVolume.set(x, y, z, norm::normalize(Value));
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
inline typename CVolumeFilter<V, VF_PREWIT_Z, N>::tResult CVolumeFilter<V, VF_PREWIT_Z, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Compute filter response
    tResult Value = Denom * (
                        tResult(SrcVolume.get(x, y, z + 1))
                        + tResult(SrcVolume.get(x + 1, y, z + 1))
                        + tResult(SrcVolume.get(x - 1, y, z + 1))
                        + tResult(SrcVolume.get(x, y - 1, z + 1))
                        + tResult(SrcVolume.get(x + 1, y - 1, z + 1))
                        + tResult(SrcVolume.get(x - 1, y - 1, z + 1))
                        + tResult(SrcVolume.get(x, y + 1, z + 1))
                        + tResult(SrcVolume.get(x + 1, y + 1, z + 1))
                        + tResult(SrcVolume.get(x - 1, y + 1, z + 1))
                        - tResult(SrcVolume.get(x, y, z - 1))
                        - tResult(SrcVolume.get(x + 1, y, z - 1))
                        - tResult(SrcVolume.get(x - 1, y, z - 1))
                        - tResult(SrcVolume.get(x, y - 1, z - 1))
                        - tResult(SrcVolume.get(x + 1, y - 1, z - 1))
                        - tResult(SrcVolume.get(x - 1, y - 1, z - 1))
                        - tResult(SrcVolume.get(x, y + 1, z - 1))
                        - tResult(SrcVolume.get(x + 1, y + 1, z - 1))
                        - tResult(SrcVolume.get(x - 1, y + 1, z - 1))
                        );
    return Value;
}

