//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsMisc.h             \n
 * Section: libVolume                           \n
 * Date:    2007/08/10                          \n
 *
 * $Id:$
 * 
 * Description:
 * - Miscellaneous filters and operators.
 */


//==============================================================================
/*
 * Methods templates.
 */

// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_DIFF_X, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the image
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                tResult Value = getResponse(SrcVolume, x, y, z);
                DstVolume.set(x, y, z, norm::normalize(Value));
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
inline typename CVolumeFilter<V, VF_DIFF_X, N>::tResult CVolumeFilter<V, VF_DIFF_X, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Volume properties
    tSize XOffset = SrcVolume.getXOffset();

    // Pointer to the voxel
    const tVoxel *p = SrcVolume.getPtr(x, y, z);

    // Compute filter response
    tResult Value = Denom * (
                        4 * tResult(*(p + XOffset))
                        + 2 * tResult(*(p + 2 * XOffset))
                        + tResult(*(p + 3 * XOffset))
                        - 4 * tResult(*(p - XOffset))
                        - 2 * tResult(*(p - 2 * XOffset))
                        - tResult(*(p - 3 * XOffset))
                        );
    return Value;
}


// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_DIFF_Y, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the volume
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                tResult Value = getResponse(SrcVolume, x, y, z);
                DstVolume.set(x, y, z, norm::normalize(Value));
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
inline typename CVolumeFilter<V, VF_DIFF_Y, N>::tResult CVolumeFilter<V, VF_DIFF_Y, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Volume properties
    tSize YOffset = SrcVolume.getYOffset();

    // Pointer to the voxel
    const tVoxel *p = SrcVolume.getPtr(x, y, z);

    // Compute filter response
    tResult Value = Denom * (
                        4 * tResult(*(p + YOffset))
                        + 2 * tResult(*(p + 2 * YOffset))
                        + tResult(*(p + 3 * YOffset))
                        - 4 * tResult(*(p - YOffset))
                        - 2 * tResult(*(p - 2 * YOffset))
                        - tResult(*(p - 3 * YOffset))
                        );
    return Value;
}


// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_DIFF_Z, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the volume
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                tResult Value = getResponse(SrcVolume, x, y, z);
                DstVolume.set(x, y, z, norm::normalize(Value));
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
inline typename CVolumeFilter<V, VF_DIFF_Z, N>::tResult CVolumeFilter<V, VF_DIFF_Z, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Volume properties
    tSize ZOffset = SrcVolume.getZOffset();

    // Pointer to the voxel
    const tVoxel *p = SrcVolume.getPtr(x, y, z);

    // Compute filter response
    tResult Value = Denom * (
                        4 * tResult(*(p + ZOffset))
                        + 2 * tResult(*(p + 2 * ZOffset))
                        + tResult(*(p + 3 * ZOffset))
                        - 4 * tResult(*(p - ZOffset))
                        - 2 * tResult(*(p - 2 * ZOffset))
                        - tResult(*(p - 3 * ZOffset))
                        );
    return Value;
}

